<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Penduduk extends Model
{
    use HasFactory;

    protected $table = 'penduduk';

    protected $fillable = [
        'nik',
        'nama',
        'no_kk',
        'keluarga_id',
        'is_kepala_keluarga',
        'status_keluarga',
        'hubungan_keluarga',
        'ayah_nik',
        'nama_ayah',
        'ibu_nik',
        'nama_ibu',
        'alamat',
        'dusun',
        'desa',
        'rt',
        'rw',
        'tempat_lahir',
        'tanggal_lahir',
        'no_akta_lahir',
        'tanggal_akta_lahir',
        'tempat_akta_lahir',
        'jenis_kelamin',
        'status_kawin',
        'agama',
        'pekerjaan',
        'pendidikan',
        'pendidikan_terakhir',
        'kewarganegaraan',
        'status_hidup',
        'tanggal_meninggal',
        'sebab_kematian',
        'status_mutasi',
        'tanggal_mutasi',
        'keterangan_mutasi',
        'asal_mutasi',
        'tujuan_mutasi',
        'golongan_darah',
        'cacat_fisik',
        'no_wa',
        'password',
        'password_set_at',
        'password_must_change',
        'last_login_at',
        'foto',
        'latitude',
        'longitude'
    ];

    protected $casts = [
        'tanggal_lahir' => 'date',
        'tanggal_akta_lahir' => 'date',
        'tanggal_meninggal' => 'date',
        'tanggal_mutasi' => 'date',
        'password_set_at' => 'datetime',
        'last_login_at' => 'datetime',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'is_kepala_keluarga' => 'boolean',
        'password_must_change' => 'boolean'
    ];

    // Relationships
    public function keluarga()
    {
        return $this->belongsTo(Keluarga::class, 'keluarga_id', 'id');
    }
    
    public function keluargaByKK()
    {
        return $this->belongsTo(Keluarga::class, 'no_kk', 'no_kk');
    }

    public function anggotaKeluarga()
    {
        return $this->hasOne(AnggotaKeluarga::class, 'nik', 'nik');
    }

    public function pelayananSurat()
    {
        return $this->hasMany(PelayananSurat::class, 'nik', 'nik');
    }

    public function pengaduan()
    {
        return $this->hasMany(Pengaduan::class, 'nik', 'nik');
    }

    public function bantuan()
    {
        return $this->hasMany(Bantuan::class, 'nik', 'nik');
    }

    // Family Tree Relationships
    public function ayah()
    {
        return $this->belongsTo(Penduduk::class, 'ayah_nik', 'nik');
    }

    public function ibu()
    {
        return $this->belongsTo(Penduduk::class, 'ibu_nik', 'nik');
    }

    public function anakSebagaiAyah()
    {
        return $this->hasMany(Penduduk::class, 'ayah_nik', 'nik');
    }

    public function anakSebagaiIbu()
    {
        return $this->hasMany(Penduduk::class, 'ibu_nik', 'nik');
    }

    // Get all children (as father or mother)
    public function anak()
    {
        return $this->hasMany(Penduduk::class, 'ayah_nik', 'nik')
                    ->unionAll(
                        $this->hasMany(Penduduk::class, 'ibu_nik', 'nik')->getQuery()
                    );
    }

    public function orangTua()
    {
        $orangTua = collect();
        if ($this->ayah) {
            $orangTua->push($this->ayah);
        }
        if ($this->ibu) {
            $orangTua->push($this->ibu);
        }
        return $orangTua;
    }

    // Get siblings (same parents)
    public function saudara()
    {
        $query = self::where('nik', '!=', $this->nik);
        
        if ($this->ayah_nik && $this->ibu_nik) {
            // Full siblings (same father and mother)
            $query->where(function($q) {
                $q->where('ayah_nik', $this->ayah_nik)
                  ->where('ibu_nik', $this->ibu_nik);
            });
        } elseif ($this->ayah_nik) {
            // Half siblings from father
            $query->where('ayah_nik', $this->ayah_nik);
        } elseif ($this->ibu_nik) {
            // Half siblings from mother
            $query->where('ibu_nik', $this->ibu_nik);
        } else {
            // No siblings if no parents defined
            return collect();
        }
        
        return $query->get();
    }

    // Get grandparents
    public function kakekNenek()
    {
        $kakekNenek = collect();
        
        if ($this->ayah) {
            if ($this->ayah->ayah) $kakekNenek->push($this->ayah->ayah);
            if ($this->ayah->ibu) $kakekNenek->push($this->ayah->ibu);
        }
        
        if ($this->ibu) {
            if ($this->ibu->ayah) $kakekNenek->push($this->ibu->ayah);
            if ($this->ibu->ibu) $kakekNenek->push($this->ibu->ibu);
        }
        
        return $kakekNenek->unique('nik');
    }

    // Get grandchildren
    public function cucu()
    {
        $cucu = collect();
        
        foreach ($this->getAllChildren() as $anak) {
            foreach ($anak->getAllChildren() as $cucuItem) {
                $cucu->push($cucuItem);
            }
        }
        
        return $cucu;
    }

    // Helper method to get all children
    public function getAllChildren()
    {
        return self::where('ayah_nik', $this->nik)
                   ->orWhere('ibu_nik', $this->nik)
                   ->get();
    }

    // Get complete family tree recursively
    public function getPohonKeluarga($maxDepth = 3, $currentDepth = 0)
    {
        if ($currentDepth >= $maxDepth) {
            return [
                'data' => $this,
                'anak' => []
            ];
        }

        $children = $this->getAllChildren();
        $pohon = [
            'data' => $this,
            'anak' => []
        ];

        foreach ($children as $child) {
            $pohon['anak'][] = $child->getPohonKeluarga($maxDepth, $currentDepth + 1);
        }

        return $pohon;
    }

    // Get family lineage upwards (ancestors)
    public function getGarisKeturunan($maxDepth = 3, $currentDepth = 0)
    {
        if ($currentDepth >= $maxDepth) {
            return [
                'data' => $this,
                'orangtua' => []
            ];
        }

        $garis = [
            'data' => $this,
            'orangtua' => []
        ];

        if ($this->ayah) {
            $garis['orangtua']['ayah'] = $this->ayah->getGarisKeturunan($maxDepth, $currentDepth + 1);
        }

        if ($this->ibu) {
            $garis['orangtua']['ibu'] = $this->ibu->getGarisKeturunan($maxDepth, $currentDepth + 1);
        }

        return $garis;
    }

    // Accessors
    public function getUmurAttribute()
    {
        if (!$this->tanggal_lahir) {
            return 0;
        }
        
        try {
            return $this->tanggal_lahir->age;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getAlamatLengkapAttribute()
    {
        $alamat = $this->alamat;
        if ($this->rt) $alamat .= " RT {$this->rt}";
        if ($this->rw) $alamat .= " RW {$this->rw}";
        if ($this->dusun) $alamat .= " Dusun {$this->dusun}";
        return $alamat;
    }

    public function getStatusKawinColorAttribute()
    {
        return match($this->status_kawin) {
            'Belum Kawin' => 'primary',
            'Kawin' => 'success',
            'Cerai Hidup' => 'warning', 
            'Cerai Mati' => 'secondary',
            default => 'secondary'
        };
    }

    // Scopes
    public function scopeByDusun($query, $dusun)
    {
        return $query->where('dusun', $dusun);
    }

    public function scopeByRtRw($query, $rt = null, $rw = null)
    {
        if ($rt) $query->where('rt', $rt);
        if ($rw) $query->where('rw', $rw);
        return $query;
    }

    public function scopeHasCoordinates($query)
    {
        return $query->whereNotNull('latitude')->whereNotNull('longitude');
    }

    // Scope untuk status hidup
    public function scopeHidup($query)
    {
        return $query->where('status_hidup', 'Hidup');
    }

    public function scopeMeninggal($query)
    {
        return $query->where('status_hidup', 'Meninggal');
    }

    // Scope untuk mutasi penduduk
    public function scopeByStatusMutasi($query, $status)
    {
        return $query->where('status_mutasi', $status);
    }

    public function scopePindahMasuk($query)
    {
        return $query->where('status_mutasi', 'Pindah Masuk');
    }

    public function scopePindahKeluar($query)
    {
        return $query->where('status_mutasi', 'Pindah Keluar');
    }

    // Scope untuk statistik pendidikan
    public function scopeByPendidikan($query, $pendidikan)
    {
        return $query->where('pendidikan_terakhir', $pendidikan);
    }

    // Scope untuk pekerjaan
    public function scopeByPekerjaan($query, $pekerjaan)
    {
        return $query->where('pekerjaan', $pekerjaan);
    }

    // Method untuk validasi NIK
    public static function isNikExists($nik, $excludeId = null)
    {
        $query = self::where('nik', $nik);
        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }
        return $query->exists();
    }

    // Method untuk validasi NIK format
    public static function isValidNikFormat($nik)
    {
        // NIK harus 16 digit
        if (strlen($nik) !== 16 || !is_numeric($nik)) {
            return false;
        }

        // Validasi kode wilayah (6 digit pertama)
        $kodeWilayah = substr($nik, 0, 6);
        
        // Validasi tanggal lahir (6 digit: DDMMYY)
        $tanggalLahir = substr($nik, 6, 6);
        $tanggal = (int)substr($tanggalLahir, 0, 2);
        $bulan = (int)substr($tanggalLahir, 2, 2);
        $tahun = (int)substr($tanggalLahir, 4, 2);

        // Untuk perempuan, tanggal ditambah 40
        if ($tanggal > 40) {
            $tanggal -= 40;
        }

        // Validasi tanggal
        if ($tanggal < 1 || $tanggal > 31 || $bulan < 1 || $bulan > 12) {
            return false;
        }

        return true;
    }

    // Method untuk extract gender dari NIK
    public static function getGenderFromNik($nik)
    {
        if (strlen($nik) !== 16) {
            return null;
        }

        $tanggal = (int)substr($nik, 6, 2);
        return ($tanggal > 40) ? 'P' : 'L';
    }

    // Method untuk extract tanggal lahir dari NIK
    public static function getBirthDateFromNik($nik)
    {
        if (strlen($nik) !== 16) {
            return null;
        }

        $tanggal = (int)substr($nik, 6, 2);
        $bulan = (int)substr($nik, 8, 2);
        $tahun = (int)substr($nik, 10, 2);

        // Untuk perempuan, tanggal dikurangi 40
        if ($tanggal > 40) {
            $tanggal -= 40;
        }

        // Asumsi tahun 2000-an jika < 50, 1900-an jika >= 50
        $tahunLengkap = ($tahun < 50) ? 2000 + $tahun : 1900 + $tahun;

        try {
            return \Carbon\Carbon::createFromDate($tahunLengkap, $bulan, $tanggal);
        } catch (\Exception $e) {
            return null;
        }
    }

    // Method untuk riwayat mutasi
    public function riwayatMutasi()
    {
        return $this->hasMany(MutasiPenduduk::class, 'nik', 'nik')->orderBy('tanggal_mutasi', 'desc');
    }

    // Helper methods untuk mapping nama ke NIK
    
    /**
     * Mencari NIK berdasarkan nama (untuk mapping ayah/ibu)
     */
    public static function findNikByNama($nama, $jenisKelamin = null)
    {
        $query = self::where('nama', 'LIKE', "%{$nama}%");
        
        if ($jenisKelamin) {
            $query->where('jenis_kelamin', $jenisKelamin);
        }
        
        return $query->first()?->nik;
    }
    
    /**
     * Mapping otomatis nama ayah ke NIK
     */
    public function mapNamaAyahToNik()
    {
        if ($this->nama_ayah && !$this->ayah_nik) {
            $nikAyah = self::findNikByNama($this->nama_ayah, 'L');
            if ($nikAyah) {
                $this->ayah_nik = $nikAyah;
                $this->save();
                return true;
            }
        }
        return false;
    }
    
    /**
     * Mapping otomatis nama ibu ke NIK
     */
    public function mapNamaIbuToNik()
    {
        if ($this->nama_ibu && !$this->ibu_nik) {
            $nikIbu = self::findNikByNama($this->nama_ibu, 'P');
            if ($nikIbu) {
                $this->ibu_nik = $nikIbu;
                $this->save();
                return true;
            }
        }
        return false;
    }
    
    /**
     * Mapping otomatis kedua orang tua
     */
    public function mapOrangTuaToNik()
    {
        $ayahMapped = $this->mapNamaAyahToNik();
        $ibuMapped = $this->mapNamaIbuToNik();
        
        return [
            'ayah_mapped' => $ayahMapped,
            'ibu_mapped' => $ibuMapped,
            'success' => $ayahMapped || $ibuMapped
        ];
    }
    
    /**
     * Batch mapping untuk semua penduduk yang belum memiliki NIK orang tua
     */
    public static function batchMapOrangTua()
    {
        $results = [
            'processed' => 0,
            'ayah_mapped' => 0,
            'ibu_mapped' => 0,
            'errors' => []
        ];
        
        // Ambil penduduk yang memiliki nama ayah/ibu tapi belum ada NIK
        $pendudukNeedMapping = self::where(function($query) {
            $query->where(function($q) {
                $q->whereNotNull('nama_ayah')
                  ->whereNull('ayah_nik');
            })->orWhere(function($q) {
                $q->whereNotNull('nama_ibu')
                  ->whereNull('ibu_nik');
            });
        })->get();
        
        foreach ($pendudukNeedMapping as $penduduk) {
            try {
                $mapping = $penduduk->mapOrangTuaToNik();
                $results['processed']++;
                
                if ($mapping['ayah_mapped']) {
                    $results['ayah_mapped']++;
                }
                if ($mapping['ibu_mapped']) {
                    $results['ibu_mapped']++;
                }
            } catch (\Exception $e) {
                $results['errors'][] = [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * Get informasi akta lahir lengkap
     */
    public function getAktaLahirLengkapAttribute()
    {
        if (!$this->no_akta_lahir) {
            return null;
        }
        
        return [
            'nomor' => $this->no_akta_lahir,
            'tanggal' => $this->tanggal_akta_lahir?->format('d-m-Y'),
            'tempat' => $this->tempat_akta_lahir,
            'lengkap' => $this->no_akta_lahir && $this->tanggal_akta_lahir && $this->tempat_akta_lahir
        ];
    }
    
    /**
     * Scope untuk pencarian berdasarkan nama ayah/ibu
     */
    public function scopeByNamaOrangTua($query, $nama, $tipe = 'both')
    {
        return $query->where(function($q) use ($nama, $tipe) {
            if ($tipe === 'ayah' || $tipe === 'both') {
                $q->where('nama_ayah', 'LIKE', "%{$nama}%");
            }
            if ($tipe === 'ibu' || $tipe === 'both') {
                $q->orWhere('nama_ibu', 'LIKE', "%{$nama}%");
            }
        });
    }
    
    /**
     * Scope untuk penduduk yang memiliki akta lahir
     */
    public function scopeHasAktaLahir($query)
    {
        return $query->whereNotNull('no_akta_lahir');
    }
    
    /**
     * Scope untuk penduduk yang belum memiliki akta lahir
     */
    public function scopeWithoutAktaLahir($query)
    {
        return $query->whereNull('no_akta_lahir');
    }

    // Method untuk password management
    public function setPassword($password)
    {
        $this->update([
            'password' => bcrypt($password),
            'password_set_at' => now(),
            'password_must_change' => false
        ]);
    }

    public function generateDefaultPassword()
    {
        // Generate password default: 6 digit terakhir NIK
        $defaultPassword = substr($this->nik, -6);
        $this->setPassword($defaultPassword);
        return $defaultPassword;
    }

    public function checkPassword($password)
    {
        return $this->password && \Hash::check($password, $this->password);
    }

    public function hasPassword()
    {
        return !empty($this->password);
    }

    public function mustChangePassword()
    {
        return $this->password_must_change || empty($this->password);
    }

    public function updateLastLogin()
    {
        $this->update(['last_login_at' => now()]);
    }

    // Scope untuk penduduk yang sudah memiliki password
    public function scopeHasPassword($query)
    {
        return $query->whereNotNull('password');
    }

    // Scope untuk penduduk yang belum memiliki password
    public function scopeWithoutPassword($query)
    {
        return $query->whereNull('password');
    }

    // Accessor untuk status password
    public function getPasswordStatusAttribute()
    {
        if (empty($this->password)) {
            return 'belum_diatur';
        } elseif ($this->password_must_change) {
            return 'harus_diganti';
        } else {
            return 'aktif';
        }
    }
} 
