<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PpobTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'user_id',
        'product_type',
        'provider',
        'target_number',
        'amount',
        'admin_fee',
        'total_amount',
        'status',
        'payment_url',
        'payment_data',
        'payment_response',
        'processed_at'
    ];

    protected $casts = [
        'payment_data' => 'array',
        'payment_response' => 'array',
        'processed_at' => 'datetime'
    ];

    /**
     * Relationship with User
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get status badge color
     */
    public function getStatusBadgeAttribute()
    {
        return match($this->status) {
            'pending' => 'bg-yellow-100 text-yellow-800',
            'paid' => 'bg-green-100 text-green-800',
            'processing' => 'bg-blue-100 text-blue-800',
            'success' => 'bg-green-100 text-green-800',
            'failed' => 'bg-red-100 text-red-800',
            'cancelled' => 'bg-gray-100 text-gray-800',
            default => 'bg-gray-100 text-gray-800'
        };
    }

    /**
     * Get status label
     */
    public function getStatusLabelAttribute()
    {
        return match($this->status) {
            'pending' => 'Menunggu Pembayaran',
            'paid' => 'Dibayar',
            'processing' => 'Diproses',
            'success' => 'Berhasil',
            'failed' => 'Gagal',
            'cancelled' => 'Dibatalkan',
            default => 'Tidak Diketahui'
        };
    }

    /**
     * Get product type label
     */
    public function getProductTypeLabelAttribute()
    {
        return match($this->product_type) {
            'pulsa' => 'Pulsa',
            'paket_data' => 'Paket Data',
            'pln' => 'PLN',
            'pdam' => 'PDAM',
            'bpjs' => 'BPJS',
            'tv_kabel' => 'TV Kabel',
            'internet' => 'Internet',
            'finance' => 'Multifinance',
            default => 'Lainnya'
        };
    }

    /**
     * Format amount to currency
     */
    public function getFormattedAmountAttribute()
    {
        return 'Rp ' . number_format($this->amount, 0, ',', '.');
    }

    /**
     * Format total amount to currency
     */
    public function getFormattedTotalAmountAttribute()
    {
        return 'Rp ' . number_format($this->total_amount, 0, ',', '.');
    }

    /**
     * Format admin fee to currency
     */
    public function getFormattedAdminFeeAttribute()
    {
        return 'Rp ' . number_format($this->admin_fee, 0, ',', '.');
    }
}
