<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Umkm extends Model
{
    use HasFactory;

    protected $table = 'umkm';

    protected $fillable = [
        'nama_umkm',
        'kategori',
        'pemilik_id',
        'deskripsi',
        'alamat',
        'nomor_telepon',
        'email',
        'website',
        'instagram',
        'facebook',
        'whatsapp',
        'latitude',
        'longitude',
        'jam_operasional',
        'hari_libur',
        'logo',
        'galeri_foto',
        'video_profil',
        'nomor_izin_usaha',
        'tanggal_izin',
        'npwp',
        'sertifikat',
        'skala_usaha',
        'jumlah_karyawan',
        'modal_awal',
        'omzet_bulanan',
        'produk_unggulan',
        'kapasitas_produksi_harian',
        'status',
        'is_verified',
        'verified_at',
        'verified_by',
        'catatan_verifikasi',
        'aktif_marketplace',
        'bisa_cod',
        'bisa_transfer',
        'minimal_order',
        'kebijakan_pengiriman',
        'kebijakan_pengembalian',
        'slug',
        'tags',
        'rating_rata_rata',
        'jumlah_ulasan',
        'total_penjualan',
        'view_count',
        'user_id'
    ];

    protected $casts = [
        'jam_operasional' => 'array',
        'hari_libur' => 'array',
        'galeri_foto' => 'array',
        'sertifikat' => 'array',
        'tags' => 'array',
        'tanggal_izin' => 'date',
        'verified_at' => 'datetime',
        'modal_awal' => 'decimal:2',
        'omzet_bulanan' => 'decimal:2',
        'minimal_order' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'is_verified' => 'boolean',
        'aktif_marketplace' => 'boolean',
        'bisa_cod' => 'boolean',
        'bisa_transfer' => 'boolean',
    ];

    // Boot method untuk auto-generate slug
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($umkm) {
            if (empty($umkm->slug)) {
                $umkm->slug = Str::slug($umkm->nama_umkm);
                
                // Ensure unique slug
                $originalSlug = $umkm->slug;
                $counter = 1;
                while (static::where('slug', $umkm->slug)->exists()) {
                    $umkm->slug = $originalSlug . '-' . $counter;
                    $counter++;
                }
            }
        });
    }

    // Relationships
    public function pemilik(): BelongsTo
    {
        return $this->belongsTo(Penduduk::class, 'pemilik_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function verifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function produk(): HasMany
    {
        return $this->hasMany(ProdukUmkm::class);
    }

    public function transaksi()
    {
        return $this->morphMany(Transaksi::class, 'transactionable');
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopeVerified($query)
    {
        return $query->where('is_verified', true);
    }

    public function scopeMarketplace($query)
    {
        return $query->where('aktif_marketplace', true);
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori', $kategori);
    }

    // Accessors & Mutators
    public function getLogoUrlAttribute()
    {
        if ($this->logo) {
            return asset('storage/umkm/logo/' . $this->logo);
        }
        return asset('images/default-umkm-logo.png');
    }

    public function getGaleriFotoUrlsAttribute()
    {
        if ($this->galeri_foto) {
            return collect($this->galeri_foto)->map(function ($foto) {
                return asset('storage/umkm/galeri/' . $foto);
            })->toArray();
        }
        return [];
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'nonaktif' => '<span class="badge bg-secondary">Non-aktif</span>',
            'pending_verifikasi' => '<span class="badge bg-warning">Pending Verifikasi</span>',
        ];
        
        return $badges[$this->status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getKategoriLabelAttribute()
    {
        $labels = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        return $labels[$this->kategori] ?? 'Unknown';
    }

    // Methods
    public function incrementViewCount()
    {
        $this->increment('view_count');
    }

    public function updateRating()
    {
        // Update rating berdasarkan produk
        $avgRating = $this->produk()->avg('rating_rata_rata');
        $totalReviews = $this->produk()->sum('jumlah_review');
        
        $this->update([
            'rating_rata_rata' => round($avgRating, 1),
            'jumlah_ulasan' => $totalReviews
        ]);
    }

    public function getTotalPendapatan()
    {
        return $this->transaksi()
            ->where('status', 'PAID')
            ->sum('amount');
    }

    public function isOpen($day = null, $time = null)
    {
        if (!$this->jam_operasional) {
            return true; // Assume open if no schedule set
        }

        $day = $day ?? strtolower(now()->format('l'));
        $time = $time ?? now()->format('H:i');
        
        $dayInIndonesian = [
            'monday' => 'senin',
            'tuesday' => 'selasa', 
            'wednesday' => 'rabu',
            'thursday' => 'kamis',
            'friday' => 'jumat',
            'saturday' => 'sabtu',
            'sunday' => 'minggu'
        ];
        
        $indonesianDay = $dayInIndonesian[$day] ?? $day;
        
        if (in_array($indonesianDay, $this->hari_libur ?? [])) {
            return false;
        }
        
        $schedule = $this->jam_operasional[$indonesianDay] ?? null;
        if (!$schedule) {
            return false;
        }
        
        [$open, $close] = explode('-', $schedule);
        return $time >= $open && $time <= $close;
    }

    public function getDistance($latitude, $longitude)
    {
        if (!$this->latitude || !$this->longitude) {
            return null;
        }
        
        // Haversine formula
        $earthRadius = 6371; // km
        
        $latFrom = deg2rad($latitude);
        $lonFrom = deg2rad($longitude);
        $latTo = deg2rad($this->latitude);
        $lonTo = deg2rad($this->longitude);
        
        $latDelta = $latTo - $latFrom;
        $lonDelta = $lonTo - $lonFrom;
        
        $a = sin($latDelta / 2) * sin($latDelta / 2) +
             cos($latFrom) * cos($latTo) *
             sin($lonDelta / 2) * sin($lonDelta / 2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        
        return round($earthRadius * $c, 2);
    }
}
