<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class WilayahAdministratif extends Model
{
    use HasFactory;

    protected $table = 'wilayah_administratif';

    protected $fillable = [
        'jenis_wilayah',
        'kode_wilayah',
        'nama_wilayah',
        'parent_id',
        'kepala_wilayah',
        'nik_kepala',
        'nomor_hp_kepala',
        'alamat_kantor',
        'latitude',
        'longitude',
        'batas_wilayah',
        'luas_wilayah',
        'jumlah_kk',
        'jumlah_penduduk',
        'status',
        'keterangan',
        'urutan_tampil',
        'user_id'
    ];

    protected $casts = [
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'luas_wilayah' => 'decimal:2',
        'jumlah_kk' => 'integer',
        'jumlah_penduduk' => 'integer',
        'urutan_tampil' => 'integer'
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(WilayahAdministratif::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(WilayahAdministratif::class, 'parent_id');
    }

    public function penduduk(): HasMany
    {
        return $this->hasMany(Penduduk::class, 'wilayah_id');
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_wilayah', $jenis);
    }

    public function scopeDusun($query)
    {
        return $query->where('jenis_wilayah', 'dusun');
    }

    public function scopeRw($query)
    {
        return $query->where('jenis_wilayah', 'rw');
    }

    public function scopeRt($query)
    {
        return $query->where('jenis_wilayah', 'rt');
    }

    public function scopeOrderByUrutan($query)
    {
        return $query->orderBy('urutan_tampil', 'asc');
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'nonaktif' => '<span class="badge bg-secondary">Non-Aktif</span>'
        ];

        return $badges[$this->status] ?? '';
    }

    public function getJenisWilayahBadgeAttribute()
    {
        $badges = [
            'dusun' => '<span class="badge bg-primary">Dusun</span>',
            'rw' => '<span class="badge bg-info">RW</span>',
            'rt' => '<span class="badge bg-warning">RT</span>'
        ];

        return $badges[$this->jenis_wilayah] ?? '';
    }

    public function getFormattedLuasWilayahAttribute()
    {
        if ($this->luas_wilayah) {
            return number_format($this->luas_wilayah, 2, ',', '.') . ' Ha';
        }
        return '-';
    }

    public function getKepadatanPendudukAttribute()
    {
        if ($this->luas_wilayah > 0) {
            return round($this->jumlah_penduduk / $this->luas_wilayah, 2);
        }
        return 0;
    }

    public function getNamaLengkapAttribute()
    {
        $prefix = strtoupper($this->jenis_wilayah);
        return $prefix . ' ' . $this->nama_wilayah;
    }

    public function getHierarkiAttribute()
    {
        $hierarki = [];
        
        if ($this->parent) {
            $hierarki[] = $this->parent->nama_lengkap;
            if ($this->parent->parent) {
                $hierarki[] = $this->parent->parent->nama_lengkap;
            }
        }
        
        $hierarki[] = $this->nama_lengkap;
        
        return implode(' > ', array_reverse($hierarki));
    }

    // Methods
    public function updateStatistikPenduduk()
    {
        $jumlahKK = 0;
        $jumlahPenduduk = 0;

        if ($this->jenis_wilayah === 'dusun') {
            // Hitung dari RW di bawahnya
            foreach ($this->children as $rw) {
                $rw->updateStatistikPenduduk();
                $jumlahKK += $rw->jumlah_kk;
                $jumlahPenduduk += $rw->jumlah_penduduk;
            }
        } elseif ($this->jenis_wilayah === 'rw') {
            // Hitung dari RT di bawahnya
            foreach ($this->children as $rt) {
                $rt->updateStatistikPenduduk();
                $jumlahKK += $rt->jumlah_kk;
                $jumlahPenduduk += $rt->jumlah_penduduk;
            }
        } else {
            // RT: hitung langsung dari penduduk
            $jumlahKK = Keluarga::whereHas('anggota', function($query) {
                $query->where('rt', $this->nama_wilayah);
            })->count();
            
            $jumlahPenduduk = Penduduk::where('rt', $this->nama_wilayah)->count();
        }

        $this->update([
            'jumlah_kk' => $jumlahKK,
            'jumlah_penduduk' => $jumlahPenduduk
        ]);
    }

    public function nonaktifkan()
    {
        $this->update(['status' => 'nonaktif']);
        
        // Nonaktifkan juga anak-anaknya
        $this->children()->update(['status' => 'nonaktif']);
    }

    public function aktifkan()
    {
        $this->update(['status' => 'aktif']);
    }

    public static function getJenisWilayahOptions()
    {
        return [
            'dusun' => 'Dusun',
            'rw' => 'RW (Rukun Warga)',
            'rt' => 'RT (Rukun Tetangga)'
        ];
    }

    public static function getStatusOptions()
    {
        return [
            'aktif' => 'Aktif',
            'nonaktif' => 'Non-Aktif'
        ];
    }

    public static function getTreeStructure()
    {
        $dusun = self::dusun()->aktif()->orderByUrutan()->get();
        $tree = [];

        foreach ($dusun as $dus) {
            $tree[$dus->id] = [
                'data' => $dus,
                'rw' => []
            ];

            $rwList = $dus->children()->rw()->aktif()->orderByUrutan()->get();
            foreach ($rwList as $rw) {
                $tree[$dus->id]['rw'][$rw->id] = [
                    'data' => $rw,
                    'rt' => $rw->children()->rt()->aktif()->orderByUrutan()->get()
                ];
            }
        }

        return $tree;
    }

    public static function getStatistikWilayah()
    {
        return [
            'total_dusun' => self::dusun()->aktif()->count(),
            'total_rw' => self::rw()->aktif()->count(),
            'total_rt' => self::rt()->aktif()->count(),
            'total_kk' => self::aktif()->sum('jumlah_kk'),
            'total_penduduk' => self::aktif()->sum('jumlah_penduduk'),
            'rata_rata_kk_per_rt' => self::rt()->aktif()->avg('jumlah_kk'),
            'rata_rata_penduduk_per_rt' => self::rt()->aktif()->avg('jumlah_penduduk'),
            'luas_total' => self::aktif()->sum('luas_wilayah')
        ];
    }

    public static function getWilayahByJenis($jenis)
    {
        return self::byJenis($jenis)
            ->aktif()
            ->orderByUrutan()
            ->get();
    }

    public static function getDusunWithRwRt()
    {
        return self::dusun()
            ->aktif()
            ->with(['children.children'])
            ->orderByUrutan()
            ->get();
    }
} 