<?php

namespace App\Services;

use App\Models\User;
use App\Models\Penduduk;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class OtpService
{
    protected $whatsappService;
    protected $otpLength = 6;
    protected $otpExpiry = 5; // minutes

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Generate and send OTP to WhatsApp
     */
    public function sendOtp($phone, $type = 'login')
    {
        try {
            // Format phone number
            $formattedPhone = $this->formatPhoneNumber($phone);
            
            // Check if phone number exists in database
            $penduduk = Penduduk::where('no_wa', 'like', '%' . substr($phone, -10))->first();
            
            if (!$penduduk && $type === 'login') {
                return [
                    'success' => false,
                    'message' => 'Nomor WhatsApp tidak terdaftar dalam sistem'
                ];
            }

            // Generate OTP
            $otp = $this->generateOtp();
            
            // Store OTP in cache
            $cacheKey = "otp_{$type}_{$formattedPhone}";
            $otpData = [
                'otp' => $otp,
                'phone' => $formattedPhone,
                'type' => $type,
                'attempts' => 0,
                'user_data' => $penduduk ? [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'phone' => $penduduk->no_wa
                ] : null,
                'created_at' => now()
            ];

            Cache::put($cacheKey, $otpData, now()->addMinutes($this->otpExpiry));

            // Send OTP via WhatsApp
            $message = $this->getOtpMessage($otp, $type, $penduduk ? $penduduk->nama : null);
            
            $result = $this->whatsappService->sendMessage(
                $formattedPhone, 
                $message, 
                'otp_' . $type,
                null,
                'OTP'
            );

            if ($result['success']) {
                Log::info('OTP sent successfully', [
                    'phone' => $formattedPhone,
                    'type' => $type,
                    'otp' => $otp // Consider removing in production
                ]);

                return [
                    'success' => true,
                    'message' => 'Kode OTP telah dikirim ke WhatsApp Anda',
                    'expires_in' => $this->otpExpiry * 60, // seconds
                    'phone_masked' => $this->maskPhoneNumber($formattedPhone)
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal mengirim OTP: ' . $result['message']
                ];
            }

        } catch (\Exception $e) {
            Log::error('OTP Service Error', [
                'phone' => $phone,
                'type' => $type,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengirim OTP'
            ];
        }
    }

    /**
     * Verify OTP
     */
    public function verifyOtp($phone, $otp, $type = 'login')
    {
        try {
            $formattedPhone = $this->formatPhoneNumber($phone);
            $cacheKey = "otp_{$type}_{$formattedPhone}";
            
            $otpData = Cache::get($cacheKey);
            
            if (!$otpData) {
                return [
                    'success' => false,
                    'message' => 'Kode OTP tidak ditemukan atau sudah kadaluwarsa'
                ];
            }

            // Increment attempts
            $otpData['attempts']++;
            
            // Check max attempts
            if ($otpData['attempts'] > 3) {
                Cache::forget($cacheKey);
                return [
                    'success' => false,
                    'message' => 'Terlalu banyak percobaan. Silakan minta OTP baru'
                ];
            }

            // Update cache with new attempt count
            Cache::put($cacheKey, $otpData, now()->addMinutes($this->otpExpiry));

            // Verify OTP
            if ($otpData['otp'] !== $otp) {
                return [
                    'success' => false,
                    'message' => 'Kode OTP tidak valid',
                    'attempts_left' => 3 - $otpData['attempts']
                ];
            }

            // OTP is valid, remove from cache
            Cache::forget($cacheKey);

            // Handle successful verification based on type
            if ($type === 'login') {
                return $this->handleLoginVerification($otpData);
            } elseif ($type === 'register') {
                return $this->handleRegisterVerification($otpData);
            }

            return [
                'success' => true,
                'message' => 'OTP berhasil diverifikasi',
                'data' => $otpData
            ];

        } catch (\Exception $e) {
            Log::error('OTP Verification Error', [
                'phone' => $phone,
                'type' => $type,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat verifikasi OTP'
            ];
        }
    }

    /**
     * Handle login verification
     */
    protected function handleLoginVerification($otpData)
    {
        if (!$otpData['user_data']) {
            return [
                'success' => false,
                'message' => 'Data pengguna tidak ditemukan'
            ];
        }

        $penduduk = Penduduk::where('nik', $otpData['user_data']['nik'])->first();
        
        if (!$penduduk) {
            return [
                'success' => false,
                'message' => 'Data penduduk tidak ditemukan'
            ];
        }

        // Create or get user account
        $user = User::firstOrCreate(
            ['email' => $penduduk->nik . '@warga.desa'],
            [
                'name' => $penduduk->nama,
                'phone' => $penduduk->no_wa,
                'role' => 'warga',
                'is_active' => true,
                'password' => bcrypt(Str::random(16)), // Random password
                'nik' => $penduduk->nik,
                'verified_at' => now()
            ]
        );

        // Update last login
        $user->update(['last_login_at' => now()]);

        // Generate API token
        $token = $user->createToken('mobile_app')->plainTextToken;

        return [
            'success' => true,
            'message' => 'Login berhasil',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'nama' => $user->name,
                    'nik' => $penduduk->nik,
                    'phone' => $user->phone,
                    'role' => $user->role
                ],
                'penduduk' => [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'alamat' => $penduduk->alamat,
                    'dusun' => $penduduk->dusun,
                    'rt' => $penduduk->rt,
                    'rw' => $penduduk->rw
                ],
                'token' => $token,
                'token_type' => 'Bearer'
            ]
        ];
    }

    /**
     * Handle register verification
     */
    protected function handleRegisterVerification($otpData)
    {
        return [
            'success' => true,
            'message' => 'Verifikasi berhasil. Silakan lengkapi pendaftaran',
            'data' => [
                'phone' => $otpData['phone'],
                'verified' => true
            ]
        ];
    }

    /**
     * Generate OTP
     */
    protected function generateOtp()
    {
        return str_pad(random_int(0, pow(10, $this->otpLength) - 1), $this->otpLength, '0', STR_PAD_LEFT);
    }

    /**
     * Format phone number
     */
    protected function formatPhoneNumber($phone)
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Remove leading zero and add country code (62 for Indonesia)
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        } elseif (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }

        return $phone;
    }

    /**
     * Mask phone number for privacy
     */
    protected function maskPhoneNumber($phone)
    {
        if (strlen($phone) < 8) {
            return $phone;
        }

        return substr($phone, 0, 4) . str_repeat('*', strlen($phone) - 8) . substr($phone, -4);
    }

    /**
     * Get OTP message template
     */
    protected function getOtpMessage($otp, $type, $nama = null)
    {
        $greeting = $nama ? "Halo {$nama}" : "Halo";
        
        $templates = [
            'login' => "🔐 *KODE VERIFIKASI LOGIN*\n\n{$greeting},\n\nKode OTP untuk login ke Portal Warga:\n\n*{$otp}*\n\nKode berlaku selama {$this->otpExpiry} menit.\nJangan bagikan kode ini kepada siapapun!\n\n_Aplikasi Desa Digital_",
            
            'register' => "📝 *KODE VERIFIKASI PENDAFTARAN*\n\n{$greeting},\n\nKode OTP untuk pendaftaran akun:\n\n*{$otp}*\n\nKode berlaku selama {$this->otpExpiry} menit.\nJangan bagikan kode ini kepada siapapun!\n\n_Aplikasi Desa Digital_",
            
            'reset_password' => "🔄 *KODE RESET PASSWORD*\n\n{$greeting},\n\nKode OTP untuk reset password:\n\n*{$otp}*\n\nKode berlaku selama {$this->otpExpiry} menit.\nJangan bagikan kode ini kepada siapapun!\n\n_Aplikasi Desa Digital_"
        ];

        return $templates[$type] ?? $templates['login'];
    }

    /**
     * Check if OTP is still valid
     */
    public function isOtpValid($phone, $type = 'login')
    {
        $formattedPhone = $this->formatPhoneNumber($phone);
        $cacheKey = "otp_{$type}_{$formattedPhone}";
        
        return Cache::has($cacheKey);
    }

    /**
     * Get OTP attempts
     */
    public function getOtpAttempts($phone, $type = 'login')
    {
        $formattedPhone = $this->formatPhoneNumber($phone);
        $cacheKey = "otp_{$type}_{$formattedPhone}";
        
        $otpData = Cache::get($cacheKey);
        
        return $otpData ? $otpData['attempts'] : 0;
    }

    /**
     * Resend OTP (with rate limiting)
     */
    public function resendOtp($phone, $type = 'login')
    {
        // Check rate limiting
        $rateLimitKey = "otp_rate_limit_{$phone}";
        $attempts = Cache::get($rateLimitKey, 0);
        
        if ($attempts >= 3) {
            return [
                'success' => false,
                'message' => 'Terlalu banyak permintaan. Coba lagi dalam 15 menit',
                'retry_after' => 900 // 15 minutes
            ];
        }

        // Increment rate limit counter
        Cache::put($rateLimitKey, $attempts + 1, now()->addMinutes(15));

        return $this->sendOtp($phone, $type);
    }

    /**
     * Clear rate limiting for testing
     */
    public function clearRateLimit($phone)
    {
        $rateLimitKey = "otp_rate_limit_{$phone}";
        Cache::forget($rateLimitKey);
    }
} 