<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('produk_umkm', function (Blueprint $table) {
            $table->id();
            $table->foreignId('umkm_id')->constrained('umkm')->onDelete('cascade');
            $table->string('nama_produk');
            $table->string('sku')->unique()->nullable(); // Stock Keeping Unit
            $table->text('deskripsi');
            $table->text('deskripsi_singkat')->nullable();
            
            // Kategori & Klasifikasi
            $table->enum('kategori_produk', [
                'makanan_segar',
                'makanan_olahan', 
                'minuman',
                'kerajinan_kayu',
                'kerajinan_bambu',
                'kerajinan_tekstil',
                'fashion_pakaian',
                'fashion_aksesoris',
                'elektronik',
                'peralatan_rumah',
                'kosmetik_kecantikan',
                'obat_herbal',
                'tanaman_hias',
                'hasil_pertanian',
                'hasil_peternakan',
                'jasa_digital',
                'jasa_fisik',
                'lainnya'
            ]);
            $table->json('tags')->nullable(); // ["organik", "halal", "handmade", "eco-friendly"]
            
            // Harga & Stok
            $table->decimal('harga_jual', 12, 2);
            $table->decimal('harga_modal', 12, 2)->nullable();
            $table->decimal('harga_diskon', 12, 2)->nullable();
            $table->date('tanggal_mulai_diskon')->nullable();
            $table->date('tanggal_selesai_diskon')->nullable();
            $table->integer('stok')->default(0);
            $table->integer('minimal_stok')->default(5); // Alert ketika stok menipis
            $table->boolean('unlimited_stok')->default(false); // Untuk jasa
            
            // Spesifikasi Produk
            $table->string('satuan', 20)->default('pcs'); // pcs, kg, liter, meter, dll
            $table->decimal('berat', 8, 2)->nullable(); // dalam gram
            $table->json('dimensi')->nullable(); // {"panjang": 10, "lebar": 5, "tinggi": 3} dalam cm
            $table->string('warna')->nullable();
            $table->string('ukuran')->nullable(); // S, M, L, XL atau custom
            $table->text('bahan')->nullable();
            $table->text('cara_penggunaan')->nullable();
            $table->date('tanggal_kadaluarsa')->nullable();
            
            // Media & Dokumentasi
            $table->string('foto_utama')->nullable();
            $table->json('galeri_foto')->nullable(); // Array multiple photos
            $table->string('video_produk')->nullable();
            $table->json('sertifikat_produk')->nullable(); // ["halal.jpg", "organic.pdf"]
            
            // Pengiriman & Logistik
            $table->boolean('bisa_dikirim')->default(true);
            $table->boolean('hanya_cod')->default(false);
            $table->decimal('ongkir_tambahan', 10, 2)->default(0);
            $table->integer('estimasi_pengiriman_hari')->default(1); // Hari kerja
            $table->text('catatan_pengiriman')->nullable();
            
            // Rating & Review
            $table->decimal('rating_rata_rata', 3, 2)->default(0); // 0.00 - 5.00
            $table->integer('jumlah_review')->default(0);
            $table->integer('total_terjual')->default(0);
            $table->integer('view_count')->default(0);
            $table->integer('wishlist_count')->default(0);
            
            // Status & Visibilitas
            $table->enum('status', ['aktif', 'nonaktif', 'habis_stok', 'pending_review'])->default('pending_review');
            $table->boolean('is_featured')->default(false); // Produk unggulan
            $table->boolean('is_bestseller')->default(false);
            $table->boolean('is_new')->default(true);
            $table->integer('urutan_tampil')->default(0);
            
            // SEO & Marketing
            $table->string('slug')->unique();
            $table->string('meta_title')->nullable();
            $table->text('meta_description')->nullable();
            $table->json('keyword')->nullable(); // SEO keywords
            
            // Promo & Marketing
            $table->boolean('eligible_for_promo')->default(true);
            $table->integer('minimal_pembelian')->default(1);
            $table->integer('maksimal_pembelian')->nullable();
            $table->text('syarat_ketentuan')->nullable();
            
            // Metadata
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->timestamps();
            
            // Indexes
            $table->index(['umkm_id', 'status']);
            $table->index(['kategori_produk', 'status']);
            $table->index(['is_featured', 'is_bestseller']);
            $table->index(['harga_jual', 'status']);
            $table->index('slug');
            $table->fullText(['nama_produk', 'deskripsi', 'deskripsi_singkat']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('produk_umkm');
    }
};
