@extends('layouts.app')

@section('title', 'Tambah Penduduk')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.penduduk.index') }}">Data Penduduk</a></li>
    <li class="breadcrumb-item active">Tambah Penduduk</li>
@endsection

@section('content')
<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="mb-2">
            <i class="fas fa-user-plus me-2 text-primary"></i>
            Tambah Data Penduduk
        </h2>
        <p class="text-muted mb-0">Tambahkan data penduduk baru ke dalam sistem</p>
    </div>
    <div>
        <a href="{{ route('admin.penduduk.index') }}" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>
            Kembali
        </a>
    </div>
</div>

<form action="{{ route('admin.penduduk.store') }}" method="POST" enctype="multipart/form-data" id="pendudukForm">
    @csrf
    
    <div class="row">
        <!-- Left Column - Main Data -->
        <div class="col-lg-8">
            <!-- Personal Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-user me-2 text-primary"></i>
                        Data Pribadi
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nik" class="form-label required">NIK</label>
                                <input type="text" 
                                       class="form-control @error('nik') is-invalid @enderror" 
                                       id="nik" 
                                       name="nik" 
                                       value="{{ old('nik') }}" 
                                       placeholder="Nomor Induk Kependudukan"
                                       maxlength="16"
                                       required>
                                @error('nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">16 digit angka NIK</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama" class="form-label required">Nama Lengkap</label>
                                <input type="text" 
                                       class="form-control @error('nama') is-invalid @enderror" 
                                       id="nama" 
                                       name="nama" 
                                       value="{{ old('nama') }}" 
                                       placeholder="Nama lengkap sesuai KTP"
                                       required>
                                @error('nama')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_kk" class="form-label">Nomor KK</label>
                                <input type="text" 
                                       class="form-control @error('no_kk') is-invalid @enderror" 
                                       id="no_kk" 
                                       name="no_kk" 
                                       value="{{ old('no_kk') }}" 
                                       placeholder="Nomor Kartu Keluarga"
                                       maxlength="16">
                                @error('no_kk')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">16 digit angka KK (opsional)</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <!-- Placeholder for balance -->
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="jenis_kelamin" class="form-label required">Jenis Kelamin</label>
                                <select class="form-select @error('jenis_kelamin') is-invalid @enderror" 
                                        id="jenis_kelamin" 
                                        name="jenis_kelamin" 
                                        required>
                                    <option value="">Pilih Jenis Kelamin</option>
                                    <option value="L" {{ old('jenis_kelamin') == 'L' ? 'selected' : '' }}>Laki-laki</option>
                                    <option value="P" {{ old('jenis_kelamin') == 'P' ? 'selected' : '' }}>Perempuan</option>
                                </select>
                                @error('jenis_kelamin')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tempat_lahir" class="form-label required">Tempat Lahir</label>
                                <input type="text" 
                                       class="form-control @error('tempat_lahir') is-invalid @enderror" 
                                       id="tempat_lahir" 
                                       name="tempat_lahir" 
                                       value="{{ old('tempat_lahir') }}" 
                                       placeholder="Kota/Kabupaten tempat lahir"
                                       required>
                                @error('tempat_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tanggal_lahir" class="form-label required">Tanggal Lahir</label>
                                <input type="date" 
                                       class="form-control @error('tanggal_lahir') is-invalid @enderror" 
                                       id="tanggal_lahir" 
                                       name="tanggal_lahir" 
                                       value="{{ old('tanggal_lahir') }}" 
                                       required>
                                @error('tanggal_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="agama" class="form-label required">Agama</label>
                                <select class="form-select @error('agama') is-invalid @enderror" 
                                        id="agama" 
                                        name="agama" 
                                        required>
                                    <option value="">Pilih Agama</option>
                                    <option value="Islam" {{ old('agama') == 'Islam' ? 'selected' : '' }}>Islam</option>
                                    <option value="Kristen" {{ old('agama') == 'Kristen' ? 'selected' : '' }}>Kristen</option>
                                    <option value="Katolik" {{ old('agama') == 'Katolik' ? 'selected' : '' }}>Katolik</option>
                                    <option value="Hindu" {{ old('agama') == 'Hindu' ? 'selected' : '' }}>Hindu</option>
                                    <option value="Buddha" {{ old('agama') == 'Buddha' ? 'selected' : '' }}>Buddha</option>
                                    <option value="Konghucu" {{ old('agama') == 'Konghucu' ? 'selected' : '' }}>Konghucu</option>
                                </select>
                                @error('agama')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status_perkawinan" class="form-label required">Status Perkawinan</label>
                                <select class="form-select @error('status_perkawinan') is-invalid @enderror" 
                                        id="status_perkawinan" 
                                        name="status_perkawinan" 
                                        required>
                                    <option value="">Pilih Status Kawin</option>
                                    <option value="Belum Kawin" {{ old('status_perkawinan') == 'Belum Kawin' ? 'selected' : '' }}>Belum Kawin</option>
                                    <option value="Kawin" {{ old('status_perkawinan') == 'Kawin' ? 'selected' : '' }}>Kawin</option>
                                    <option value="Cerai Hidup" {{ old('status_perkawinan') == 'Cerai Hidup' ? 'selected' : '' }}>Cerai Hidup</option>
                                    <option value="Cerai Mati" {{ old('status_perkawinan') == 'Cerai Mati' ? 'selected' : '' }}>Cerai Mati</option>
                                </select>
                                @error('status_perkawinan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="pekerjaan" class="form-label">Pekerjaan</label>
                                <input type="text" 
                                       class="form-control @error('pekerjaan') is-invalid @enderror" 
                                       id="pekerjaan" 
                                       name="pekerjaan" 
                                       value="{{ old('pekerjaan') }}" 
                                       placeholder="Pekerjaan/Profesi">
                                @error('pekerjaan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="pendidikan_terakhir" class="form-label">Pendidikan Terakhir</label>
                                <select class="form-select @error('pendidikan_terakhir') is-invalid @enderror" 
                                        id="pendidikan_terakhir" 
                                        name="pendidikan_terakhir">
                                    <option value="">Pilih Pendidikan Terakhir</option>
                                    <option value="Tidak Sekolah" {{ old('pendidikan_terakhir') == 'Tidak Sekolah' ? 'selected' : '' }}>Tidak Sekolah</option>
                                    <option value="SD" {{ old('pendidikan_terakhir') == 'SD' ? 'selected' : '' }}>SD/Sederajat</option>
                                    <option value="SMP" {{ old('pendidikan_terakhir') == 'SMP' ? 'selected' : '' }}>SMP/Sederajat</option>
                                    <option value="SMA" {{ old('pendidikan_terakhir') == 'SMA' ? 'selected' : '' }}>SMA/Sederajat</option>
                                    <option value="D1" {{ old('pendidikan_terakhir') == 'D1' ? 'selected' : '' }}>Diploma 1</option>
                                    <option value="D2" {{ old('pendidikan_terakhir') == 'D2' ? 'selected' : '' }}>Diploma 2</option>
                                    <option value="D3" {{ old('pendidikan_terakhir') == 'D3' ? 'selected' : '' }}>Diploma 3</option>
                                    <option value="S1" {{ old('pendidikan_terakhir') == 'S1' ? 'selected' : '' }}>Sarjana (S1)</option>
                                    <option value="S2" {{ old('pendidikan_terakhir') == 'S2' ? 'selected' : '' }}>Magister (S2)</option>
                                    <option value="S3" {{ old('pendidikan_terakhir') == 'S3' ? 'selected' : '' }}>Doktor (S3)</option>
                                </select>
                                @error('pendidikan_terakhir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="golongan_darah" class="form-label">Golongan Darah</label>
                                <select class="form-select @error('golongan_darah') is-invalid @enderror" 
                                        id="golongan_darah" 
                                        name="golongan_darah">
                                    <option value="">Pilih Golongan Darah</option>
                                    <option value="A" {{ old('golongan_darah') == 'A' ? 'selected' : '' }}>A</option>
                                    <option value="B" {{ old('golongan_darah') == 'B' ? 'selected' : '' }}>B</option>
                                    <option value="AB" {{ old('golongan_darah') == 'AB' ? 'selected' : '' }}>AB</option>
                                    <option value="O" {{ old('golongan_darah') == 'O' ? 'selected' : '' }}>O</option>
                                </select>
                                @error('golongan_darah')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="kewarganegaraan" class="form-label">Kewarganegaraan</label>
                        <select class="form-select @error('kewarganegaraan') is-invalid @enderror" 
                                id="kewarganegaraan" 
                                name="kewarganegaraan">
                            <option value="WNI" {{ old('kewarganegaraan', 'WNI') == 'WNI' ? 'selected' : '' }}>WNI (Warga Negara Indonesia)</option>
                            <option value="WNA" {{ old('kewarganegaraan') == 'WNA' ? 'selected' : '' }}>WNA (Warga Negara Asing)</option>
                        </select>
                        @error('kewarganegaraan')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_wa" class="form-label">Nomor WhatsApp</label>
                                <input type="tel" 
                                       class="form-control @error('no_wa') is-invalid @enderror" 
                                       id="no_wa" 
                                       name="no_wa" 
                                       value="{{ old('no_wa') }}" 
                                       placeholder="628123456789"
                                       maxlength="15">
                                @error('no_wa')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    Format: 628xxx (dimulai dengan 628)
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status_hidup" class="form-label">Status Hidup</label>
                                <select class="form-select @error('status_hidup') is-invalid @enderror" 
                                        id="status_hidup" 
                                        name="status_hidup" 
                                        required>
                                    <option value="">Pilih Status Hidup</option>
                                    <option value="Hidup" {{ old('status_hidup') == 'Hidup' ? 'selected' : '' }}>Hidup</option>
                                    <option value="Meninggal" {{ old('status_hidup') == 'Meninggal' ? 'selected' : '' }}>Meninggal</option>
                                </select>
                                @error('status_hidup')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Address Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-map-marker-alt me-2 text-success"></i>
                        Alamat & Lokasi
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="alamat" class="form-label required">Alamat Lengkap</label>
                        <textarea class="form-control @error('alamat') is-invalid @enderror" 
                                  id="alamat" 
                                  name="alamat" 
                                  rows="3" 
                                  placeholder="Alamat lengkap (nama jalan, nomor rumah, dll)"
                                  required>{{ old('alamat') }}</textarea>
                        @error('alamat')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="row">
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label for="rt" class="form-label">RT</label>
                                <input type="text" 
                                       class="form-control @error('rt') is-invalid @enderror" 
                                       id="rt" 
                                       name="rt" 
                                       value="{{ old('rt') }}" 
                                       placeholder="001"
                                       maxlength="3">
                                @error('rt')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label for="rw" class="form-label">RW</label>
                                <input type="text" 
                                       class="form-control @error('rw') is-invalid @enderror" 
                                       id="rw" 
                                       name="rw" 
                                       value="{{ old('rw') }}" 
                                       placeholder="001"
                                       maxlength="3">
                                @error('rw')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label for="dusun" class="form-label">Dusun</label>
                                <input type="text" 
                                       class="form-control @error('dusun') is-invalid @enderror" 
                                       id="dusun" 
                                       name="dusun" 
                                       value="{{ old('dusun') }}" 
                                       placeholder="Nama dusun">
                                @error('dusun')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label for="desa" class="form-label">Desa</label>
                                <input type="text" 
                                       class="form-control @error('desa') is-invalid @enderror" 
                                       id="desa" 
                                       name="desa" 
                                       value="{{ old('desa') }}" 
                                       placeholder="Nama desa">
                                @error('desa')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <!-- Location Picker -->
                    <div class="mb-3">
                        <label class="form-label">Koordinat Lokasi</label>
                        <div class="row">
                            <div class="col-md-6">
                                <label for="latitude" class="form-label visually-hidden">Latitude</label>
                                <input type="text" 
                                       class="form-control @error('latitude') is-invalid @enderror" 
                                       id="latitude" 
                                       name="latitude" 
                                       value="{{ old('latitude') }}" 
                                       placeholder="Latitude"
                                       readonly>
                                @error('latitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="longitude" class="form-label visually-hidden">Longitude</label>
                                <input type="text" 
                                       class="form-control @error('longitude') is-invalid @enderror" 
                                       id="longitude" 
                                       name="longitude" 
                                       value="{{ old('longitude') }}" 
                                       placeholder="Longitude"
                                       readonly>
                                @error('longitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="form-text">
                            Klik pada peta di bawah untuk menentukan lokasi rumah
                        </div>
                    </div>
                    
                    <!-- Map -->
                    <div class="mb-3">
                        <div id="locationMap" class="map-container" style="height: 300px;"></div>
                    </div>
                    
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="getCurrentLocation()">
                        <i class="fas fa-location-arrow me-2"></i>
                        Gunakan Lokasi Saat Ini
                    </button>
                </div>
            </div>
            
            <!-- Birth Certificate Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-certificate me-2 text-info"></i>
                        Data Akta Lahir
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_akta_lahir" class="form-label">Nomor Akta Lahir</label>
                                <input type="text" 
                                       class="form-control @error('no_akta_lahir') is-invalid @enderror" 
                                       id="no_akta_lahir" 
                                       name="no_akta_lahir" 
                                       value="{{ old('no_akta_lahir') }}" 
                                       placeholder="Nomor akta kelahiran"
                                       maxlength="50">
                                @error('no_akta_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tanggal_akta_lahir" class="form-label">Tanggal Akta Lahir</label>
                                <input type="date" 
                                       class="form-control @error('tanggal_akta_lahir') is-invalid @enderror" 
                                       id="tanggal_akta_lahir" 
                                       name="tanggal_akta_lahir" 
                                       value="{{ old('tanggal_akta_lahir') }}">
                                @error('tanggal_akta_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="tempat_akta_lahir" class="form-label">Tempat Akta Lahir</label>
                        <input type="text" 
                               class="form-control @error('tempat_akta_lahir') is-invalid @enderror" 
                               id="tempat_akta_lahir" 
                               name="tempat_akta_lahir" 
                               value="{{ old('tempat_akta_lahir') }}" 
                               placeholder="Tempat pembuatan akta kelahiran">
                        @error('tempat_akta_lahir')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Parent Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-users me-2 text-purple"></i>
                        Data Orang Tua
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="ayah_nik" class="form-label">NIK Ayah</label>
                                <input type="text" 
                                       class="form-control @error('ayah_nik') is-invalid @enderror" 
                                       id="ayah_nik" 
                                       name="ayah_nik" 
                                       value="{{ old('ayah_nik') }}" 
                                       placeholder="NIK Ayah (16 digit)"
                                       maxlength="16">
                                @error('ayah_nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama_ayah" class="form-label">Nama Ayah</label>
                                <input type="text" 
                                       class="form-control @error('nama_ayah') is-invalid @enderror" 
                                       id="nama_ayah" 
                                       name="nama_ayah" 
                                       value="{{ old('nama_ayah') }}" 
                                       placeholder="Nama lengkap ayah">
                                @error('nama_ayah')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Sistem akan mencoba mencocokkan NIK dengan nama
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="ibu_nik" class="form-label">NIK Ibu</label>
                                <input type="text" 
                                       class="form-control @error('ibu_nik') is-invalid @enderror" 
                                       id="ibu_nik" 
                                       name="ibu_nik" 
                                       value="{{ old('ibu_nik') }}" 
                                       placeholder="NIK Ibu (16 digit)"
                                       maxlength="16">
                                @error('ibu_nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama_ibu" class="form-label">Nama Ibu</label>
                                <input type="text" 
                                       class="form-control @error('nama_ibu') is-invalid @enderror" 
                                       id="nama_ibu" 
                                       name="nama_ibu" 
                                       value="{{ old('nama_ibu') }}" 
                                       placeholder="Nama lengkap ibu">
                                @error('nama_ibu')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Sistem akan mencoba mencocokkan NIK dengan nama
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="hubungan_keluarga" class="form-label">Hubungan dalam Keluarga</label>
                        <select class="form-select @error('hubungan_keluarga') is-invalid @enderror" 
                                id="hubungan_keluarga" 
                                name="hubungan_keluarga">
                            <option value="">Pilih Hubungan Keluarga</option>
                            <option value="Kepala Keluarga" {{ old('hubungan_keluarga') == 'Kepala Keluarga' ? 'selected' : '' }}>Kepala Keluarga</option>
                            <option value="Istri" {{ old('hubungan_keluarga') == 'Istri' ? 'selected' : '' }}>Istri</option>
                            <option value="Anak" {{ old('hubungan_keluarga') == 'Anak' ? 'selected' : '' }}>Anak</option>
                            <option value="Menantu" {{ old('hubungan_keluarga') == 'Menantu' ? 'selected' : '' }}>Menantu</option>
                            <option value="Cucu" {{ old('hubungan_keluarga') == 'Cucu' ? 'selected' : '' }}>Cucu</option>
                            <option value="Orang Tua" {{ old('hubungan_keluarga') == 'Orang Tua' ? 'selected' : '' }}>Orang Tua</option>
                            <option value="Mertua" {{ old('hubungan_keluarga') == 'Mertua' ? 'selected' : '' }}>Mertua</option>
                            <option value="Saudara" {{ old('hubungan_keluarga') == 'Saudara' ? 'selected' : '' }}>Saudara</option>
                            <option value="Lainnya" {{ old('hubungan_keluarga') == 'Lainnya' ? 'selected' : '' }}>Lainnya</option>
                        </select>
                        @error('hubungan_keluarga')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Family Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-home me-2 text-warning"></i>
                        Data Keluarga
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="keluarga_id" class="form-label">Keluarga (Nomor KK)</label>
                        <select class="form-select @error('keluarga_id') is-invalid @enderror" 
                                id="keluarga_id" 
                                name="keluarga_id">
                            <option value="">Pilih Keluarga atau Kosongkan untuk Buat Baru</option>
                            @forelse($keluarga ?? [] as $k)
                                <option value="{{ $k->id ?? '' }}" {{ old('keluarga_id') == ($k->id ?? '') ? 'selected' : '' }}>
                                    {{ $k->no_kk ?? 'No KK tidak tersedia' }} - {{ optional($k->kepalaKeluarga)->nama ?? $k->kepala_keluarga_nik ?? 'Kepala keluarga tidak tersedia' }}
                                </option>
                            @empty
                                <option value="" disabled>Belum ada data keluarga</option>
                            @endforelse
                        </select>
                        @error('keluarga_id')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-check">
                        <input class="form-check-input" 
                               type="checkbox" 
                               id="is_kepala_keluarga" 
                               name="is_kepala_keluarga" 
                               value="1" 
                               {{ old('is_kepala_keluarga') ? 'checked' : '' }}>
                        <label class="form-check-label" for="is_kepala_keluarga">
                            Kepala Keluarga
                        </label>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Right Column - Photo & Actions -->
        <div class="col-lg-4">
            <!-- Photo Upload -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-camera me-2 text-info"></i>
                        Foto Penduduk
                    </h5>
                </div>
                <div class="card-body text-center">
                    <div class="photo-preview mb-3">
                        <div id="photoPreview" class="default-avatar-container">
                            <i class="fas fa-user default-avatar-icon"></i>
                            <div class="default-avatar-text">Pilih Foto</div>
                        </div>
                        <img id="photoPreviewImage" 
                             src="" 
                             class="img-fluid rounded-3" 
                             style="max-height: 250px; max-width: 100%; display: none;"
                             alt="Preview Foto">
                    </div>
                    
                    <div class="mb-3">
                        <input type="file" 
                               class="form-control @error('foto') is-invalid @enderror" 
                               id="foto" 
                               name="foto" 
                               accept="image/*"
                               onchange="previewPhoto(this)">
                        @error('foto')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">
                            Format: JPG, PNG, JPEG (Maks. 2MB)
                        </div>
                    </div>
                    
                    <button type="button" class="btn btn-outline-secondary btn-sm" onclick="resetPhoto()">
                        <i class="fas fa-trash me-2"></i>
                        Reset Foto
                    </button>
                </div>
            </div>
            
            <!-- Action Buttons -->
            <div class="card">
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-gradient-primary" id="submitBtn">
                            <i class="fas fa-save me-2"></i>
                            Simpan Data
                        </button>
                        
                        <button type="button" class="btn btn-outline-warning" onclick="saveDraft()">
                            <i class="fas fa-file-alt me-2"></i>
                            Simpan Sebagai Draft
                        </button>
                        
                        <a href="{{ route('admin.penduduk.index') }}" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-2"></i>
                            Batal
                        </a>
                    </div>
                    
                    <div class="mt-3">
                        <small class="text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            Field dengan tanda (*) wajib diisi
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>
@endsection

@push('styles')
<style>
    .required::after {
        content: ' *';
        color: #dc3545;
    }
    
    .photo-preview {
        border: 2px dashed #e9ecef;
        border-radius: 10px;
        padding: 20px;
        transition: all 0.3s ease;
    }
    
    .photo-preview:hover {
        border-color: #667eea;
        background: rgba(102, 126, 234, 0.05);
    }
    
    .map-container {
        border: 2px solid #e9ecef;
        border-radius: 10px;
        overflow: hidden;
    }
    
    .form-control:focus, .form-select:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
    }
    
    .card-header {
        border-bottom: 2px solid #e9ecef;
    }
    
    .btn-gradient-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border: none;
        color: white;
        font-weight: 600;
    }
    
    .btn-gradient-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);
        color: white;
    }
    
    .leaflet-container {
        font-family: inherit;
    }
    
    .default-avatar-container {
        width: 100%;
        max-width: 250px;
        height: 250px;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border: 2px dashed #dee2e6;
        border-radius: 12px;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        margin: 0 auto;
        transition: all 0.3s ease;
    }
    
    .default-avatar-container:hover {
        border-color: #667eea;
        background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(102, 126, 234, 0.05) 100%);
    }
    
    .default-avatar-icon {
        font-size: 3rem;
        color: #9ca3af;
        margin-bottom: 10px;
    }
    
    .default-avatar-text {
        color: #6b7280;
        font-size: 0.9rem;
        font-weight: 500;
    }
</style>
@endpush

@push('scripts')
<script>
let map;
let marker;

document.addEventListener('DOMContentLoaded', function() {
    initializeMap();
    initializeForm();
});

// Initialize map
function initializeMap() {
    // Default coordinates (Bogor)
    const defaultLat = -6.5971;
    const defaultLng = 106.8060;
    
    map = L.map('locationMap').setView([defaultLat, defaultLng], 15);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    // Add click event to map
    map.on('click', function(e) {
        const lat = e.latlng.lat;
        const lng = e.latlng.lng;
        
        setMapLocation(lat, lng);
    });
    
    // Set existing coordinates if available
    const existingLat = document.getElementById('latitude').value;
    const existingLng = document.getElementById('longitude').value;
    
    if (existingLat && existingLng) {
        setMapLocation(parseFloat(existingLat), parseFloat(existingLng));
        map.setView([existingLat, existingLng], 17);
    }
}

// Set location on map
function setMapLocation(lat, lng) {
    // Remove existing marker
    if (marker) {
        map.removeLayer(marker);
    }
    
    // Add new marker
    marker = L.marker([lat, lng]).addTo(map);
    
    // Update input fields
    document.getElementById('latitude').value = lat.toFixed(6);
    document.getElementById('longitude').value = lng.toFixed(6);
    
    // Reverse geocoding (optional - get address from coordinates)
    reverseGeocode(lat, lng);
}

// Get current location
function getCurrentLocation() {
    if (navigator.geolocation) {
        showLoading();
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                setMapLocation(lat, lng);
                map.setView([lat, lng], 17);
                
                hideLoading();
                showToast('Lokasi berhasil didapatkan', 'success');
            },
            function(error) {
                hideLoading();
                let errorMessage = 'Gagal mendapatkan lokasi';
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage = 'Izin lokasi ditolak';
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage = 'Lokasi tidak tersedia';
                        break;
                    case error.TIMEOUT:
                        errorMessage = 'Timeout mendapatkan lokasi';
                        break;
                }
                
                showToast(errorMessage, 'error');
            },
            {
                enableHighAccuracy: true,
                timeout: 5000,
                maximumAge: 0
            }
        );
    } else {
        showToast('Geolocation tidak didukung browser', 'error');
    }
}

// Reverse geocoding
function reverseGeocode(lat, lng) {
    fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&zoom=18&addressdetails=1`)
        .then(response => response.json())
        .then(data => {
            if (data.display_name) {
                // Optionally populate address field
                const addressField = document.getElementById('alamat');
                if (!addressField.value) {
                    addressField.value = data.display_name;
                }
            }
        })
        .catch(error => {
            console.error('Reverse geocoding error:', error);
        });
}

// Photo preview
function previewPhoto(input) {
    if (input.files && input.files[0]) {
        const file = input.files[0];
        
        // Validate file size (2MB)
        if (file.size > 2 * 1024 * 1024) {
            showToast('Ukuran file terlalu besar. Maksimal 2MB.', 'error');
            input.value = '';
            return;
        }
        
        // Validate file type
        if (!file.type.match('image.*')) {
            showToast('File harus berupa gambar.', 'error');
            input.value = '';
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(e) {
            // Hide default avatar container and show image
            document.getElementById('photoPreview').style.display = 'none';
            const previewImg = document.getElementById('photoPreviewImage');
            previewImg.src = e.target.result;
            previewImg.style.display = 'block';
            
            // Add resize info
            showToast('Foto akan diresize otomatis ke ukuran 400x400 pixel untuk konsistensi tampilan.', 'info');
        };
        reader.readAsDataURL(file);
    }
}

// Reset photo
function resetPhoto() {
    document.getElementById('foto').value = '';
    // Show default avatar container and hide image
    document.getElementById('photoPreview').style.display = 'flex';
    document.getElementById('photoPreviewImage').style.display = 'none';
    document.getElementById('photoPreviewImage').src = '';
}

// Initialize form
function initializeForm() {
    // NIK validation
    document.getElementById('nik').addEventListener('input', function() {
        this.value = this.value.replace(/\D/g, '');
        
        if (this.value.length === 16) {
            // Auto-fill some data based on NIK if needed
            validateNIK(this.value);
        }
    });
    
    // Phone number formatting
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', function() {
            this.value = this.value.replace(/\D/g, '');
        });
    });
    
    // Date validation
    document.getElementById('tanggal_lahir').addEventListener('change', function() {
        const today = new Date();
        const birthDate = new Date(this.value);
        
        if (birthDate > today) {
            showToast('Tanggal lahir tidak boleh di masa depan', 'error');
            this.value = '';
        }
    });
    
    // Kepala keluarga logic
    document.getElementById('is_kepala_keluarga').addEventListener('change', function() {
        const keluargaSelect = document.getElementById('keluarga_id');
        
        if (this.checked) {
            keluargaSelect.value = '';
            keluargaSelect.disabled = true;
            showToast('Jika kepala keluarga, sistem akan membuat kartu keluarga baru', 'info');
        } else {
            keluargaSelect.disabled = false;
        }
    });
}

// Validate NIK
function validateNIK(nik) {
    if (nik.length !== 16) {
        return false;
    }
    
    // Basic NIK validation
    const provinceCodes = ['31', '32', '33', '34', '35', '36']; // Add more as needed
    const provinceCode = nik.substring(0, 2);
    
    if (!provinceCodes.includes(provinceCode)) {
        showToast('Kode provinsi NIK tidak valid', 'warning');
    }
    
    return true;
}

// Save as draft
function saveDraft() {
    const formData = new FormData(document.getElementById('pendudukForm'));
    formData.append('is_draft', '1');
    
    showLoading();
    
    fetch('{{ route("admin.penduduk.store") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Data berhasil disimpan sebagai draft', 'success');
        } else {
            showToast(data.message || 'Gagal menyimpan draft', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan saat menyimpan draft', 'error');
    })
    .finally(() => {
        hideLoading();
    });
}

// Form submission
document.getElementById('pendudukForm').addEventListener('submit', function(e) {
    const submitBtn = document.getElementById('submitBtn');
    
    // Disable submit button
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Menyimpan...';
    
    // Re-enable if validation fails
    if (!this.checkValidity()) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-save me-2"></i>Simpan Data';
    }
});

// Auto-save function (optional)
let autoSaveTimer;
function enableAutoSave() {
    const form = document.getElementById('pendudukForm');
    const inputs = form.querySelectorAll('input, select, textarea');
    
    inputs.forEach(input => {
        input.addEventListener('input', function() {
            clearTimeout(autoSaveTimer);
            autoSaveTimer = setTimeout(() => {
                // Auto-save logic here
                console.log('Auto-saving...');
            }, 5000); // Save after 5 seconds of inactivity
        });
    });
}

// Enable auto-save
// enableAutoSave();
</script>
@endpush 
