<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Registrasi Data Penduduk - Portal Desa</title>
    
    <!-- FontAwesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <!-- CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'inter': ['Inter', 'system-ui', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50: '#eff6ff',
                            500: '#3b82f6',
                            600: '#2563eb',
                            700: '#1d4ed8'
                        }
                    }
                }
            }
        }
    </script>
    
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        body {
            font-family: 'Inter', system-ui, sans-serif;
        }
        
        .mobile-card {
            background: white;
            border-radius: 16px;
            padding: 16px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            border: 1px solid #f1f5f9;
            transition: all 0.3s ease;
        }
        
        .mobile-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        
        .mobile-button {
            padding: 1rem 1.5rem;
            border-radius: 0.75rem;
            font-weight: 600;
            text-align: center;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            transform: translateY(0);
        }
        
        .mobile-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        }
        
        .gradient-bg {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.25);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.18);
        }
        
        .form-step {
            display: none;
        }
        
        .form-step.active {
            display: block;
        }
        
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 2rem;
        }
        
        .step {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e5e7eb;
            color: #6b7280;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin: 0 0.5rem;
            transition: all 0.3s ease;
        }
        
        .step.active {
            background: #2563eb;
            color: white;
        }
        
        .step.completed {
            background: #059669;
            color: white;
        }
        
        .step-line {
            flex: 1;
            height: 2px;
            background: #e5e7eb;
            margin: 0 0.5rem;
            align-self: center;
        }
        
        .step-line.active {
            background: #2563eb;
        }
        
        .file-upload-area {
            border: 2px dashed #e5e7eb;
            border-radius: 10px;
            padding: 2rem;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .file-upload-area:hover {
            border-color: #2563eb;
            background: rgba(37, 99, 235, 0.05);
        }
        
        .file-upload-area.dragover {
            border-color: #2563eb;
            background: rgba(37, 99, 235, 0.1);
        }
        
        .required::after {
            content: " *";
            color: #dc2626;
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="min-h-screen">
        <!-- Header -->
        <div class="gradient-bg text-white py-8">
            <div class="container mx-auto px-4">
                <div class="text-center">
                    <h1 class="text-3xl font-bold mb-2">
                        <i class="fas fa-user-plus mr-3"></i>
                        Registrasi Data Penduduk
                    </h1>
                    <p class="text-lg opacity-90">
                        Silakan lengkapi data diri Anda untuk mendaftar sebagai penduduk desa
                    </p>
                    <p class="text-sm opacity-75 mt-2">
                        <i class="fas fa-camera mr-1"></i>Upload foto pribadi dan rumah
                        <i class="fas fa-map-marker-alt ml-3 mr-1"></i>Tagging lokasi GPS
                    </p>
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="container mx-auto px-4 py-8 max-w-4xl">

            
            <!-- Progress Indicator -->
            <div class="step-indicator mb-8">
                <div class="step active" data-step="1">
                    <i class="fas fa-user"></i>
                </div>
                <div class="step-line"></div>
                <div class="step" data-step="2">
                    <i class="fas fa-home"></i>
                </div>
                <div class="step-line"></div>
                <div class="step" data-step="3">
                    <i class="fas fa-id-card"></i>
                </div>
                <div class="step-line"></div>
                <div class="step" data-step="4">
                    <i class="fas fa-camera"></i>
                </div>
                <div class="step-line"></div>
                <div class="step" data-step="5">
                    <i class="fas fa-check"></i>
                </div>
            </div>

            <!-- Alert Messages -->
            <div id="alertContainer"></div>

            <!-- Registration Form -->
            <form id="registrationForm" action="/registrasi-penduduk" method="POST" enctype="multipart/form-data">
                @csrf
                <input type="hidden" name="_token" value="{{ csrf_token() }}">
                
                <!-- Step 1: Personal Information -->
                <div class="form-step active" id="step1">
                    <div class="mobile-card">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">
                            <i class="fas fa-user mr-2 text-blue-600"></i>
                            Data Pribadi
                        </h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label for="nik" class="block text-sm font-medium text-gray-700 required mb-2">NIK</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="nik" 
                                       name="nik" 
                                       placeholder="Nomor Induk Kependudukan"
                                       maxlength="16"
                                       required>
                                <p class="text-xs text-gray-500 mt-1">16 digit angka NIK sesuai KTP</p>
                            </div>
                            
                            <div>
                                <label for="nama" class="block text-sm font-medium text-gray-700 required mb-2">Nama Lengkap</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="nama" 
                                       name="nama" 
                                       placeholder="Nama lengkap sesuai KTP"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="tempat_lahir" class="block text-sm font-medium text-gray-700 required mb-2">Tempat Lahir</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="tempat_lahir" 
                                       name="tempat_lahir" 
                                       placeholder="Kota tempat lahir"
                                       required>
                            </div>
                            
                            <div>
                                <label for="tanggal_lahir" class="block text-sm font-medium text-gray-700 required mb-2">Tanggal Lahir</label>
                                <input type="date" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="tanggal_lahir" 
                                       name="tanggal_lahir" 
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="jenis_kelamin" class="block text-sm font-medium text-gray-700 required mb-2">Jenis Kelamin</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="jenis_kelamin" 
                                        name="jenis_kelamin" 
                                        required>
                                    <option value="">Pilih Jenis Kelamin</option>
                                    <option value="L">Laki-laki</option>
                                    <option value="P">Perempuan</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="agama" class="block text-sm font-medium text-gray-700 required mb-2">Agama</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="agama" 
                                        name="agama" 
                                        required>
                                    <option value="">Pilih Agama</option>
                                    <option value="Islam">Islam</option>
                                    <option value="Kristen">Kristen</option>
                                    <option value="Katolik">Katolik</option>
                                    <option value="Hindu">Hindu</option>
                                    <option value="Buddha">Buddha</option>
                                    <option value="Konghucu">Konghucu</option>
                                </select>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="status_perkawinan" class="block text-sm font-medium text-gray-700 required mb-2">Status Perkawinan</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="status_perkawinan" 
                                        name="status_perkawinan" 
                                        required>
                                    <option value="">Pilih Status</option>
                                    <option value="Belum Kawin">Belum Kawin</option>
                                    <option value="Kawin">Kawin</option>
                                    <option value="Cerai Hidup">Cerai Hidup</option>
                                    <option value="Cerai Mati">Cerai Mati</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="kewarganegaraan" class="block text-sm font-medium text-gray-700 required mb-2">Kewarganegaraan</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="kewarganegaraan" 
                                        name="kewarganegaraan" 
                                        required>
                                    <option value="">Pilih Kewarganegaraan</option>
                                    <option value="WNI">WNI</option>
                                    <option value="WNA">WNA</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="text-right mt-6">
                        <button type="button" class="mobile-button bg-blue-600 text-white hover:bg-blue-700" onclick="nextStep()">
                            Selanjutnya <i class="fas fa-arrow-right ml-2"></i>
                        </button>
                    </div>
                </div>

                <!-- Step 2: Address Information -->
                <div class="form-step" id="step2">
                    <div class="mobile-card">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">
                            <i class="fas fa-home mr-2 text-blue-600"></i>
                            Data Alamat
                        </h3>
                        
                        <div class="mb-4">
                            <label for="alamat" class="block text-sm font-medium text-gray-700 required mb-2">Alamat Lengkap</label>
                            <textarea class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                      id="alamat" 
                                      name="alamat" 
                                      rows="3" 
                                      placeholder="Alamat lengkap tempat tinggal"
                                      required></textarea>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <label for="dusun" class="block text-sm font-medium text-gray-700 mb-2">Dusun</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="dusun" 
                                       name="dusun" 
                                       placeholder="Nama dusun">
                            </div>
                            
                            <div>
                                <label for="rt" class="block text-sm font-medium text-gray-700 mb-2">RT</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="rt" 
                                       name="rt" 
                                       placeholder="Nomor RT"
                                       maxlength="3">
                            </div>
                            
                            <div>
                                <label for="rw" class="block text-sm font-medium text-gray-700 mb-2">RW</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="rw" 
                                       name="rw" 
                                       placeholder="Nomor RW"
                                       maxlength="3">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="no_kk" class="block text-sm font-medium text-gray-700 mb-2">Nomor Kartu Keluarga</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="no_kk" 
                                       name="no_kk" 
                                       placeholder="Nomor KK (opsional)"
                                       maxlength="16">
                                <p class="text-xs text-gray-500 mt-1">Kosongkan jika belum memiliki KK</p>
                            </div>
                            
                            <div>
                                <label for="no_wa" class="block text-sm font-medium text-gray-700 mb-2">Nomor WhatsApp</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="no_wa" 
                                       name="no_wa" 
                                       placeholder="08xxxxxxxxxx">
                                <p class="text-xs text-gray-500 mt-1">Untuk notifikasi layanan</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="flex justify-between mt-6">
                        <button type="button" class="mobile-button bg-gray-500 text-white hover:bg-gray-600" onclick="prevStep()">
                            <i class="fas fa-arrow-left mr-2"></i>Sebelumnya
                        </button>
                        <button type="button" class="mobile-button bg-blue-600 text-white hover:bg-blue-700" onclick="nextStep()">
                            Selanjutnya <i class="fas fa-arrow-right ml-2"></i>
                        </button>
                    </div>
                </div>

                <!-- Step 3: Additional Information -->
                <div class="form-step" id="step3">
                    <div class="mobile-card">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">
                            <i class="fas fa-id-card mr-2 text-blue-600"></i>
                            Data Tambahan
                        </h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label for="pekerjaan" class="block text-sm font-medium text-gray-700 mb-2">Pekerjaan</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="pekerjaan" 
                                       name="pekerjaan" 
                                       placeholder="Pekerjaan saat ini">
                            </div>
                            
                            <div>
                                <label for="pendidikan_terakhir" class="block text-sm font-medium text-gray-700 mb-2">Pendidikan Terakhir</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="pendidikan_terakhir" 
                                        name="pendidikan_terakhir">
                                    <option value="">Pilih Pendidikan</option>
                                    <option value="Tidak Sekolah">Tidak Sekolah</option>
                                    <option value="SD">SD</option>
                                    <option value="SMP">SMP</option>
                                    <option value="SMA">SMA</option>
                                    <option value="D1">D1</option>
                                    <option value="D2">D2</option>
                                    <option value="D3">D3</option>
                                    <option value="S1">S1</option>
                                    <option value="S2">S2</option>
                                    <option value="S3">S3</option>
                                </select>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="golongan_darah" class="block text-sm font-medium text-gray-700 mb-2">Golongan Darah</label>
                                <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                        id="golongan_darah" 
                                        name="golongan_darah">
                                    <option value="">Pilih Golongan Darah</option>
                                    <option value="A">A</option>
                                    <option value="B">B</option>
                                    <option value="AB">AB</option>
                                    <option value="O">O</option>
                                </select>
                            </div>
                            
                            <div>
                                <label for="nama_ayah" class="block text-sm font-medium text-gray-700 mb-2">Nama Ayah</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="nama_ayah" 
                                       name="nama_ayah" 
                                       placeholder="Nama ayah kandung">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                            <div>
                                <label for="nama_ibu" class="block text-sm font-medium text-gray-700 mb-2">Nama Ibu</label>
                                <input type="text" 
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" 
                                       id="nama_ibu" 
                                       name="nama_ibu" 
                                       placeholder="Nama ibu kandung">
                            </div>
                            

                        </div>
                    </div>
                    
                    <div class="flex justify-between mt-6">
                        <button type="button" class="mobile-button bg-gray-500 text-white hover:bg-gray-600" onclick="prevStep()">
                            <i class="fas fa-arrow-left mr-2"></i>Sebelumnya
                        </button>
                        <button type="button" class="mobile-button bg-blue-600 text-white hover:bg-blue-700" onclick="nextStep()">
                            Selanjutnya <i class="fas fa-arrow-right ml-2"></i>
                        </button>
                    </div>
                </div>

                <!-- Step 4: Foto dan Lokasi -->
                <div class="form-step" id="step4">
                    <div class="mobile-card">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">
                            <i class="fas fa-camera mr-2 text-blue-600"></i>
                            Foto dan Lokasi
                        </h3>
                        
                        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                            <div class="flex items-center">
                                <i class="fas fa-exclamation-triangle text-yellow-600 mr-2"></i>
                                <span class="text-yellow-800 font-medium">Penting:</span>
                            </div>
                            <p class="text-yellow-700 text-sm mt-1">Foto dan lokasi rumah akan membantu petugas dalam verifikasi data Anda. Pastikan foto jelas dan lokasi akurat. Minimal upload satu foto untuk melanjutkan.</p>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <!-- Foto Pribadi -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Foto Pribadi</label>
                                <div class="border-2 border-dashed border-gray-300 rounded-lg p-4 text-center hover:border-blue-400 transition-colors cursor-pointer" onclick="document.getElementById('fotoInput').click()">
                                    <input type="file" name="foto" accept="image/*" class="hidden" id="fotoInput">
                                    <div class="space-y-2">
                                        <i class="fas fa-camera text-3xl text-gray-400"></i>
                                        <div class="text-sm text-gray-600">
                                            <span class="font-medium text-blue-600">Klik untuk upload</span> foto pribadi
                                        </div>
                                        <div class="text-xs text-gray-500">PNG, JPG, JPEG (Max. 2MB)</div>
                                        <div class="text-xs text-gray-400 mt-1">Foto akan membantu verifikasi identitas</div>
                                    </div>
                                </div>
                                <div id="fotoPreview" class="mt-2 hidden">
                                    <img src="" alt="Preview" class="w-20 h-20 object-cover rounded-lg">
                                </div>
                            </div>
                            
                            <!-- Foto Rumah -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Foto Rumah</label>
                                <div class="border-2 border-dashed border-gray-300 rounded-lg p-4 text-center hover:border-blue-400 transition-colors cursor-pointer" onclick="document.getElementById('fotoRumahInput').click()">
                                    <input type="file" name="foto_rumah" accept="image/*" class="hidden" id="fotoRumahInput">
                                    <div class="space-y-2">
                                        <i class="fas fa-home text-3xl text-gray-400"></i>
                                        <div class="text-sm text-gray-600">
                                            <span class="font-medium text-blue-600">Klik untuk upload</span> foto rumah
                                        </div>
                                        <div class="text-xs text-gray-500">PNG, JPG, JPEG (Max. 2MB)</div>
                                        <div class="text-xs text-gray-400 mt-1">Foto rumah untuk verifikasi alamat</div>
                                    </div>
                                </div>
                                <div id="fotoRumahPreview" class="mt-2 hidden">
                                    <img src="" alt="Preview" class="w-20 h-20 object-cover rounded-lg">
                                </div>
                            </div>
                        </div>
                        
                        <!-- Lokasi -->
                        <div class="mt-6">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Lokasi Rumah</label>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-xs text-gray-600 mb-1">Latitude</label>
                                    <input type="number" name="latitude" step="any" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="-6.123456">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-600 mb-1">Longitude</label>
                                    <input type="number" name="longitude" step="any" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="106.123456">
                                </div>
                            </div>
                            <div class="mt-2">
                                <button type="button" id="getLocationBtn" class="inline-flex items-center px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <i class="fas fa-map-marker-alt mr-2"></i>
                                    Ambil Lokasi Sekarang
                                </button>
                                <p class="text-xs text-gray-500 mt-1">Klik tombol di atas untuk mengambil lokasi otomatis dari GPS. Pastikan browser mengizinkan akses lokasi.</p>
                                <p class="text-xs text-gray-400 mt-1">Lokasi akan membantu petugas dalam verifikasi alamat Anda</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="flex justify-between mt-6">
                        <button type="button" class="mobile-button bg-gray-500 text-white hover:bg-gray-600" onclick="prevStep()">
                            <i class="fas fa-arrow-left mr-2"></i>Sebelumnya
                        </button>
                        <button type="button" class="mobile-button bg-blue-600 text-white hover:bg-blue-700" onclick="nextStep()">
                            Selanjutnya <i class="fas fa-arrow-right ml-2"></i>
                        </button>
                    </div>
                </div>

                <!-- Step 5: Review and Submit -->
                <div class="form-step" id="step5">
                    <div class="mobile-card">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">
                            <i class="fas fa-check mr-2 text-blue-600"></i>
                            Review Data
                        </h3>
                        
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                            <div class="flex items-center">
                                <i class="fas fa-info-circle text-blue-600 mr-2"></i>
                                <span class="text-blue-800 font-medium">Perhatian:</span>
                            </div>
                            <p class="text-blue-700 text-sm mt-1">Silakan periksa kembali data yang telah diisi sebelum mengirimkan. Foto dan lokasi rumah akan membantu petugas dalam verifikasi data Anda. Data akan disimpan dan diverifikasi oleh petugas.</p>
                        </div>

                        <div id="reviewData">
                            <!-- Data will be populated by JavaScript -->
                        </div>
                        
                        <div class="bg-green-50 border border-green-200 rounded-lg p-4 mt-6">
                            <div class="flex items-center">
                                <i class="fas fa-check-circle text-green-600 mr-2"></i>
                                <span class="text-green-800 font-medium">Data Lengkap:</span>
                            </div>
                            <p class="text-green-700 text-sm mt-1">Foto dan lokasi telah diupload. Data Anda siap untuk diverifikasi oleh petugas. Setelah dikirim, data akan diproses dalam 1-3 hari kerja.</p>
                        </div>
                    </div>
                    
                    <div class="flex justify-between mt-6">
                        <button type="button" class="mobile-button bg-gray-500 text-white hover:bg-gray-600" onclick="prevStep()">
                            <i class="fas fa-arrow-left mr-2"></i>Sebelumnya
                        </button>
                        <button type="submit" class="mobile-button bg-green-600 text-white hover:bg-green-700" id="submitBtn">
                            <i class="fas fa-paper-plane mr-2"></i>Kirim Registrasi
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- JavaScript -->
    <script>
        // Refresh CSRF token on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Get fresh CSRF token
            fetch('/csrf-token')
                .then(response => response.json())
                .then(data => {
                    // Update meta tag
                    const metaTag = document.querySelector('meta[name="csrf-token"]');
                    if (metaTag) {
                        metaTag.content = data.token;
                    }
                    
                    // Update form input
                    const csrfInput = document.querySelector('input[name="_token"]');
                    if (csrfInput) {
                        csrfInput.value = data.token;
                    }
                    
                    console.log('CSRF token refreshed:', data.token);
                })
                .catch(error => {
                    console.log('Failed to refresh CSRF token:', error);
                });
        });
        
        let currentStep = 1;
        const totalSteps = 5;

        // Step navigation
        function nextStep() {
            if (validateCurrentStep()) {
                if (currentStep < totalSteps) {
                    document.getElementById(`step${currentStep}`).classList.remove('active');
                    currentStep++;
                    document.getElementById(`step${currentStep}`).classList.add('active');
                    updateStepIndicator();
                    
                    if (currentStep === 5) {
                        populateReviewData();
                    }
                    
                    // Auto-save when moving to next step
                    setTimeout(saveFormData, 100);
                }
            }
        }

        function prevStep() {
            if (currentStep > 1) {
                document.getElementById(`step${currentStep}`).classList.remove('active');
                currentStep--;
                document.getElementById(`step${currentStep}`).classList.add('active');
                updateStepIndicator();
                
                // Auto-save when moving to previous step
                setTimeout(saveFormData, 100);
            }
        }

        function updateStepIndicator() {
            const steps = document.querySelectorAll('.step');
            const lines = document.querySelectorAll('.step-line');
            
            steps.forEach((step, index) => {
                const stepNum = index + 1;
                step.classList.remove('active', 'completed');
                
                if (stepNum < currentStep) {
                    step.classList.add('completed');
                } else if (stepNum === currentStep) {
                    step.classList.add('active');
                }
            });
            
            lines.forEach((line, index) => {
                const stepNum = index + 1;
                line.classList.remove('active');
                
                if (stepNum < currentStep) {
                    line.classList.add('active');
                }
            });
        }

        function validateCurrentStep() {
            const currentStepElement = document.getElementById(`step${currentStep}`);
            const requiredFields = currentStepElement.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.classList.add('border-red-500');
                    isValid = false;
                } else {
                    field.classList.remove('border-red-500');
                }
            });
            
            // Special validation for NIK
            if (currentStep === 1) {
                const nik = document.getElementById('nik').value;
                if (nik.length !== 16 || !/^\d+$/.test(nik)) {
                    document.getElementById('nik').classList.add('border-red-500');
                    showAlert('NIK harus 16 digit angka', 'error');
                    isValid = false;
                }
            }
            
                    // Additional validation for step 4 (Foto dan Lokasi)
        if (currentStep === 4) {
            const fotoInput = document.getElementById('fotoInput');
            const fotoRumahInput = document.getElementById('fotoRumahInput');
            
            if (fotoInput && fotoRumahInput && !fotoInput.files[0] && !fotoRumahInput.files[0]) {
                showAlert('Silakan upload minimal satu foto (foto pribadi atau foto rumah) untuk membantu verifikasi.', 'error');
                isValid = false;
            }
        }
            
            if (!isValid) {
                showAlert('Mohon lengkapi data yang wajib diisi', 'error');
            }
            
            return isValid;
        }

        function populateReviewData() {
            const formData = new FormData(document.getElementById('registrationForm'));
            const reviewDiv = document.getElementById('reviewData');
            let reviewHTML = '<div class="grid grid-cols-1 md:grid-cols-2 gap-6">';
            
            // Personal Information
            reviewHTML += `
                <div>
                    <h6 class="text-blue-600 font-semibold mb-3">Data Pribadi</h6>
                    <div class="space-y-2 text-sm">
                        <div><span class="font-medium">NIK:</span> ${formData.get('nik')}</div>
                        <div><span class="font-medium">Nama:</span> ${formData.get('nama')}</div>
                        <div><span class="font-medium">Tempat Lahir:</span> ${formData.get('tempat_lahir')}</div>
                        <div><span class="font-medium">Tanggal Lahir:</span> ${formData.get('tanggal_lahir')}</div>
                        <div><span class="font-medium">Jenis Kelamin:</span> ${getSelectText('jenis_kelamin')}</div>
                        <div><span class="font-medium">Agama:</span> ${getSelectText('agama')}</div>
                        <div><span class="font-medium">Status:</span> ${getSelectText('status_perkawinan')}</div>
                        <div><span class="font-medium">Kewarganegaraan:</span> ${getSelectText('kewarganegaraan')}</div>
                    </div>
                </div>
            `;
            
            // Address Information
            reviewHTML += `
                <div>
                    <h6 class="text-blue-600 font-semibold mb-3">Data Alamat</h6>
                    <div class="space-y-2 text-sm">
                        <div><span class="font-medium">Alamat:</span> ${formData.get('alamat')}</div>
                        <div><span class="font-medium">Dusun:</span> ${formData.get('dusun') || '-'}</div>
                        <div><span class="font-medium">RT/RW:</span> ${formData.get('rt') || '-'}/${formData.get('rw') || '-'}</div>
                        <div><span class="font-medium">No. KK:</span> ${formData.get('no_kk') || '-'}</div>
                        <div><span class="font-medium">WhatsApp:</span> ${formData.get('no_wa') || '-'}</div>
                    </div>
                </div>
            `;
            
            // Additional Information
            reviewHTML += `
                <div class="md:col-span-2">
                    <h6 class="text-blue-600 font-semibold mb-3">Data Tambahan</h6>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div><span class="font-medium">Pekerjaan:</span> ${formData.get('pekerjaan') || '-'}</div>
                        <div><span class="font-medium">Pendidikan:</span> ${getSelectText('pendidikan_terakhir') || '-'}</div>
                        <div><span class="font-medium">Golongan Darah:</span> ${getSelectText('golongan_darah') || '-'}</div>
                        <div><span class="font-medium">Nama Ayah:</span> ${formData.get('nama_ayah') || '-'}</div>
                        <div><span class="font-medium">Nama Ibu:</span> ${formData.get('nama_ibu') || '-'}</div>
                        <div><span class="font-medium">Foto Pribadi:</span> ${document.getElementById('fotoInput') && document.getElementById('fotoInput').files.length > 0 ? '✓ Diupload' : '-'}</div>
                        <div><span class="font-medium">Foto Rumah:</span> ${document.getElementById('fotoRumahInput') && document.getElementById('fotoRumahInput').files.length > 0 ? '✓ Diupload' : '-'}</div>
                        <div><span class="font-medium">Latitude:</span> ${formData.get('latitude') || '-'}</div>
                        <div><span class="font-medium">Longitude:</span> ${formData.get('longitude') || '-'}</div>
                    </div>
                </div>
            `;
            
            reviewHTML += '</div>';
            reviewDiv.innerHTML = reviewHTML;
        }

        function getSelectText(selectId) {
            const select = document.getElementById(selectId);
            return select.options[select.selectedIndex].text;
        }

        function showAlert(message, type) {
            const alertContainer = document.getElementById('alertContainer');
            const alertClass = type === 'error' ? 'bg-red-100 border-red-400 text-red-700' : 'bg-green-100 border-green-400 text-green-700';
            const iconClass = type === 'error' ? 'fas fa-exclamation-circle' : 'fas fa-check-circle';
            
            const alertHTML = `
                <div class="border rounded-lg p-4 mb-4 ${alertClass}">
                    <div class="flex items-center">
                        <i class="${iconClass} mr-2"></i>
                        <span>${message}</span>
                    </div>
                </div>
            `;
            alertContainer.innerHTML = alertHTML;
            
            // Auto dismiss after 5 seconds
            setTimeout(() => {
                const alert = alertContainer.querySelector('div');
                if (alert) {
                    alert.remove();
                }
            }, 5000);
        }

        // File upload handling
        document.getElementById('fotoInput').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const uploadArea = document.querySelector('.file-upload-area');
            
            if (file) {
                if (file.size > 2 * 1024 * 1024) { // 2MB
                    showAlert('Ukuran file terlalu besar. Maksimal 2MB.', 'error');
                    this.value = '';
                    return;
                }
                
                if (!file.type.startsWith('image/')) {
                    showAlert('File harus berupa gambar.', 'error');
                    this.value = '';
                    return;
                }
                
                uploadArea.innerHTML = `
                    <i class="fas fa-check-circle text-3xl text-green-600 mb-3"></i>
                    <h6 class="font-semibold mb-2">File Terpilih</h6>
                    <p class="text-gray-500 text-sm mb-1">${file.name}</p>
                    <small class="text-gray-400">${(file.size / 1024 / 1024).toFixed(2)} MB</small>
                `;
            }
            
            // Auto-save after file upload
            setTimeout(saveFormData, 100);
        });

        // Auto-save on form field changes
        document.getElementById('registrationForm').addEventListener('input', function(e) {
            if (e.target.type !== 'file') {
                setTimeout(saveFormData, 500); // Debounce save
            }
        });

        // Auto-save on select changes
        document.getElementById('registrationForm').addEventListener('change', function(e) {
            if (e.target.tagName === 'SELECT') {
                setTimeout(saveFormData, 100);
            }
        });

        // File upload preview
        const fotoInput = document.getElementById('fotoInput');
        if (fotoInput) {
            fotoInput.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    // Validate file size (2MB = 2 * 1024 * 1024 bytes)
                    if (file.size > 2 * 1024 * 1024) {
                        alert('Ukuran file terlalu besar. Maksimal 2MB.');
                        this.value = '';
                        return;
                    }
                    
                    // Validate file type
                    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png'];
                    if (!allowedTypes.includes(file.type)) {
                        alert('Format file tidak didukung. Gunakan JPG, JPEG, atau PNG.');
                        this.value = '';
                        return;
                    }
                    
                                    const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('fotoPreview');
                    if (preview && preview.querySelector('img')) {
                        preview.querySelector('img').src = e.target.result;
                        preview.classList.remove('hidden');
                    }
                };
                reader.readAsDataURL(file);
                }
                // Auto-save after file selection
                setTimeout(saveFormData, 100);
            });
        }

        const fotoRumahInput = document.getElementById('fotoRumahInput');
        if (fotoRumahInput) {
            fotoRumahInput.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    // Validate file size (2MB = 2 * 1024 * 1024 bytes)
                    if (file.size > 2 * 1024 * 1024) {
                        alert('Ukuran file terlalu besar. Maksimal 2MB.');
                        this.value = '';
                        return;
                    }
                    
                    // Validate file type
                    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png'];
                    if (!allowedTypes.includes(file.type)) {
                        alert('Format file tidak didukung. Gunakan JPG, JPEG, atau PNG.');
                        this.value = '';
                        return;
                    }
                    
                                    const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('fotoRumahPreview');
                    if (preview && preview.querySelector('img')) {
                        preview.querySelector('img').src = e.target.result;
                        preview.classList.remove('hidden');
                    }
                };
                reader.readAsDataURL(file);
                }
                // Auto-save after file selection
                setTimeout(saveFormData, 100);
            });
        }

        // Auto-save for location fields
        const latitudeInput = document.querySelector('input[name="latitude"]');
        const longitudeInput = document.querySelector('input[name="longitude"]');
        
        if (latitudeInput) {
            latitudeInput.addEventListener('input', saveFormData);
        }
        if (longitudeInput) {
            longitudeInput.addEventListener('input', saveFormData);
        }

        // GPS Location
        const getLocationBtn = document.getElementById('getLocationBtn');
        if (getLocationBtn) {
            getLocationBtn.addEventListener('click', function() {
            if (navigator.geolocation) {
                this.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Mengambil lokasi...';
                this.disabled = true;
                
                navigator.geolocation.getCurrentPosition(
                    function(position) {
                        document.querySelector('input[name="latitude"]').value = position.coords.latitude;
                        document.querySelector('input[name="longitude"]').value = position.coords.longitude;
                        
                        // Auto-save after getting location
                        setTimeout(saveFormData, 100);
                        
                        document.getElementById('getLocationBtn').innerHTML = '<i class="fas fa-check mr-2"></i>Lokasi berhasil diambil';
                        document.getElementById('getLocationBtn').classList.add('bg-green-100', 'text-green-700', 'border-green-300');
                        
                        setTimeout(() => {
                            document.getElementById('getLocationBtn').innerHTML = '<i class="fas fa-map-marker-alt mr-2"></i>Ambil Lokasi Sekarang';
                            document.getElementById('getLocationBtn').classList.remove('bg-green-100', 'text-green-700', 'border-green-300');
                            document.getElementById('getLocationBtn').disabled = false;
                        }, 2000);
                    },
                    function(error) {
                        console.error('Error getting location:', error);
                        alert('Gagal mengambil lokasi. Pastikan Anda mengizinkan akses lokasi.');
                        document.getElementById('getLocationBtn').innerHTML = '<i class="fas fa-map-marker-alt mr-2"></i>Ambil Lokasi Sekarang';
                        document.getElementById('getLocationBtn').disabled = false;
                    },
                    {
                        enableHighAccuracy: true,
                        timeout: 10000,
                        maximumAge: 60000
                    }
                );
            } else {
                alert('Browser Anda tidak mendukung geolokasi.');
            }
        });
        }

        // Form submission
        document.getElementById('registrationForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            console.log('=== FORM SUBMISSION START ===');
            console.log('Form action:', this.action);
            console.log('Form method:', this.method);
            console.log('Form enctype:', this.enctype);
            
            const submitBtn = document.getElementById('submitBtn');
            const originalText = submitBtn.innerHTML;
            
            // Show loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Mengirim...';
            
            // Get CSRF token from multiple sources
            let csrfToken = '';
            
            // Try to get from meta tag
            const csrfMeta = document.querySelector('meta[name="csrf-token"]');
            if (csrfMeta) {
                csrfToken = csrfMeta.content;
                console.log('CSRF token from meta tag:', csrfToken);
            }
            
            // If not found in meta, try to get from form input
            if (!csrfToken) {
                const csrfInput = document.querySelector('input[name="_token"]');
                if (csrfInput) {
                    csrfToken = csrfInput.value;
                    console.log('CSRF token from form input:', csrfToken);
                }
            }
            
            // If still not found, try to get from cookie
            if (!csrfToken) {
                const cookies = document.cookie.split(';');
                for (let cookie of cookies) {
                    const [name, value] = cookie.trim().split('=');
                    if (name === 'XSRF-TOKEN') {
                        csrfToken = decodeURIComponent(value);
                        console.log('CSRF token from cookie:', csrfToken);
                        break;
                    }
                }
            }
            
            // If still not found, fetch from server
            if (!csrfToken) {
                console.log('Fetching CSRF token from server...');
                fetch('/csrf-token')
                    .then(response => response.json())
                    .then(data => {
                        csrfToken = data.token;
                        console.log('CSRF token fetched from server:', csrfToken);
                        
                        // Update meta tag and form input
                        if (csrfMeta) {
                            csrfMeta.content = csrfToken;
                        }
                        const csrfInput = document.querySelector('input[name="_token"]');
                        if (csrfInput) {
                            csrfInput.value = csrfToken;
                        }
                    })
                    .catch(error => {
                        console.log('Failed to fetch CSRF token:', error);
                    });
            }
            
            console.log('CSRF Token found:', csrfToken ? 'Yes' : 'No');
            console.log('CSRF Token value:', csrfToken);
            
            // Debug form data
            const formData = new FormData(this);
            console.log('Form data entries:');
            for (let [key, value] of formData.entries()) {
                console.log(key + ':', value);
            }
            
            // Test CSRF token first
            if (csrfToken) {
                console.log('Testing CSRF token...');
                fetch('/test-csrf', {
                    method: 'POST',
                    headers: {
                        'Accept': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': csrfToken
                    }
                })
                .then(response => {
                    console.log('CSRF test response:', response.status);
                    if (response.ok) {
                        console.log('CSRF token valid');
                    } else {
                        console.log('CSRF token invalid');
                    }
                })
                .catch(error => {
                    console.log('CSRF test error:', error);
                });
            }
            
            // Prepare headers
            const headers = {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            };
            
            if (csrfToken) {
                headers['X-CSRF-TOKEN'] = csrfToken;
            }
            
            // Submit form
            console.log('Submitting form with headers:', headers);
            console.log('Form action:', this.action);
            console.log('Form method:', this.method);
            
            fetch(this.action, {
                method: 'POST',
                body: new FormData(this),
                headers: headers
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                return response.json();
            })
            .then(data => {
                console.log('Response data:', data);
                
                if (data.success) {
                    showAlert(data.message, 'success');
                    // Clear saved form data on successful submission
                    clearFormData();
                    // Redirect after 2 seconds
                    setTimeout(() => {
                        window.location.href = data.redirect || '/';
                    }, 2000);
                } else {
                    let errorMessage = 'Terjadi kesalahan saat registrasi';
                    
                    if (data.message) {
                        errorMessage = data.message;
                    } else if (data.errors) {
                        // Handle validation errors
                        const errorKeys = Object.keys(data.errors);
                        if (errorKeys.length > 0) {
                            errorMessage = data.errors[errorKeys[0]][0];
                        }
                    }
                    
                    showAlert(errorMessage, 'error');
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                }
            })
            .catch(error => {
                console.error('Fetch error:', error);
                console.error('Error details:', {
                    message: error.message,
                    stack: error.stack
                });
                
                // Try to get response text for more details
                if (error.response) {
                    error.response.text().then(text => {
                        console.log('Response text:', text);
                    });
                }
                
                let errorMessage = 'Terjadi kesalahan jaringan';
                
                if (error.message.includes('Failed to fetch')) {
                    errorMessage = 'Tidak dapat terhubung ke server. Periksa koneksi internet Anda.';
                } else if (error.message.includes('HTTP error')) {
                    if (error.message.includes('404')) {
                        errorMessage = 'URL tidak ditemukan. Periksa konfigurasi server.';
                    } else if (error.message.includes('419')) {
                        errorMessage = 'Sesi telah berakhir. Mencoba memperbarui sesi...';
                        
                        // Try to refresh session first
                        fetch('/refresh-session')
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    showAlert('Sesi berhasil diperbarui. Silakan coba submit lagi.', 'success');
                                    // Update CSRF tokens
                                    const metaTag = document.querySelector('meta[name="csrf-token"]');
                                    if (metaTag) {
                                        metaTag.content = data.csrf_token;
                                    }
                                    const csrfInput = document.querySelector('input[name="_token"]');
                                    if (csrfInput) {
                                        csrfInput.value = data.csrf_token;
                                    }
                                    submitBtn.disabled = false;
                                    submitBtn.innerHTML = originalText;
                                } else {
                                    // If refresh fails, try traditional submission
                                    console.log('Session refresh failed, trying traditional submission...');
                                    submitFormTraditionally();
                                }
                            })
                            .catch(() => {
                                // If refresh fails, try traditional submission
                                console.log('Session refresh failed, trying traditional submission...');
                                submitFormTraditionally();
                            });
                    } else if (error.message.includes('422')) {
                        errorMessage = 'Data yang dimasukkan tidak valid. Periksa kembali data Anda.';
                    } else if (error.message.includes('500')) {
                        errorMessage = 'Server error. Periksa log Laravel untuk detail error.';
                    } else {
                        errorMessage = 'Server error. Silakan coba lagi nanti.';
                    }
                }
                
                if (!error.message.includes('419')) {
                    showAlert(errorMessage, 'error');
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                }
                
                // If CSRF error, try traditional form submission
                if (error.message.includes('419')) {
                    console.log('Trying traditional form submission...');
                    showAlert('Mencoba submit form secara tradisional...', 'info');
                    
                    // Remove event listener temporarily
                    this.removeEventListener('submit', arguments.callee);
                    
                    // Submit form normally after a short delay
                    setTimeout(() => {
                        console.log('Submitting form traditionally...');
                        this.submit();
                    }, 1000);
                }
            });
        });
        
        // Function to submit form traditionally
        function submitFormTraditionally() {
            const form = document.getElementById('registrationForm');
            if (form) {
                // Remove event listener temporarily
                form.removeEventListener('submit', arguments.callee);
                
                // Submit form normally
                console.log('Submitting form traditionally...');
                form.submit();
            }
        }

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            updateStepIndicator();
            loadFormData(); // Load saved form data
            
            // Add auto-save for all form fields
            const form = document.getElementById('registrationForm');
            if (form) {
                const inputs = form.querySelectorAll('input, select, textarea');
                inputs.forEach(input => {
                    input.addEventListener('input', saveFormData);
                    input.addEventListener('change', saveFormData);
                });
            }
            
            // Session keep-alive - ping server every 5 minutes
            setInterval(() => {
                fetch('/csrf-token')
                    .then(response => response.json())
                    .then(data => {
                        console.log('Session keep-alive ping successful');
                    })
                    .catch(error => {
                        console.log('Session keep-alive ping failed:', error);
                    });
            }, 5 * 60 * 1000); // 5 minutes
        });

        // Auto-save form data to localStorage with debouncing
        let saveTimeout;
        function saveFormData() {
            clearTimeout(saveTimeout);
            saveTimeout = setTimeout(() => {
                const formData = new FormData(document.getElementById('registrationForm'));
                const data = {};
                
                // Save all form fields
                for (let [key, value] of formData.entries()) {
                    data[key] = value;
                }
                
                // Save current step
                data.currentStep = currentStep;
                
                localStorage.setItem('registrasiPendudukData', JSON.stringify(data));
            }, 500); // Debounce for 500ms
        }

        // Load saved form data from localStorage
        function loadFormData() {
            const savedData = localStorage.getItem('registrasiPendudukData');
            if (savedData) {
                const data = JSON.parse(savedData);
                
                // Restore form fields
                Object.keys(data).forEach(key => {
                    if (key !== 'currentStep') {
                        const field = document.getElementById(key);
                        if (field) {
                            if (field.type === 'file') {
                                // Skip file fields
                                return;
                            }
                            field.value = data[key];
                        }
                    }
                });
                
                // Restore current step
                if (data.currentStep && data.currentStep > 1) {
                    // Go to the saved step
                    for (let i = 1; i < data.currentStep; i++) {
                        document.getElementById(`step${i}`).classList.remove('active');
                    }
                    currentStep = data.currentStep;
                    document.getElementById(`step${currentStep}`).classList.add('active');
                    updateStepIndicator();
                    
                    if (currentStep === 5) {
                        populateReviewData();
                    }
                }
                

            }
        }

        // Populate review data
        function populateReviewData() {
            const formData = new FormData(document.getElementById('registrationForm'));
            const reviewData = document.getElementById('reviewData');
            
            let html = '<div class="space-y-4">';
            
            // Personal Information
            html += '<div class="bg-gray-50 rounded-lg p-4">';
            html += '<h4 class="font-semibold text-gray-800 mb-3">Data Pribadi</h4>';
            html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">';
            html += `<div><strong>NIK:</strong> ${formData.get('nik') || '-'}</div>`;
            html += `<div><strong>Nama:</strong> ${formData.get('nama') || '-'}</div>`;
            html += `<div><strong>Tempat Lahir:</strong> ${formData.get('tempat_lahir') || '-'}</div>`;
            html += `<div><strong>Tanggal Lahir:</strong> ${formData.get('tanggal_lahir') || '-'}</div>`;
            html += `<div><strong>Jenis Kelamin:</strong> ${formData.get('jenis_kelamin') === 'L' ? 'Laki-laki' : 'Perempuan'}</div>`;
            html += `<div><strong>Agama:</strong> ${formData.get('agama') || '-'}</div>`;
            html += '</div></div>';
            
            // Address Information
            html += '<div class="bg-gray-50 rounded-lg p-4">';
            html += '<h4 class="font-semibold text-gray-800 mb-3">Data Alamat</h4>';
            html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">';
            html += `<div><strong>Alamat:</strong> ${formData.get('alamat') || '-'}</div>`;
            html += `<div><strong>Dusun:</strong> ${formData.get('dusun') || '-'}</div>`;
            html += `<div><strong>RT:</strong> ${formData.get('rt') || '-'}</div>`;
            html += `<div><strong>RW:</strong> ${formData.get('rw') || '-'}</div>`;
            html += '</div></div>';
            
            // Additional Information
            html += '<div class="bg-gray-50 rounded-lg p-4">';
            html += '<h4 class="font-semibold text-gray-800 mb-3">Data Tambahan</h4>';
            html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">';
            html += `<div><strong>Pekerjaan:</strong> ${formData.get('pekerjaan') || '-'}</div>`;
            html += `<div><strong>Pendidikan:</strong> ${formData.get('pendidikan_terakhir') || '-'}</div>`;
            html += `<div><strong>Golongan Darah:</strong> ${formData.get('golongan_darah') || '-'}</div>`;
            html += `<div><strong>No. WhatsApp:</strong> ${formData.get('no_wa') || '-'}</div>`;
            html += `<div><strong>Nama Ayah:</strong> ${formData.get('nama_ayah') || '-'}</div>`;
            html += `<div><strong>Nama Ibu:</strong> ${formData.get('nama_ibu') || '-'}</div>`;
            html += '</div></div>';
            
            // Foto dan Lokasi
            html += '<div class="bg-gray-50 rounded-lg p-4">';
            html += '<h4 class="font-semibold text-gray-800 mb-3">Foto dan Lokasi</h4>';
            html += '<div class="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">';
            html += `<div><strong>Foto Pribadi:</strong> ${formData.get('foto') ? '✓ Diupload' : 'Tidak diupload'}</div>`;
            html += `<div><strong>Foto Rumah:</strong> ${formData.get('foto_rumah') ? '✓ Diupload' : 'Tidak diupload'}</div>`;
            html += `<div><strong>Latitude:</strong> ${formData.get('latitude') || '-'}</div>`;
            html += `<div><strong>Longitude:</strong> ${formData.get('longitude') || '-'}</div>`;
            html += '</div></div>';
            
            html += '</div>';
            reviewData.innerHTML = html;
        }

        // Clear saved form data
        function clearFormData() {
            localStorage.removeItem('registrasiPendudukData');
            document.getElementById('registrationForm').reset();
            currentStep = 1;
            document.getElementById('step1').classList.add('active');
            for (let i = 2; i <= 5; i++) {
                document.getElementById(`step${i}`).classList.remove('active');
            }
            updateStepIndicator();
            showAlert('Data form telah dihapus', 'success');
        }


    </script>
</body>
</html> 