<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Penduduk;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class GenerateWargaPasswords extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'warga:generate-passwords 
                            {--dusun= : Filter by dusun}
                            {--rt= : Filter by RT}
                            {--rw= : Filter by RW}
                            {--limit=100 : Maximum number of passwords to generate}
                            {--force-change : Force password change on first login}
                            {--dry-run : Show what would be generated without actually doing it}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate default passwords for penduduk who don\'t have passwords yet';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔐 Portal Warga - Bulk Password Generator');
        $this->info('=====================================');
        
        $dryRun = $this->option('dry-run');
        $forceChange = $this->option('force-change');
        $limit = (int) $this->option('limit');
        
        if ($dryRun) {
            $this->warn('🧪 DRY RUN MODE - No passwords will be actually generated');
        }
        
        // Build query
        $query = Penduduk::whereNull('password');
        
        // Apply filters
        if ($dusun = $this->option('dusun')) {
            $query->where('dusun', $dusun);
            $this->info("📍 Filter: Dusun = {$dusun}");
        }
        
        if ($rt = $this->option('rt')) {
            $query->where('rt', $rt);
            $this->info("📍 Filter: RT = {$rt}");
        }
        
        if ($rw = $this->option('rw')) {
            $query->where('rw', $rw);
            $this->info("📍 Filter: RW = {$rw}");
        }
        
        $totalCount = $query->count();
        $this->info("📊 Total penduduk tanpa password: {$totalCount}");
        
        if ($totalCount === 0) {
            $this->warn('✅ Tidak ada penduduk yang perlu digenerate passwordnya!');
            return 0;
        }
        
        // Apply limit
        $pendudukList = $query->limit($limit)->get();
        $processCount = $pendudukList->count();
        
        $this->info("🎯 Akan memproses: {$processCount} penduduk");
        
        if ($limit < $totalCount) {
            $this->warn("⚠️  Hanya memproses {$limit} dari {$totalCount} penduduk (gunakan --limit untuk mengubah)");
        }
        
        if (!$dryRun && !$this->confirm('Lanjutkan generate password?')) {
            $this->info('❌ Dibatalkan oleh user');
            return 1;
        }
        
        $this->info('');
        $this->info('🚀 Memulai proses...');
        
        $successCount = 0;
        $errorCount = 0;
        $results = [];
        
        $progressBar = $this->output->createProgressBar($processCount);
        $progressBar->start();
        
        foreach ($pendudukList as $penduduk) {
            try {
                $password = substr($penduduk->nik, -6); // 6 digit terakhir NIK
                
                if (!$dryRun) {
                    $penduduk->update([
                        'password' => Hash::make($password),
                        'password_must_change' => $forceChange
                    ]);
                }
                
                $results[] = [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'password' => $password,
                    'status' => 'Success'
                ];
                
                $successCount++;
                
            } catch (\Exception $e) {
                $results[] = [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'password' => null,
                    'status' => 'Error: ' . $e->getMessage()
                ];
                
                $errorCount++;
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        $this->info('');
        $this->info('');
        
        // Show results
        $this->info('📋 HASIL GENERATE PASSWORD:');
        $this->info('==========================');
        
        if ($dryRun) {
            $this->warn('🧪 DRY RUN - Password tidak benar-benar digenerate');
        }
        
        $this->info("✅ Berhasil: {$successCount}");
        if ($errorCount > 0) {
            $this->error("❌ Gagal: {$errorCount}");
        }
        
        // Show detailed results
        if ($this->option('verbose') || $processCount <= 20) {
            $this->info('');
            $this->info('📄 Detail:');
            
            $headers = ['NIK', 'Nama', 'Password', 'Status'];
            $tableData = [];
            
            foreach ($results as $result) {
                $tableData[] = [
                    $result['nik'],
                    Str::limit($result['nama'], 25),
                    $result['password'] ? "****{$result['password']}" : '-',
                    $result['status']
                ];
            }
            
            $this->table($headers, $tableData);
        }
        
        // Export to file if requested
        if (!$dryRun && $this->confirm('Ekspor hasil ke file CSV?', false)) {
            $filename = storage_path('app/password_generation_' . date('Y-m-d_H-i-s') . '.csv');
            
            $file = fopen($filename, 'w');
            fputcsv($file, ['NIK', 'Nama', 'Password', 'Status', 'Generated At']);
            
            foreach ($results as $result) {
                fputcsv($file, [
                    $result['nik'],
                    $result['nama'],
                    $result['password'],
                    $result['status'],
                    now()->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
            $this->info("📁 File diekspor ke: {$filename}");
        }
        
        $this->info('');
        $this->info('🎉 Selesai!');
        
        return $errorCount > 0 ? 1 : 0;
    }
}
