<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProdukUmkm;
use App\Models\Umkm;
use App\Models\ObjekWisata;
use App\Models\TiketWisata;
use App\Models\Transaksi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MarketplaceController extends Controller
{
    public function index(Request $request)
    {
        // Statistics cards
        $stats = [
            'total_umkm' => Umkm::aktif()->count(),
            'total_produk' => ProdukUmkm::aktif()->count(),
            'total_objek_wisata' => ObjekWisata::aktif()->count(),
            'total_transaksi' => Transaksi::where('status', 'PAID')->count(),
            'total_pendapatan' => Transaksi::where('status', 'PAID')->sum('amount'),
        ];
        
        // Top Products (renamed from produkUnggulan)
        $topProducts = ProdukUmkm::with(['umkm'])
                                 ->orderBy('total_terjual', 'desc')
                                 ->limit(8)
                                 ->get();
        
        // Featured tourism objects
        $wisataUnggulan = ObjekWisata::featured()
                                    ->aktif()
                                    ->orderBy('rating_rata_rata', 'desc')
                                    ->limit(6)
                                    ->get();
        
        // UMKM Performance (renamed from umkmTerbaik)
        $umkmPerformance = Umkm::with(['produk'])
                               ->aktif()
                               ->verified()
                               ->orderBy('rating_rata_rata', 'desc')
                               ->limit(6)
                               ->get();
        
        // Recent Orders (renamed from transaksiTerbaru)
        $recentOrders = Transaksi::with(['transactable'])
                                ->where('status', 'PAID')
                                ->orderBy('paid_at', 'desc')
                                ->limit(10)
                                ->get();
        
        // Revenue Chart data
        $revenueChart = [
            'labels' => [],
            'data' => []
        ];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $revenue = Transaksi::where('status', 'PAID')
                               ->whereYear('paid_at', $date->year)
                               ->whereMonth('paid_at', $date->month)
                               ->sum('amount');
            
            $revenueChart['labels'][] = $date->format('M Y');
            $revenueChart['data'][] = $revenue;
        }
        
        // Category Chart data
        $categoryChart = [
            'labels' => [],
            'data' => []
        ];
        $categories = ProdukUmkm::selectRaw('kategori_produk, COUNT(*) as total')
                                ->where('status', 'aktif')
                                ->groupBy('kategori_produk')
                                ->orderBy('total', 'desc')
                                ->limit(5)
                                ->get();
        
        foreach ($categories as $category) {
            $categoryChart['labels'][] = $category->kategori_produk;
            $categoryChart['data'][] = $category->total;
        }
        
        return view('admin.marketplace.index', compact(
            'stats', 
            'topProducts', 
            'wisataUnggulan', 
            'umkmPerformance', 
            'recentOrders',
            'revenueChart',
            'categoryChart'
        ));
    }
    
    public function produk(Request $request)
    {
        $query = ProdukUmkm::with(['umkm']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_produk', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhereHas('umkm', function($qq) use ($search) {
                      $qq->where('nama_umkm', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_produk', $request->kategori);
        }
        
        // Filter by UMKM
        if ($request->filled('umkm_id')) {
            $query->where('umkm_id', $request->umkm_id);
        }
        
        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }
        
        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if (in_array($sortBy, ['nama_produk', 'harga', 'stok', 'rating_rata_rata', 'total_terjual', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $produk = $query->paginate(20)->withQueryString();
        
        // Filter options
        $umkmList = Umkm::aktif()->pluck('nama_umkm', 'id');
        $kategoriList = ProdukUmkm::getKategoriList();
        
        return view('admin.marketplace.produk', compact('produk', 'umkmList', 'kategoriList'));
    }
    
    public function wisata(Request $request)
    {
        $query = ObjekWisata::query();
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_wisata', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('alamat_lengkap', 'like', "%{$search}%");
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_wisata', $request->kategori);
        }
        
        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }
        
        $wisata = $query->paginate(20)->withQueryString();
        
        return view('admin.marketplace.wisata', compact('wisata'));
    }
    
    public function analytics(Request $request)
    {
        $periode = $request->get('periode', 'bulan');
        $tahun = $request->get('tahun', now()->year);
        $bulan = $request->get('bulan', now()->month);
        
        // Revenue analytics
        $revenueQuery = Transaksi::where('status', 'PAID');
        
        if ($periode === 'harian') {
            $revenueQuery->whereYear('paid_at', $tahun)
                        ->whereMonth('paid_at', $bulan);
        } elseif ($periode === 'bulanan') {
            $revenueQuery->whereYear('paid_at', $tahun);
        }
        
        $revenueData = $revenueQuery->get();
        
        $analytics = [
            'total_revenue' => $revenueData->sum('amount'),
            'total_transactions' => $revenueData->count(),
            'average_transaction' => $revenueData->count() > 0 ? $revenueData->sum('amount') / $revenueData->count() : 0,
            'total_fee' => $revenueData->sum('total_fee'),
            'net_revenue' => $revenueData->sum('amount') - $revenueData->sum('total_fee'),
        ];
        
        // Product performance
        $topProducts = ProdukUmkm::with(['umkm'])
                                 ->orderBy('total_terjual', 'desc')
                                 ->limit(10)
                                 ->get();
        
        // Tourism performance
        $topWisata = ObjekWisata::orderBy('total_pengunjung_keseluruhan', 'desc')
                               ->limit(10)
                               ->get();
        
        // UMKM performance
        $topUmkm = Umkm::with(['produk'])
                       ->withSum('produk', 'total_terjual')
                       ->orderBy('produk_sum_total_terjual', 'desc')
                       ->limit(10)
                       ->get();
        
        // Category breakdown
        $kategoriProduk = DB::table('produk_umkm')
                           ->select('kategori_produk', DB::raw('COUNT(*) as total_produk'), DB::raw('SUM(total_terjual) as total_terjual'))
                           ->where('status', 'aktif')
                           ->groupBy('kategori_produk')
                           ->orderBy('total_terjual', 'desc')
                           ->get();
        
        $kategoriWisata = DB::table('objek_wisata')
                           ->select('kategori_wisata', DB::raw('COUNT(*) as total_objek'), DB::raw('SUM(total_pengunjung_keseluruhan) as total_pengunjung'))
                           ->where('status', 'aktif')
                           ->groupBy('kategori_wisata')
                           ->orderBy('total_pengunjung', 'desc')
                           ->get();
        
        return view('admin.marketplace.analytics', compact(
            'analytics',
            'topProducts',
            'topWisata', 
            'topUmkm',
            'kategoriProduk',
            'kategoriWisata',
            'periode',
            'tahun',
            'bulan'
        ));
    }
    
    public function pengaturan()
    {
        // Get marketplace settings - using simple array since setting() helper doesn't exist
        $settings = [
            'commission_rate' => 5, // Default 5%
            'min_order_amount' => 10000, // Default Rp 10,000
            'max_order_amount' => 10000000, // Default Rp 10,000,000
            'auto_approve_umkm' => false,
            'auto_approve_produk' => false,
            'enable_reviews' => true,
            'enable_wishlist' => true,
            'enable_notifications' => true,
            'contact_email' => '',
            'contact_phone' => '',
            'social_media' => [],
        ];

        return view('admin.marketplace.pengaturan', compact('settings'));
    }
    
    public function updatePengaturan(Request $request)
    {
        $request->validate([
            'commission_rate' => 'required|numeric|min:0|max:100',
            'min_order_amount' => 'required|numeric|min:0',
            'max_order_amount' => 'required|numeric|min:0',
            'contact_email' => 'nullable|email',
            'contact_phone' => 'nullable|string|max:20',
        ]);

        // For now, just return success since we don't have a proper settings storage
        // TODO: Implement proper settings storage (create marketplace_settings table or use config)
        
        return redirect()->back()->with('success', 'Pengaturan marketplace berhasil diperbarui.');
    }
    
    public function exportLaporan(Request $request)
    {
        $type = $request->get('type', 'revenue');
        $periode = $request->get('periode', 'bulan');
        $tahun = $request->get('tahun', now()->year);
        $bulan = $request->get('bulan', now()->month);
        
        $filename = "marketplace_{$type}_{$periode}_{$tahun}" . ($periode === 'harian' ? "_{$bulan}" : '') . ".csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($type, $periode, $tahun, $bulan) {
            $file = fopen('php://output', 'w');
            
            if ($type === 'revenue') {
                $this->exportRevenueReport($file, $periode, $tahun, $bulan);
            } elseif ($type === 'products') {
                $this->exportProductsReport($file);
            } elseif ($type === 'tourism') {
                $this->exportTourismReport($file);
            } elseif ($type === 'umkm') {
                $this->exportUmkmReport($file);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    private function exportRevenueReport($file, $periode, $tahun, $bulan)
    {
        fputcsv($file, ['Tanggal', 'Total Transaksi', 'Total Revenue', 'Total Fee', 'Net Revenue']);
        
        $query = Transaksi::where('status', 'PAID');
        
        if ($periode === 'harian') {
            $query->whereYear('paid_at', $tahun)->whereMonth('paid_at', $bulan);
            $data = $query->selectRaw('DATE(paid_at) as date, COUNT(*) as total_transactions, SUM(amount) as total_amount, SUM(total_fee) as total_fee')
                          ->groupBy('date')
                          ->orderBy('date')
                          ->get();
        } else {
            $query->whereYear('paid_at', $tahun);
            $data = $query->selectRaw('YEAR(paid_at) as year, MONTH(paid_at) as month, COUNT(*) as total_transactions, SUM(amount) as total_amount, SUM(total_fee) as total_fee')
                          ->groupBy('year', 'month')
                          ->orderBy('year', 'month')
                          ->get();
        }
        
        foreach ($data as $row) {
            $date = $periode === 'harian' ? $row->date : "{$row->year}-{$row->month}";
            fputcsv($file, [
                $date,
                $row->total_transactions,
                $row->total_amount,
                $row->total_fee,
                $row->total_amount - $row->total_fee
            ]);
        }
    }
    
    private function exportProductsReport($file)
    {
        fputcsv($file, ['Nama Produk', 'UMKM', 'Kategori', 'Harga', 'Stok', 'Total Terjual', 'Rating', 'Status']);
        
        $products = ProdukUmkm::with(['umkm'])->get();
        
        foreach ($products as $product) {
            fputcsv($file, [
                $product->nama_produk,
                $product->umkm->nama_umkm,
                $product->kategori_produk,
                $product->harga,
                $product->stok,
                $product->total_terjual,
                $product->rating_rata_rata,
                $product->status
            ]);
        }
    }
    
    private function exportTourismReport($file)
    {
        fputcsv($file, ['Nama Objek Wisata', 'Kategori', 'Harga Tiket Dewasa', 'Harga Tiket Anak', 'Total Pengunjung', 'Rating', 'Status']);
        
        $tourism = ObjekWisata::all();
        
        foreach ($tourism as $objek) {
            fputcsv($file, [
                $objek->nama_wisata,
                $objek->kategori_wisata,
                $objek->harga_tiket_dewasa,
                $objek->harga_tiket_anak,
                $objek->total_pengunjung_keseluruhan,
                $objek->rating_rata_rata,
                $objek->status
            ]);
        }
    }
    
    private function exportUmkmReport($file)
    {
        fputcsv($file, ['Nama UMKM', 'Kategori', 'Total Produk', 'Total Terjual', 'Rating', 'Status', 'Verified']);
        
        $umkm = Umkm::withCount('produk')->get();
        
        foreach ($umkm as $u) {
            fputcsv($file, [
                $u->nama_umkm,
                $u->kategori_umkm,
                $u->produk_count,
                $u->produk->sum('total_terjual'),
                $u->rating_rata_rata,
                $u->status,
                $u->verified_at ? 'Ya' : 'Tidak'
            ]);
        }
    }
}

