<?php

namespace App\Http\Controllers;

use App\Models\ApbdesRencana;
use App\Models\ApbdesRealisasi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ApbdesController extends Controller
{
    // ===== RENCANA ANGGARAN =====
    
    public function indexRencana(Request $request)
    {
        $tahun = $request->input('tahun', date('Y'));
        $kategori = $request->input('kategori');

        $query = ApbdesRencana::with('user')->byTahun($tahun);

        if ($kategori) {
            $query->byKategori($kategori);
        }

        $rencana = $query->orderBy('kode_rekening')->paginate(20);

        $stats = [
            'total_pendapatan' => ApbdesRencana::getTotalByKategori($tahun, 'pendapatan'),
            'total_belanja' => ApbdesRencana::getTotalByKategori($tahun, 'belanja'),
            'total_pembiayaan' => ApbdesRencana::getTotalByKategori($tahun, 'pembiayaan'),
        ];

        return view('admin.apbdes.rencana.index', compact('rencana', 'stats', 'tahun', 'kategori'));
    }

    public function storeRencana(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tahun_anggaran' => 'required|integer|min:2020|max:2050',
            'kategori' => 'required|in:pendapatan,belanja,pembiayaan',
            'bidang' => 'nullable|string',
            'kode_rekening' => 'required|string|max:50|unique:apbdes_rencana,kode_rekening',
            'uraian' => 'required|string|max:255',
            'anggaran' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();
        $data['sisa_anggaran'] = $data['anggaran'];
        $data['status'] = 'aktif';

        $rencana = ApbdesRencana::create($data);

        return redirect()->route('admin.apbdes.rencana.index')
            ->with('success', 'Rencana anggaran berhasil ditambahkan');
    }

    public function showRencana($id)
    {
        $rencana = ApbdesRencana::with(['user', 'realisasi'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $rencana
        ]);
    }

    public function updateRencana(Request $request, $id)
    {
        $rencana = ApbdesRencana::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'tahun_anggaran' => 'required|integer|min:2020|max:2050',
            'kategori' => 'required|in:pendapatan,belanja,pembiayaan',
            'bidang' => 'nullable|in:pemerintahan,pembangunan,pembinaan,pemberdayaan,dana_desa,add,pad,surplus_defisit',
            'kode_rekening' => 'required|string|max:50|unique:apbdes_rencana,kode_rekening,' . $id,
            'uraian' => 'required|string|max:255',
            'keterangan' => 'nullable|string',
            'anggaran' => 'required|numeric|min:0',
            'status' => 'required|in:aktif,nonaktif'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();
        
        // Recalculate sisa anggaran if anggaran changed
        if ($data['anggaran'] != $rencana->anggaran) {
            $data['sisa_anggaran'] = $data['anggaran'] - $rencana->realisasi;
        }

        $rencana->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Rencana anggaran berhasil diperbarui',
            'data' => $rencana->load('user')
        ]);
    }

    public function destroyRencana($id)
    {
        $rencana = ApbdesRencana::findOrFail($id);

        // Check if has realisasi
        if ($rencana->realisasi()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat menghapus rencana anggaran yang sudah memiliki realisasi'
            ], 400);
        }

        $rencana->delete();

        return response()->json([
            'success' => true,
            'message' => 'Rencana anggaran berhasil dihapus'
        ]);
    }

    // ===== REALISASI ANGGARAN =====

    public function indexRealisasi(Request $request)
    {
        $tahun = $request->input('tahun', date('Y'));
        $kategori = $request->input('kategori');

        $query = ApbdesRealisasi::with(['rencana', 'user']);

        if ($tahun) {
            $query->whereHas('rencana', function($q) use ($tahun) {
                $q->where('tahun_anggaran', $tahun);
            });
        }

        if ($kategori) {
            $query->whereHas('rencana', function($q) use ($kategori) {
                $q->where('kategori', $kategori);
            });
        }

        if ($request->has('rencana_id')) {
            $query->where('apbdes_rencana_id', $request->rencana_id);
        }

        if ($request->has('status')) {
            $query->where('status_verifikasi', $request->status);
        }

        $realisasi = $query->orderBy('tanggal_transaksi', 'desc')->paginate(20);
        
        // Get list of rencana for dropdown
        $rencanaList = ApbdesRencana::where('status', 'aktif')
            ->orderBy('kode_rekening')
            ->get();

        return view('admin.apbdes.realisasi.index', compact('realisasi', 'rencanaList', 'tahun', 'kategori'));
    }

    public function storeRealisasi(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'apbdes_rencana_id' => 'required|exists:apbdes_rencana,id',
            'tanggal_transaksi' => 'required|date',
            'nomor_bukti' => 'required|string|max:100',
            'uraian_transaksi' => 'required|string',
            'jumlah' => 'required|numeric|min:0',
            'jenis' => 'required|in:pemasukan,pengeluaran',
            'sumber_dana' => 'nullable|string|max:255',
            'penerima_pembayaran' => 'nullable|string|max:255',
            'keterangan' => 'nullable|string',
            'bukti_transaksi' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();
        $data['status_verifikasi'] = 'pending';

        // Handle file upload
        if ($request->hasFile('bukti_transaksi')) {
            $file = $request->file('bukti_transaksi');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('public/apbdes/bukti', $filename);
            $data['bukti_transaksi'] = str_replace('public/', '', $path);
        }

        $realisasi = ApbdesRealisasi::create($data);

        return redirect()->route('admin.apbdes.realisasi.index')
            ->with('success', 'Realisasi anggaran berhasil ditambahkan');
    }

    public function showRealisasi($id)
    {
        $realisasi = ApbdesRealisasi::with(['rencana', 'user', 'verifier'])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $realisasi
        ]);
    }

    public function updateRealisasi(Request $request, $id)
    {
        $realisasi = ApbdesRealisasi::findOrFail($id);

        // Check if already verified
        if ($realisasi->status_verifikasi === 'verified') {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat mengubah realisasi yang sudah diverifikasi'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'tanggal_transaksi' => 'required|date',
            'nomor_bukti' => 'required|string|max:100|unique:apbdes_realisasi,nomor_bukti,' . $id,
            'uraian_transaksi' => 'required|string',
            'jumlah' => 'required|numeric|min:0',
            'jenis' => 'required|in:pemasukan,pengeluaran',
            'sumber_dana' => 'nullable|string|max:255',
            'penerima_pembayaran' => 'nullable|string|max:255',
            'keterangan' => 'nullable|string',
            'bukti_transaksi' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();

        // Handle file upload
        if ($request->hasFile('bukti_transaksi')) {
            // Delete old file
            if ($realisasi->bukti_transaksi) {
                Storage::delete('public/' . $realisasi->bukti_transaksi);
            }

            $file = $request->file('bukti_transaksi');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('public/apbdes/bukti', $filename);
            $data['bukti_transaksi'] = 'apbdes/bukti/' . $filename;
        }

        $realisasi->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Realisasi anggaran berhasil diperbarui',
            'data' => $realisasi->load(['rencana', 'user'])
        ]);
    }

    public function destroyRealisasi($id)
    {
        $realisasi = ApbdesRealisasi::findOrFail($id);

        // Check if verified
        if ($realisasi->status_verifikasi === 'verified') {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat menghapus realisasi yang sudah diverifikasi'
            ], 400);
        }

        // Delete file if exists
        if ($realisasi->bukti_transaksi) {
            Storage::delete('public/' . $realisasi->bukti_transaksi);
        }

        $realisasi->delete();

        return response()->json([
            'success' => true,
            'message' => 'Realisasi anggaran berhasil dihapus'
        ]);
    }

    public function verifyRealisasi(Request $request, $id)
    {
        $realisasi = ApbdesRealisasi::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:verified,rejected'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $status = $request->input('status');
        
        if ($status === 'verified') {
            $realisasi->verify();
            $message = 'Realisasi berhasil diverifikasi';
        } else {
            $realisasi->reject();
            $message = 'Realisasi berhasil ditolak';
        }

        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    public function rejectRealisasi($id)
    {
        $realisasi = ApbdesRealisasi::findOrFail($id);

        if ($realisasi->status_verifikasi === 'verified') {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat menolak realisasi yang sudah diverifikasi'
            ], 400);
        }

        $realisasi->reject();

        return response()->json([
            'success' => true,
            'message' => 'Realisasi berhasil ditolak',
            'data' => $realisasi->load(['rencana', 'user', 'verifier'])
        ]);
    }

    // ===== LAPORAN =====

    public function laporanAnggaran(Request $request)
    {
        $tahun = $request->input('tahun', date('Y'));
        $kategori = $request->input('kategori');
        $status = $request->input('status');

        // Get all rencana with realisasi data
        $query = ApbdesRencana::with(['realisasi' => function($q) {
            $q->where('status_verifikasi', 'verified');
        }])->byTahun($tahun)->aktif();

        if ($kategori) {
            $query->byKategori($kategori);
        }

        $laporan = $query->get()->map(function($item) {
            $totalRealisasi = $item->realisasi->sum('jumlah');
            return (object) [
                'id' => $item->id,
                'kode_rekening' => $item->kode_rekening,
                'uraian' => $item->uraian,
                'kategori' => $item->kategori,
                'anggaran' => $item->anggaran,
                'total_realisasi' => $totalRealisasi,
                'sisa_anggaran' => $item->anggaran - $totalRealisasi,
                'persentase' => $item->anggaran > 0 ? ($totalRealisasi / $item->anggaran) * 100 : 0,
                'status' => $item->status
            ];
        });

        // Calculate summary
        $totalAnggaran = $laporan->sum('anggaran');
        $totalRealisasi = $laporan->sum('total_realisasi');
        $persentase = $totalAnggaran > 0 ? ($totalRealisasi / $totalAnggaran) * 100 : 0;

        $summary = [
            'total_anggaran' => $totalAnggaran,
            'total_realisasi' => $totalRealisasi,
            'sisa_anggaran' => $totalAnggaran - $totalRealisasi,
            'persentase' => round($persentase, 2)
        ];

        // Chart data
        $chartData = [
            'pendapatan' => [
                'anggaran' => ApbdesRencana::getTotalByKategori($tahun, 'pendapatan'),
                'realisasi' => ApbdesRealisasi::getTotalPemasukan(null, $tahun . '-01-01', $tahun . '-12-31')
            ],
            'belanja' => [
                'anggaran' => ApbdesRencana::getTotalByKategori($tahun, 'belanja'),
                'realisasi' => ApbdesRealisasi::getTotalPengeluaran(null, $tahun . '-01-01', $tahun . '-12-31')
            ],
            'pembiayaan' => [
                'anggaran' => ApbdesRencana::getTotalByKategori($tahun, 'pembiayaan'),
                'realisasi' => 0 // Pembiayaan usually doesn't have realisasi
            ]
        ];

        return view('admin.apbdes.laporan', compact('laporan', 'summary', 'chartData', 'tahun', 'kategori', 'status'));
    }

    public function downloadBuktiTransaksi($id)
    {
        $realisasi = ApbdesRealisasi::findOrFail($id);

        if (!$realisasi->bukti_transaksi || !Storage::exists('public/' . $realisasi->bukti_transaksi)) {
            return response()->json([
                'success' => false,
                'message' => 'File bukti transaksi tidak ditemukan'
            ], 404);
        }

        return Storage::download('public/' . $realisasi->bukti_transaksi);
    }
} 