<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\OtpService;
use App\Services\DukcapilService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class OtpController extends Controller
{
    protected $otpService;
    protected $dukcapilService;

    public function __construct(OtpService $otpService, DukcapilService $dukcapilService)
    {
        $this->otpService = $otpService;
        $this->dukcapilService = $dukcapilService;
    }

    /**
     * Send OTP to phone number
     */
    public function sendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10|max:15',
            'type' => 'in:login,register,reset_password'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $request->phone;
        $type = $request->type ?? 'login';

        try {
            $result = $this->otpService->sendOtp($phone, $type);

            $statusCode = $result['success'] ? 200 : 400;

            return response()->json([
                'success' => $result['success'],
                'message' => $result['message'],
                'data' => $result['success'] ? [
                    'phone_masked' => $result['phone_masked'] ?? null,
                    'expires_in' => $result['expires_in'] ?? 300,
                    'type' => $type
                ] : null
            ], $statusCode);

        } catch (\Exception $e) {
            Log::error('Send OTP API Error', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Verify OTP and login
     */
    public function verifyOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10|max:15',
            'otp' => 'required|string|size:6',
            'type' => 'in:login,register,reset_password'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $request->phone;
        $otp = $request->otp;
        $type = $request->type ?? 'login';

        try {
            $result = $this->otpService->verifyOtp($phone, $otp, $type);

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'message' => $result['message'],
                    'data' => $result['data'] ?? null
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => $result['message'],
                    'attempts_left' => $result['attempts_left'] ?? null
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Verify OTP API Error', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Resend OTP
     */
    public function resendOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10|max:15',
            'type' => 'in:login,register,reset_password'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $request->phone;
        $type = $request->type ?? 'login';

        try {
            $result = $this->otpService->resendOtp($phone, $type);

            $statusCode = $result['success'] ? 200 : 429;

            return response()->json([
                'success' => $result['success'],
                'message' => $result['message'],
                'data' => $result['success'] ? [
                    'phone_masked' => $result['phone_masked'] ?? null,
                    'expires_in' => $result['expires_in'] ?? 300,
                    'type' => $type
                ] : null,
                'retry_after' => $result['retry_after'] ?? null
            ], $statusCode);

        } catch (\Exception $e) {
            Log::error('Resend OTP API Error', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Check OTP status
     */
    public function checkOtpStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10|max:15',
            'type' => 'in:login,register,reset_password'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $request->phone;
        $type = $request->type ?? 'login';

        try {
            $isValid = $this->otpService->isOtpValid($phone, $type);
            $attempts = $this->otpService->getOtpAttempts($phone, $type);

            return response()->json([
                'success' => true,
                'data' => [
                    'is_valid' => $isValid,
                    'attempts' => $attempts,
                    'max_attempts' => 3,
                    'phone_masked' => $this->maskPhoneNumber($phone)
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Check OTP Status API Error', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Verify NIK with Dukcapil
     */
    public function verifyNik(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nik' => 'required|string|size:16'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'NIK tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $nik = $request->nik;

        try {
            $result = $this->dukcapilService->verifyNik($nik);

            return response()->json([
                'success' => $result['success'],
                'message' => $result['message'],
                'data' => $result['data']
            ], $result['success'] ? 200 : 400);

        } catch (\Exception $e) {
            Log::error('NIK Verification API Error', [
                'nik' => $nik,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat verifikasi NIK'
            ], 500);
        }
    }

    /**
     * Check if phone number is registered
     */
    public function checkPhoneRegistration(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|min:10|max:15'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Nomor telefon tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        $phone = $request->phone;

        try {
            // Check in Penduduk table
            $penduduk = \App\Models\Penduduk::where('no_wa', 'like', '%' . substr($phone, -10))->first();

            if ($penduduk) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'is_registered' => true,
                        'nama' => $penduduk->nama,
                        'nik_masked' => $this->maskNik($penduduk->nik),
                        'can_login' => true
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'is_registered' => false,
                        'can_register' => true,
                        'message' => 'Nomor tidak terdaftar. Silakan daftar terlebih dahulu.'
                    ]
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Check Phone Registration API Error', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Get app configuration for mobile
     */
    public function getAppConfig()
    {
        try {
            return response()->json([
                'success' => true,
                'data' => [
                    'app_name' => config('app.name', 'Aplikasi Desa'),
                    'app_version' => '1.0.0',
                    'api_version' => 'v1',
                    'otp_length' => 6,
                    'otp_expiry_minutes' => 5,
                    'max_otp_attempts' => 3,
                    'features' => [
                        'dukcapil_verification' => true,
                        'whatsapp_otp' => true,
                        'offline_mode' => true,
                        'biometric_login' => false
                    ],
                    'contact' => [
                        'phone' => config('app.contact_phone'),
                        'email' => config('app.contact_email'),
                        'address' => config('app.office_address')
                    ],
                    'social_media' => [
                        'facebook' => config('app.facebook_url'),
                        'instagram' => config('app.instagram_url'),
                        'website' => config('app.website_url')
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Get App Config API Error', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Mask phone number for privacy
     */
    protected function maskPhoneNumber($phone)
    {
        if (strlen($phone) < 8) {
            return $phone;
        }

        return substr($phone, 0, 4) . str_repeat('*', strlen($phone) - 8) . substr($phone, -4);
    }

    /**
     * Mask NIK for privacy
     */
    protected function maskNik($nik)
    {
        if (strlen($nik) !== 16) {
            return $nik;
        }

        return substr($nik, 0, 6) . str_repeat('*', 6) . substr($nik, -4);
    }
} 