<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ObjekWisata;
use App\Models\TiketWisata;
use Illuminate\Http\Request;

class WisataController extends Controller
{
    public function index(Request $request)
    {
        $query = ObjekWisata::aktif();
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_wisata', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('alamat_lengkap', 'like', "%{$search}%");
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_wisata', $request->kategori);
        }
        
        // Filter by featured
        if ($request->filled('featured') && $request->featured == '1') {
            $query->featured();
        }
        
        // Filter by price range
        if ($request->filled('harga_min')) {
            $query->where('harga_tiket_dewasa', '>=', $request->harga_min);
        }
        
        if ($request->filled('harga_max')) {
            $query->where('harga_tiket_dewasa', '<=', $request->harga_max);
        }
        
        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if (in_array($sortBy, ['nama_wisata', 'harga_tiket_dewasa', 'rating_rata_rata', 'total_pengunjung_keseluruhan', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $wisata = $query->paginate(20);
        
        return response()->json([
            'success' => true,
            'data' => $wisata,
            'message' => 'Data objek wisata berhasil diambil'
        ]);
    }
    
    public function show($id)
    {
        $wisata = ObjekWisata::aktif()->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $wisata,
            'message' => 'Detail objek wisata berhasil diambil'
        ]);
    }
    
    public function featured()
    {
        $wisata = ObjekWisata::aktif()
                            ->featured()
                            ->orderBy('rating_rata_rata', 'desc')
                            ->limit(10)
                            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $wisata,
            'message' => 'Objek wisata unggulan berhasil diambil'
        ]);
    }
    
    public function popular()
    {
        $wisata = ObjekWisata::aktif()
                            ->orderBy('total_pengunjung_keseluruhan', 'desc')
                            ->limit(10)
                            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $wisata,
            'message' => 'Objek wisata populer berhasil diambil'
        ]);
    }
    
    public function categories()
    {
        $categories = [
            'alam' => 'Wisata Alam',
            'budaya' => 'Wisata Budaya',
            'sejarah' => 'Wisata Sejarah',
            'religi' => 'Wisata Religi',
            'kuliner' => 'Wisata Kuliner',
            'adventure' => 'Wisata Adventure',
            'edukasi' => 'Wisata Edukasi',
            'agrowisata' => 'Agrowisata',
            'pantai' => 'Wisata Pantai',
            'gunung' => 'Wisata Gunung',
            'air_terjun' => 'Air Terjun',
            'danau' => 'Wisata Danau',
            'goa' => 'Wisata Goa',
            'taman' => 'Taman Wisata',
            'museum' => 'Museum',
            'candi' => 'Candi',
            'lainnya' => 'Lainnya'
        ];
        
        return response()->json([
            'success' => true,
            'data' => $categories,
            'message' => 'Kategori objek wisata berhasil diambil'
        ]);
    }
    
    public function bookTicket(Request $request)
    {
        $request->validate([
            'objek_wisata_id' => 'required|exists:objek_wisata,id',
            'nama_pengunjung' => 'required|string|max:255',
            'email_pengunjung' => 'required|email',
            'telepon_pengunjung' => 'required|string|max:20',
            'tanggal_kunjungan' => 'required|date|after_or_equal:today',
            'jumlah_tiket' => 'required|integer|min:1|max:10',
            'jenis_tiket' => 'required|in:dewasa,anak',
        ]);
        
        $objekWisata = ObjekWisata::findOrFail($request->objek_wisata_id);
        
        // Check if tourism object is active
        if ($objekWisata->status !== 'aktif') {
            return response()->json([
                'success' => false,
                'message' => 'Objek wisata sedang tidak aktif'
            ], 422);
        }
        
        // Calculate total price
        $hargaPerTiket = $request->jenis_tiket === 'dewasa' ? 
                        $objekWisata->harga_tiket_dewasa : 
                        $objekWisata->harga_tiket_anak;
        
        $totalBayar = $hargaPerTiket * $request->jumlah_tiket;
        
        // Create ticket booking
        $tiket = TiketWisata::create([
            'objek_wisata_id' => $request->objek_wisata_id,
            'nama_pengunjung' => $request->nama_pengunjung,
            'email_pengunjung' => $request->email_pengunjung,
            'telepon_pengunjung' => $request->telepon_pengunjung,
            'tanggal_kunjungan' => $request->tanggal_kunjungan,
            'jumlah_tiket' => $request->jumlah_tiket,
            'jenis_tiket' => $request->jenis_tiket,
            'harga_per_tiket' => $hargaPerTiket,
            'total_bayar' => $totalBayar,
            'status_tiket' => 'pending_payment',
            'status_tiket' => 'aktif',
            'kode_booking' => TiketWisata::generateKodeBooking(),
            'qr_code' => TiketWisata::generateQRCode(),
        ]);
        
        return response()->json([
            'success' => true,
            'data' => $tiket->load('objekWisata'),
            'message' => 'Booking tiket berhasil dibuat'
        ]);
    }
    
    public function myTickets(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);
        
        $tickets = TiketWisata::with(['objekWisata'])
                             ->where('email_pengunjung', $request->email)
                             ->orderBy('created_at', 'desc')
                             ->paginate(20);
        
        return response()->json([
            'success' => true,
            'data' => $tickets,
            'message' => 'Tiket saya berhasil diambil'
        ]);
    }
    
    public function ticketDetail($kodeBooking)
    {
        $ticket = TiketWisata::with(['objekWisata'])
                            ->where('kode_booking', $kodeBooking)
                            ->firstOrFail();
        
        return response()->json([
            'success' => true,
            'data' => $ticket,
            'message' => 'Detail tiket berhasil diambil'
        ]);
    }
} 
