<?php

namespace App\Http\Controllers;

use App\Models\Penduduk;
use App\Models\PelayananSurat;
use App\Models\Pengaduan;
use App\Models\Bantuan;
use App\Models\Berita;
use App\Models\NotifikasiWa;
use App\Models\TemplateSurat;
use App\Models\Pbb;
use App\Models\TagihanPbb;
use App\Models\Polling;
use App\Models\PerangkatDesa;
use App\Models\AgendaDesa;
use App\Models\InventarisDesa;
use App\Models\ArsipSuratKeluar;
use App\Models\MutasiPenduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function __construct()
    {
        // Constructor without WhatsApp service dependency
    }

    public function index()
    {
        // Enhanced statistics including mutasi penduduk
        $stats = [
            'total_penduduk' => Penduduk::hidup()->count(),
            'penduduk_meninggal' => Penduduk::meninggal()->count(),
            'total_keluarga' => Penduduk::distinct('no_kk')->count(),
            'total_surat' => PelayananSurat::count(),
            'total_pengaduan' => Pengaduan::count(),
            'surat_pending' => PelayananSurat::where('status', 'Menunggu')->count(),
            'pengaduan_baru' => Pengaduan::where('status', 'Baru')->count(),
            'bantuan_aktif' => Bantuan::where('status', 'Diterima')->count(),
            'total_bantuan' => Bantuan::count(),
            'berita_bulan_ini' => Berita::whereMonth('created_at', now()->month)->count(),
            'laki_laki' => Penduduk::hidup()->where('jenis_kelamin', 'L')->count(),
            'perempuan' => Penduduk::hidup()->where('jenis_kelamin', 'P')->count(),
            
            // Mutasi Penduduk Statistics
            'lahir_bulan_ini' => MutasiPenduduk::where('jenis_mutasi', 'Lahir')->whereMonth('tanggal_mutasi', now()->month)->count(),
            'meninggal_bulan_ini' => MutasiPenduduk::where('jenis_mutasi', 'Meninggal')->whereMonth('tanggal_mutasi', now()->month)->count(),
            'pindah_masuk_bulan_ini' => MutasiPenduduk::where('jenis_mutasi', 'Pindah Masuk')->whereMonth('tanggal_mutasi', now()->month)->count(),
            'pindah_keluar_bulan_ini' => MutasiPenduduk::where('jenis_mutasi', 'Pindah Keluar')->whereMonth('tanggal_mutasi', now()->month)->count(),
            'total_mutasi_tahun_ini' => MutasiPenduduk::whereYear('tanggal_mutasi', now()->year)->count(),
            
            // Enhanced Module Statistics
            'total_template_surat' => TemplateSurat::where('aktif', true)->count(),
            'total_pbb' => Pbb::where('status_objek', 'Aktif')->count(),
            'pbb_belum_bayar' => TagihanPbb::where('status_bayar', 'belum_bayar')
                                         ->whereYear('tahun_pajak', now()->year)
                                         ->count(),
            'total_polling' => Polling::whereIn('status', ['Draft', 'Aktif', 'Selesai'])->count(),
            'polling_aktif' => Polling::where('status', 'Aktif')
                                    ->where('mulai_voting', '<=', now())
                                    ->where('selesai_voting', '>=', now())
                                    ->count(),
            'total_perangkat' => PerangkatDesa::where('status', 'aktif')->count(),
            'agenda_bulan_ini' => AgendaDesa::whereMonth('tanggal_mulai', now()->month)
                                          ->whereYear('tanggal_mulai', now()->year)
                                          ->count(),
            'agenda_minggu_ini' => AgendaDesa::whereBetween('tanggal_mulai', [
                                             now()->startOfWeek(),
                                             now()->endOfWeek()
                                           ])->count(),
            'total_inventaris' => InventarisDesa::count(),
            'inventaris_baik' => InventarisDesa::where('kondisi', 'Baik')->count(),
            'inventaris_perlu_maintenance' => InventarisDesa::where('kondisi', '!=', 'Baik')->count(),
            'total_nilai_inventaris' => InventarisDesa::sum('nilai_sekarang'),
            'total_arsip_surat' => ArsipSuratKeluar::count(),
            'arsip_bulan_ini' => ArsipSuratKeluar::whereMonth('tanggal_surat', now()->month)
                                               ->whereYear('tanggal_surat', now()->year)
                                               ->count(),
            
            // Chart Data
            'monthly_services' => $this->getMonthlyServices(),
            'map_penduduk' => $this->getMapPenduduk(),
            'map_pengaduan' => $this->getMapPengaduan(),
            'map_keluarga' => $this->getMapKeluarga(),
            'statistik_pendidikan' => $this->getStatistikPendidikan(),
            'statistik_pekerjaan' => $this->getStatistikPekerjaan(),
            'statistik_usia' => $this->getStatistikUsia(),
            'statistik_mutasi' => $this->getStatistikMutasi(),
            'statistik_agama' => $this->getStatistikAgama(),
            'statistik_kawin' => $this->getStatistikPerkawinan(),
            
            // Enhanced Statistics
            'statistik_surat_jenis' => $this->getStatistikJenisSurat(),
            'statistik_pbb_status' => $this->getStatistikPbbStatus(),
            'agenda_upcoming' => $this->getAgendaUpcoming(),
            'inventaris_per_kategori' => $this->getInventarisPerKategori()
        ];

        // Data terbaru untuk aktivitas
        $recentData = [
            'surat_terbaru' => PelayananSurat::with('penduduk')
                                           ->orderBy('created_at', 'desc')
                                           ->limit(5)
                                           ->get(),
            'pengaduan_terbaru' => Pengaduan::with('penduduk')
                                          ->orderBy('created_at', 'desc')
                                          ->limit(5)
                                          ->get(),
            'berita_terbaru' => Berita::orderBy('created_at', 'desc')
                                    ->limit(5)
                                    ->get()
        ];

        return view('admin.dashboard', compact('stats', 'recentData'));
    }

    /**
     * Get dashboard stats for AJAX refresh
     */
    public function stats()
    {
        $stats = [
            'total_penduduk' => Penduduk::count(),
            'total_keluarga' => Penduduk::distinct('no_kk')->count(),
            'total_surat' => PelayananSurat::count(),
            'total_pengaduan' => Pengaduan::count(),
            'surat_pending' => PelayananSurat::where('status', 'Menunggu')->count(),
            'pengaduan_baru' => Pengaduan::where('status', 'Baru')->count(),
            'bantuan_aktif' => Bantuan::where('status', 'Diterima')->count(),
            'recent_activities' => $this->getRecentActivities()
        ];

        return response()->json($stats);
    }

    /**
     * Get monthly services data for chart
     */
    private function getMonthlyServices()
    {
        $monthlyData = [];
        $monthlyLabels = [];
        
        // Generate data for last 12 months
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthlyLabels[] = $date->format('M Y'); // Format: "Jan 2024"
            
            $count = PelayananSurat::whereMonth('created_at', $date->month)
                                 ->whereYear('created_at', $date->year)
                                 ->count();
            
            $monthlyData[] = $count;
        }
        
        return [
            'data' => $monthlyData,
            'labels' => $monthlyLabels
        ];
    }

    /**
     * Get map data for penduduk
     */
    private function getMapPenduduk()
    {
        return Penduduk::whereNotNull('latitude')
                      ->whereNotNull('longitude')
                      ->select('nik', 'nama', 'alamat', 'latitude', 'longitude')
                      ->limit(100)
                      ->get();
    }

    /**
     * Get map data for pengaduan
     */
    private function getMapPengaduan()
    {
        return Pengaduan::with('penduduk:nik,nama,latitude,longitude')
                       ->whereHas('penduduk', function($q) {
                           $q->whereNotNull('latitude')->whereNotNull('longitude');
                       })
                       ->select('id', 'nik', 'judul', 'status', 'created_at')
                       ->limit(50)
                       ->get()
                       ->map(function($item) {
                           return [
                               'id' => $item->id,
                               'judul' => $item->judul,
                               'status' => $item->status,
                               'status_color' => $this->getStatusColor($item->status),
                               'tanggal' => $item->created_at->format('d/m/Y'),
                               'latitude' => $item->penduduk->latitude,
                               'longitude' => $item->penduduk->longitude
                           ];
                       });
    }

    /**
     * Get map data for keluarga
     */
    private function getMapKeluarga()
    {
        // Get one representative from each family (first member with coordinates)
        $keluargaData = Penduduk::whereNotNull('latitude')
                               ->whereNotNull('longitude')
                               ->select('no_kk', 'nama', 'latitude', 'longitude')
                               ->orderBy('created_at')
                               ->get()
                               ->groupBy('no_kk')
                               ->map(function($group) {
                                   $representative = $group->first(); // Get the first member with coordinates
                                   $anggota = Penduduk::where('no_kk', $representative->no_kk)->count();
                                   
                                   return [
                                       'no_kk' => $representative->no_kk,
                                       'kepala_keluarga' => $representative->nama,
                                       'jumlah_anggota' => $anggota,
                                       'latitude' => $representative->latitude,
                                       'longitude' => $representative->longitude
                                   ];
                               })
                               ->take(100);

        return $keluargaData->values(); // Reset array keys
    }

    /**
     * Get education statistics
     */
    private function getStatistikPendidikan()
    {
        return Penduduk::hidup()
                      ->whereNotNull('pendidikan_terakhir')
                      ->select('pendidikan_terakhir as pendidikan', DB::raw('count(*) as total'))
                      ->groupBy('pendidikan_terakhir')
                      ->orderByDesc('total')
                      ->get()
                      ->map(function($item) {
                          return [
                              'pendidikan' => $item->pendidikan ?: 'Tidak Diketahui',
                              'total' => $item->total,
                              'persentase' => 0 // Will be calculated in frontend
                          ];
                      });
    }

    /**
     * Get occupation statistics
     */
    private function getStatistikPekerjaan()
    {
        return Penduduk::hidup()
                      ->whereNotNull('pekerjaan')
                      ->select('pekerjaan', DB::raw('count(*) as total'))
                      ->groupBy('pekerjaan')
                      ->orderByDesc('total')
                      ->limit(10)
                      ->get()
                      ->map(function($item) {
                          return [
                              'pekerjaan' => $item->pekerjaan,
                              'total' => $item->total,
                              'persentase' => 0 // Will be calculated in frontend
                          ];
                      });
    }

    /**
     * Get age group statistics
     */
    private function getStatistikUsia()
    {
        $today = now();
        
        return [
            'balita' => Penduduk::hidup()->whereBetween(DB::raw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())'), [0, 4])->count(),
            'anak' => Penduduk::hidup()->whereBetween(DB::raw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())'), [5, 12])->count(),
            'remaja' => Penduduk::hidup()->whereBetween(DB::raw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())'), [13, 17])->count(),
            'dewasa' => Penduduk::hidup()->whereBetween(DB::raw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())'), [18, 59])->count(),
            'lansia' => Penduduk::hidup()->where(DB::raw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE())'), '>=', 60)->count()
        ];
    }

    /**
     * Get recent activities for dashboard
     */
    private function getRecentActivities()
    {
        $activities = collect();

        // Recent surat
        $recentSurat = PelayananSurat::with('penduduk')
                                    ->orderBy('created_at', 'desc')
                                    ->limit(3)
                                    ->get()
                                    ->map(function($item) {
                                        return [
                                            'description' => "Permohonan {$item->jenis_surat}",
                                            'user' => $item->penduduk->nama ?? 'N/A',
                                            'status' => $item->status,
                                            'status_color' => $this->getStatusColor($item->status),
                                            'created_at' => $item->created_at->format('d/m/Y H:i')
                                        ];
                                    });

        // Recent pengaduan
        $recentPengaduan = Pengaduan::with('penduduk')
                                   ->orderBy('created_at', 'desc')
                                   ->limit(2)
                                   ->get()
                                   ->map(function($item) {
                                       return [
                                           'description' => "Pengaduan: {$item->judul}",
                                           'user' => $item->penduduk->nama ?? 'N/A',
                                           'status' => $item->status,
                                           'status_color' => $this->getStatusColor($item->status),
                                           'created_at' => $item->created_at->format('d/m/Y H:i')
                                       ];
                                   });

        return $activities->merge($recentSurat)->merge($recentPengaduan)->take(5);
    }

    /**
     * Get status color for badges
     */
    private function getStatusColor($status)
    {
        return match(strtolower($status)) {
            'menunggu', 'baru' => 'warning',
            'diproses' => 'info',
            'selesai', 'diterima' => 'success',
            'ditolak' => 'danger',
            default => 'secondary'
        };
    }

    public function laporanPenduduk(Request $request)
    {
        $query = Penduduk::query();

        // Filter berdasarkan parameter
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }

        if ($request->jenis_kelamin) {
            $query->where('jenis_kelamin', $request->jenis_kelamin);
        }

        if ($request->umur_min && $request->umur_max) {
            $query->whereRaw('YEAR(CURDATE()) - YEAR(tanggal_lahir) BETWEEN ? AND ?', 
                            [$request->umur_min, $request->umur_max]);
        }

        $penduduk = $query->with(['keluarga', 'anggotaKeluarga'])->get();

        // Statistik
        $statistik = [
            'total' => $penduduk->count(),
            'laki_laki' => $penduduk->where('jenis_kelamin', 'L')->count(),
            'perempuan' => $penduduk->where('jenis_kelamin', 'P')->count(),
            'rata_rata_umur' => $penduduk->avg('umur'),
            'per_dusun' => $penduduk->groupBy('dusun')->map->count(),
            'per_agama' => $penduduk->groupBy('agama')->map->count(),
            'per_pekerjaan' => $penduduk->groupBy('pekerjaan')->map->count()
        ];

        return view('admin.laporan.penduduk', compact('penduduk', 'statistik'));
    }

    public function laporanPelayanan(Request $request)
    {
        $query = PelayananSurat::with('penduduk');

        // Filter berdasarkan tanggal
        if ($request->tanggal_dari && $request->tanggal_sampai) {
            $query->whereBetween('tanggal_pengajuan', [$request->tanggal_dari, $request->tanggal_sampai]);
        }

        // Filter berdasarkan status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan jenis surat
        if ($request->jenis_surat) {
            $query->where('jenis_surat', $request->jenis_surat);
        }

        $pelayananSurat = $query->orderBy('tanggal_pengajuan', 'desc')->get();

        // Statistik
        $statistik = [
            'total' => $pelayananSurat->count(),
            'selesai' => $pelayananSurat->where('status', 'Selesai')->count(),
            'diproses' => $pelayananSurat->where('status', 'Diproses')->count(),
            'menunggu' => $pelayananSurat->where('status', 'Menunggu')->count(),
            'ditolak' => $pelayananSurat->where('status', 'Ditolak')->count(),
            'per_jenis' => $pelayananSurat->groupBy('jenis_surat')->map->count(),
            'rata_rata_proses' => $pelayananSurat->where('status', 'Selesai')
                                               ->filter(function($item) {
                                                   return $item->tanggal_selesai && $item->tanggal_pengajuan;
                                               })
                                               ->map(function($item) {
                                                   return $item->tanggal_pengajuan->diffInDays($item->tanggal_selesai);
                                               })
                                               ->avg()
        ];

        return view('admin.laporan.pelayanan', compact('pelayananSurat', 'statistik'));
    }

    public function laporanBantuan(Request $request)
    {
        $query = Bantuan::with('penduduk');

        // Filter berdasarkan periode
        if ($request->periode) {
            $query->where('periode', $request->periode);
        }

        // Filter berdasarkan jenis bantuan
        if ($request->jenis_bantuan) {
            $query->where('jenis_bantuan', $request->jenis_bantuan);
        }

        // Filter berdasarkan status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        $bantuan = $query->orderBy('created_at', 'desc')->get();

        // Statistik
        $statistik = [
            'total' => $bantuan->count(),
            'diterima' => $bantuan->where('status', 'Diterima')->count(),
            'diverifikasi' => $bantuan->where('status', 'Diverifikasi')->count(),
            'diajukan' => $bantuan->where('status', 'Diajukan')->count(),
            'ditolak' => $bantuan->where('status', 'Ditolak')->count(),
            'total_nominal' => $bantuan->where('status', 'Diterima')->sum('nominal'),
            'per_jenis' => $bantuan->groupBy('jenis_bantuan')->map->count(),
            'per_periode' => $bantuan->groupBy('periode')->map->count()
        ];

        return view('admin.laporan.bantuan', compact('bantuan', 'statistik'));
    }

    /**
     * WhatsApp Notification Center
     */
    public function notifikasiWa()
    {
        // Get notification statistics (using existing table structure)
        $stats = [
            'total_notifications' => NotifikasiWa::count(),
            'unread_notifications' => 0, // Will be implemented after migration
            'pending_messages' => NotifikasiWa::where('status', 'Pending')->count(),
            'sent_messages' => NotifikasiWa::where('status', 'Terkirim')->count(),
            'failed_messages' => NotifikasiWa::where('status', 'Gagal')->count(),
            'today_messages' => NotifikasiWa::whereDate('created_at', today())->count(),
        ];

        // Get recent notifications (using existing table structure)
        $recentNotifications = NotifikasiWa::orderBy('created_at', 'desc')
                                          ->limit(10)
                                          ->get()
                                          ->map(function ($notification) {
                                              // Map legacy fields to new structure for compatibility
                                              $notification->phone_number = $notification->no_wa;
                                              $notification->message = $notification->pesan;
                                              $notification->message_type = $notification->jenis;
                                              $notification->is_read = true; // Default to read for now
                                              return $notification;
                                          });

        // Get message templates
        $templates = [
            'surat_approved' => 'Surat {jenis_surat} Anda telah disetujui. Silakan ambil di kantor desa pada jam kerja.',
            'surat_rejected' => 'Surat {jenis_surat} Anda ditolak. Alasan: {alasan}. Hubungi kantor desa untuk info lebih lanjut.',
            'pengaduan_response' => 'Pengaduan Anda "{judul}" telah ditanggapi: {tanggapan}',
            'bantuan_approved' => 'Bantuan {jenis_bantuan} Anda telah disetujui. Silakan ambil sesuai jadwal yang ditentukan.',
            'berita_broadcast' => 'BERITA DESA: {judul_berita}. {isi_singkat}. Info lengkap: {link}',
            'reminder_general' => 'PENGINGAT: {pesan_reminder}. Terima kasih.',
        ];

        return view('admin.notifikasi-wa', compact('stats', 'recentNotifications', 'templates'));
    }

    /**
     * API: Get WhatsApp notifications for AJAX
     */
    public function apiGetNotifications(Request $request)
    {
        $type = $request->get('type', 'recent');
        $query = NotifikasiWa::with('penduduk');

        switch ($type) {
            case 'recent':
                $query->where('created_at', '>=', now()->subDays(7))
                      ->orderBy('created_at', 'desc');
                break;
            case 'sent':
                $query->where('status', 'sent')
                      ->orderBy('created_at', 'desc');
                break;
            case 'failed':
                $query->where('status', 'failed')
                      ->orderBy('created_at', 'desc');
                break;
            case 'pending':
                $query->where('status', 'pending')
                      ->orderBy('created_at', 'desc');
                break;
        }

        $notifications = $query->limit(50)->get()->map(function ($notification) {
            return [
                'id' => $notification->id,
                'type' => $this->getNotificationType($notification->message_type),
                'title' => $this->getNotificationTitle($notification),
                'message' => $notification->message,
                'recipient' => $notification->phone_number,
                'status' => $notification->status,
                'is_read' => $notification->is_read,
                'created_at' => $notification->created_at->toISOString(),
                'penduduk_nama' => $notification->penduduk->nama ?? 'System',
            ];
        });

        // Get counts (using existing table structure)
        $counts = [
            'unread' => 0, // Will be implemented after migration
            'whatsapp' => NotifikasiWa::where('status', 'Pending')->count(),
            'total' => NotifikasiWa::count(),
            'sent' => NotifikasiWa::where('status', 'Terkirim')->count(),
            'failed' => NotifikasiWa::where('status', 'Gagal')->count(),
        ];

        return response()->json([
            'success' => true,
            'data' => $notifications,
            'counts' => $counts,
        ]);
    }

    /**
     * API: Check for new notifications
     */
    public function apiCheckNewNotifications(Request $request)
    {
        $lastCheck = $request->get('last_check', now()->subMinutes(5));
        
        $newNotifications = NotifikasiWa::where('created_at', '>', $lastCheck)
                                       ->count();

        $latestNotification = NotifikasiWa::with('penduduk')
                                         ->orderBy('created_at', 'desc')
                                         ->first();

        return response()->json([
            'success' => true,
            'new_count' => $newNotifications,
            'latest' => $latestNotification ? [
                'title' => $this->getNotificationTitle($latestNotification),
                'message' => $latestNotification->message,
                'created_at' => $latestNotification->created_at->toISOString(),
            ] : null,
        ]);
    }

    /**
     * API: Mark notification as read
     */
    public function apiMarkAsRead(Request $request, $id)
    {
        $notification = NotifikasiWa::findOrFail($id);
        // Temporary implementation - will be updated after migration
        // $notification->update(['is_read' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Notifikasi ditandai sudah dibaca (fitur akan aktif setelah update database)',
        ]);
    }

    /**
     * API: Resend WhatsApp message
     */
    public function apiResendMessage(Request $request, $id)
    {
        $notification = NotifikasiWa::findOrFail($id);
        
        try {
            // Here you would integrate with your WhatsApp service
            // For now, we'll just update the status
            $notification->update([
                'status' => 'sent',
                'sent_at' => now(),
                'response_data' => json_encode(['resent' => true, 'timestamp' => now()])
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pesan berhasil dikirim ulang',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim ulang pesan: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * API: Send broadcast message
     */
    public function apiBroadcastMessage(Request $request, $id)
    {
        $notification = NotifikasiWa::findOrFail($id);
        
        try {
            // Get all residents with phone numbers
            $recipients = Penduduk::whereNotNull('no_telepon')
                                 ->where('no_telepon', '!=', '')
                                 ->get();

            $sentCount = 0;
            foreach ($recipients as $recipient) {
                // Create individual notification for each recipient
                NotifikasiWa::create([
                    'nik' => $recipient->nik,
                    'phone_number' => $recipient->no_telepon,
                    'message' => $notification->message,
                    'message_type' => 'broadcast',
                    'status' => 'sent', // In real implementation, this would be 'pending'
                    'sent_at' => now(),
                ]);
                $sentCount++;
            }

            return response()->json([
                'success' => true,
                'message' => 'Broadcast berhasil dikirim',
                'sent_count' => $sentCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim broadcast: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * API: Send custom WhatsApp message
     */
    public function apiSendCustomMessage(Request $request)
    {
        $request->validate([
            'recipients' => 'required|array',
            'message' => 'required|string|max:1000',
            'message_type' => 'required|string',
        ]);

        try {
            $sentCount = 0;
            foreach ($request->recipients as $phone) {
                NotifikasiWa::create([
                    'phone_number' => $phone,
                    'message' => $request->message,
                    'message_type' => $request->message_type,
                    'status' => 'sent', // In real implementation, send via WhatsApp API
                    'sent_at' => now(),
                ]);
                $sentCount++;
            }

            return response()->json([
                'success' => true,
                'message' => 'Pesan berhasil dikirim',
                'sent_count' => $sentCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim pesan: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Helper: Get notification type for UI
     */
    private function getNotificationType($messageType)
    {
        return match($messageType) {
            'surat_approved', 'surat_rejected' => 'whatsapp',
            'pengaduan_response' => 'success',
            'bantuan_approved' => 'info',
            'berita_broadcast' => 'warning',
            'broadcast' => 'info',
            default => 'whatsapp'
        };
    }

    /**
     * Helper: Get notification title
     */
    private function getNotificationTitle($notification)
    {
        return match($notification->message_type) {
            'surat_approved' => 'Surat Disetujui',
            'surat_rejected' => 'Surat Ditolak',
            'pengaduan_response' => 'Pengaduan Ditanggapi',
            'bantuan_approved' => 'Bantuan Disetujui',
            'berita_broadcast' => 'Berita Desa',
            'broadcast' => 'Broadcast Message',
            'reminder' => 'Pengingat',
            default => 'Notifikasi WhatsApp'
        };
    }

    /**
     * Get statistics for letter types
     */
    private function getStatistikJenisSurat()
    {
        return PelayananSurat::select('jenis_surat', DB::raw('count(*) as total'))
                            ->groupBy('jenis_surat')
                            ->orderByDesc('total')
                            ->limit(10)
                            ->get();
    }

    /**
     * Get PBB payment status statistics
     */
    private function getStatistikPbbStatus()
    {
        return TagihanPbb::select('status_bayar', DB::raw('count(*) as total'))
                        ->whereYear('tahun_pajak', now()->year)
                        ->groupBy('status_bayar')
                        ->get();
    }

    /**
     * Get upcoming agenda items
     */
    private function getAgendaUpcoming()
    {
        return AgendaDesa::where('tanggal_mulai', '>=', now())
                        ->where('status', 'dijadwalkan')
                        ->orderBy('tanggal_mulai')
                        ->limit(5)
                        ->get();
    }

    /**
     * Get inventory statistics by category
     */
    private function getInventarisPerKategori()
    {
        return InventarisDesa::select('kategori', 
                                    DB::raw('count(*) as total'),
                                    DB::raw('sum(nilai_sekarang) as total_nilai'))
                             ->groupBy('kategori')
                             ->orderByDesc('total_nilai')
                             ->get();
    }

    /**
     * API method for enhanced statistics
     */
    public function apiStatistics()
    {
        $stats = [
            'penduduk' => [
                'total' => Penduduk::hidup()->count(),
                'laki_laki' => Penduduk::hidup()->where('jenis_kelamin', 'L')->count(),
                'perempuan' => Penduduk::hidup()->where('jenis_kelamin', 'P')->count(),
                'mutasi_bulan_ini' => [
                    'lahir' => Penduduk::where('status_mutasi', 'Lahir')
                                     ->whereMonth('tanggal_mutasi', now()->month)
                                     ->count(),
                    'meninggal' => Penduduk::where('status_mutasi', 'Meninggal')
                                         ->whereMonth('tanggal_mutasi', now()->month)
                                         ->count(),
                    'pindah_masuk' => Penduduk::pindahMasuk()
                                            ->whereMonth('tanggal_mutasi', now()->month)
                                            ->count(),
                    'pindah_keluar' => Penduduk::pindahKeluar()
                                             ->whereMonth('tanggal_mutasi', now()->month)
                                             ->count()
                ]
            ],
            'surat' => [
                'total' => PelayananSurat::count(),
                'pending' => PelayananSurat::where('status', 'Menunggu')->count(),
                'selesai' => PelayananSurat::where('status', 'Selesai')->count(),
                'per_jenis' => $this->getStatistikJenisSurat()
            ],
            'pbb' => [
                'total' => Pbb::where('status_objek', 'Aktif')->count(),
                'tagihan_tahun_ini' => TagihanPbb::whereYear('tahun_pajak', now()->year)->count(),
                'belum_bayar' => TagihanPbb::where('status_bayar', 'belum_bayar')
                                          ->whereYear('tahun_pajak', now()->year)
                                          ->count(),
                'total_tagihan' => TagihanPbb::where('status_bayar', 'belum_bayar')
                                            ->whereYear('tahun_pajak', now()->year)
                                            ->sum('jumlah_pajak')
            ],
            'inventaris' => [
                'total' => InventarisDesa::count(),
                'total_nilai' => InventarisDesa::sum('nilai_sekarang'),
                'baik' => InventarisDesa::where('kondisi', 'Baik')->count(),
                'rusak' => InventarisDesa::where('kondisi', '!=', 'Baik')->count(),
                'per_kategori' => $this->getInventarisPerKategori()
            ],
            'agenda' => [
                'total_aktif' => AgendaDesa::where('status', 'dijadwalkan')->count(),
                'bulan_ini' => AgendaDesa::whereMonth('tanggal_mulai', now()->month)
                                       ->whereYear('tanggal_mulai', now()->year)
                                       ->count(),
                'upcoming' => $this->getAgendaUpcoming()
            ]
        ];

        return response()->json($stats);
    }

    /**
     * Get mutation statistics for dashboard
     */
    private function getStatistikMutasi()
    {
        $currentYear = now()->year;
        
        return [
            'total_tahun_ini' => MutasiPenduduk::whereYear('tanggal_mutasi', $currentYear)->count(),
            'per_jenis' => MutasiPenduduk::select('jenis_mutasi', DB::raw('count(*) as total'))
                                       ->whereYear('tanggal_mutasi', $currentYear)
                                       ->groupBy('jenis_mutasi')
                                       ->get(),
            'bulanan' => MutasiPenduduk::selectRaw('MONTH(tanggal_mutasi) as bulan, jenis_mutasi, COUNT(*) as total')
                                     ->whereYear('tanggal_mutasi', $currentYear)
                                     ->groupBy('bulan', 'jenis_mutasi')
                                     ->get()
                                     ->groupBy('bulan'),
            'tren_kelahiran' => MutasiPenduduk::where('jenis_mutasi', 'Lahir')
                                            ->whereYear('tanggal_mutasi', $currentYear)
                                            ->selectRaw('MONTH(tanggal_mutasi) as bulan, COUNT(*) as total')
                                            ->groupBy('bulan')
                                            ->orderBy('bulan')
                                            ->get(),
            'tren_kematian' => MutasiPenduduk::where('jenis_mutasi', 'Meninggal')
                                           ->whereYear('tanggal_mutasi', $currentYear)
                                           ->selectRaw('MONTH(tanggal_mutasi) as bulan, COUNT(*) as total')
                                           ->groupBy('bulan')
                                           ->orderBy('bulan')
                                           ->get()
        ];
    }

    /**
     * Get religion statistics for dashboard
     */
    private function getStatistikAgama()
    {
        return Penduduk::hidup()
                      ->whereNotNull('agama')
                      ->select('agama', DB::raw('count(*) as total'))
                      ->groupBy('agama')
                      ->orderByDesc('total')
                      ->get()
                      ->map(function($item) {
                          return [
                              'agama' => $item->agama ?: 'Tidak Diketahui',
                              'total' => $item->total,
                              'persentase' => 0 // Will be calculated in frontend
                          ];
                      });
    }

    /**
     * Get marital status statistics for dashboard
     */
    private function getStatistikPerkawinan()
    {
        return Penduduk::hidup()
                      ->whereNotNull('status_kawin')
                      ->select('status_kawin as status', DB::raw('count(*) as total'))
                      ->groupBy('status_kawin')
                      ->orderByDesc('total')
                      ->get()
                      ->map(function($item) {
                          return [
                              'status' => $item->status ?: 'Tidak Diketahui',
                              'total' => $item->total,
                              'color' => $this->getStatusKawinColor($item->status),
                              'persentase' => 0 // Will be calculated in frontend
                          ];
                      });
    }

    /**
     * Helper method to get marital status colors
     */
    private function getStatusKawinColor($status)
    {
        return match($status) {
            'Belum Kawin' => '#007bff',      // Blue
            'Kawin' => '#28a745',            // Green
            'Cerai Hidup' => '#ffc107',      // Yellow
            'Cerai Mati' => '#6c757d',       // Gray
            default => '#17a2b8'             // Teal
        };
    }
} 
