<?php

namespace App\Http\Controllers;

use App\Models\MutasiPenduduk;
use App\Models\Penduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MutasiPendudukController extends Controller
{
    public function index(Request $request)
    {
        $query = MutasiPenduduk::with('penduduk');

        // Filter by jenis mutasi
        if ($request->jenis_mutasi) {
            $query->where('jenis_mutasi', $request->jenis_mutasi);
        }

        // Filter by tahun
        if ($request->tahun) {
            $query->whereYear('tanggal_mutasi', $request->tahun);
        }

        // Filter by bulan
        if ($request->bulan && $request->tahun) {
            $query->whereMonth('tanggal_mutasi', $request->bulan);
        }

        // Search by NIK or nama
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('nik', 'like', '%' . $request->search . '%')
                  ->orWhereHas('penduduk', function($sq) use ($request) {
                      $sq->where('nama', 'like', '%' . $request->search . '%');
                  });
            });
        }

        $mutasi = $query->orderBy('tanggal_mutasi', 'desc')->paginate(15);

        // Statistik untuk dashboard
        $statistik = [
            'total' => MutasiPenduduk::count(),
            'bulan_ini' => MutasiPenduduk::whereMonth('tanggal_mutasi', date('m'))
                                        ->whereYear('tanggal_mutasi', date('Y'))
                                        ->count(),
            'by_jenis' => MutasiPenduduk::selectRaw('jenis_mutasi, COUNT(*) as total')
                                       ->groupBy('jenis_mutasi')
                                       ->pluck('total', 'jenis_mutasi')
                                       ->toArray()
        ];

        return view('admin.mutasi-penduduk.index', compact('mutasi', 'statistik'));
    }

    public function create()
    {
        $penduduk = Penduduk::select('nik', 'nama')->orderBy('nama')->get();
        return view('admin.mutasi-penduduk.create', compact('penduduk'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_mutasi' => 'required|in:Lahir,Meninggal,Pindah Masuk,Pindah Keluar,Status Update',
            'tanggal_mutasi' => 'required|date',
            'keterangan' => 'nullable|string',
            'asal_daerah' => 'nullable|string|max:255',
            'tujuan_daerah' => 'nullable|string|max:255',
            'sebab_meninggal' => 'nullable|string|max:255',
            'tempat_meninggal' => 'nullable|string|max:255',
            'no_surat_keterangan' => 'nullable|string|max:255',
            'catatan_admin' => 'nullable|string'
        ]);

        // Backup data penduduk sebelum perubahan
        $penduduk = Penduduk::where('nik', $validated['nik'])->first();
        $validated['data_lama'] = $penduduk->toArray();
        $validated['admin_user'] = Auth::user()->name ?? 'System';

        $mutasi = MutasiPenduduk::create($validated);

        // Update status hidup penduduk jika meninggal
        if ($validated['jenis_mutasi'] === 'Meninggal') {
            $penduduk->update(['status_hidup' => 'Meninggal']);
        }

        return redirect()->route('admin.mutasi-penduduk.index')
                        ->with('success', 'Data mutasi penduduk berhasil ditambahkan');
    }

    public function show(MutasiPenduduk $mutasiPenduduk)
    {
        $mutasiPenduduk->load('penduduk');
        return view('admin.mutasi-penduduk.show', compact('mutasiPenduduk'));
    }

    public function edit(MutasiPenduduk $mutasiPenduduk)
    {
        $penduduk = Penduduk::select('nik', 'nama')->orderBy('nama')->get();
        return view('admin.mutasi-penduduk.edit', compact('mutasiPenduduk', 'penduduk'));
    }

    public function update(Request $request, MutasiPenduduk $mutasiPenduduk)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_mutasi' => 'required|in:Lahir,Meninggal,Pindah Masuk,Pindah Keluar,Status Update',
            'tanggal_mutasi' => 'required|date',
            'keterangan' => 'nullable|string',
            'asal_daerah' => 'nullable|string|max:255',
            'tujuan_daerah' => 'nullable|string|max:255',
            'sebab_meninggal' => 'nullable|string|max:255',
            'tempat_meninggal' => 'nullable|string|max:255',
            'no_surat_keterangan' => 'nullable|string|max:255',
            'catatan_admin' => 'nullable|string'
        ]);

        $validated['admin_user'] = Auth::user()->name ?? 'System';
        $mutasiPenduduk->update($validated);

        return redirect()->route('admin.mutasi-penduduk.index')
                        ->with('success', 'Data mutasi penduduk berhasil diperbarui');
    }

    public function destroy(MutasiPenduduk $mutasiPenduduk)
    {
        $mutasiPenduduk->delete();

        return redirect()->route('admin.mutasi-penduduk.index')
                        ->with('success', 'Data mutasi penduduk berhasil dihapus');
    }

    public function statistik(Request $request)
    {
        $tahun = $request->get('tahun', date('Y'));
        
        $statistikBulanan = MutasiPenduduk::getStatistikBulanan($tahun);
        $statistikTahunan = MutasiPenduduk::getStatistikTahunan();

        // Statistik per jenis mutasi
        $jenisStatistik = MutasiPenduduk::selectRaw('jenis_mutasi, COUNT(*) as total')
                                       ->whereYear('tanggal_mutasi', $tahun)
                                       ->groupBy('jenis_mutasi')
                                       ->get();

        return view('admin.mutasi-penduduk.statistik', compact('statistikBulanan', 'statistikTahunan', 'jenisStatistik', 'tahun'));
    }

    // API untuk mendapatkan data mutasi sebagai JSON
    public function apiData(Request $request)
    {
        $query = MutasiPenduduk::with('penduduk:nik,nama');

        if ($request->jenis_mutasi) {
            $query->where('jenis_mutasi', $request->jenis_mutasi);
        }

        if ($request->tahun) {
            $query->whereYear('tanggal_mutasi', $request->tahun);
        }

        $mutasi = $query->orderBy('tanggal_mutasi', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $mutasi
        ]);
    }

    // API untuk statistik chart
    public function apiStatistik(Request $request)
    {
        $tahun = $request->get('tahun', date('Y'));
        
        $bulanan = MutasiPenduduk::selectRaw('MONTH(tanggal_mutasi) as bulan, jenis_mutasi, COUNT(*) as total')
                                ->whereYear('tanggal_mutasi', $tahun)
                                ->groupBy('bulan', 'jenis_mutasi')
                                ->get()
                                ->groupBy('bulan');

        return response()->json([
            'success' => true,
            'data' => $bulanan
        ]);
    }
}
