<?php

namespace App\Http\Controllers;

use App\Models\Pbb;
use App\Models\TagihanPbb;
use App\Models\Penduduk;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PbbController extends Controller
{
    public function index(Request $request)
    {
        $query = Pbb::with(['pemilik', 'tagihan' => function($q) {
            $q->where('tahun', now()->year);
        }]);

        // Filter pencarian
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nop', 'like', "%{$search}%")
                  ->orWhere('alamat_objek', 'like', "%{$search}%")
                  ->orWhereHas('pemilik', function($pq) use ($search) {
                      $pq->where('nama', 'like', "%{$search}%");
                  });
            });
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan kelas
        if ($request->filled('kelas')) {
            $query->where('kelas_tanah', $request->kelas)
                  ->orWhere('kelas_bangunan', $request->kelas);
        }

        $pbbData = $query->latest()->paginate(15);
        
        return view('admin.pbb.index', compact('pbbData'));
    }

    public function create()
    {
        $penduduk = Penduduk::hidup()->orderBy('nama')->get();
        
        return view('admin.pbb.create', compact('penduduk'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nop' => 'required|string|size:18|unique:pbb,nop',
            'nik_pemilik' => 'required|exists:penduduk,nik',
            'alamat_objek' => 'required|string|max:500',
            'rt' => 'required|string|max:3',
            'rw' => 'required|string|max:3',
            'dusun' => 'required|string|max:100',
            'luas_tanah' => 'required|numeric|min:0',
            'luas_bangunan' => 'nullable|numeric|min:0',
            'njop_tanah_m2' => 'required|numeric|min:0',
            'njop_bangunan_m2' => 'nullable|numeric|min:0',
            'kelas_tanah' => 'required|string|max:10',
            'kelas_bangunan' => 'nullable|string|max:10',
            'tahun_pendataan' => 'required|integer|min:1990|max:' . (now()->year + 1),
            'status' => 'required|in:Aktif,Tidak Aktif,Hapus'
        ]);

        DB::beginTransaction();
        try {
            $pbb = Pbb::create($request->all());
            
            // Hitung dan buat tagihan untuk tahun ini
            $pbb->hitungPajak(now()->year);
            
            DB::commit();
            
            // Log activity
            AuditLog::logDataChange('create', $pbb, null, $request->all());
            
            return redirect()->route('admin.pbb.index')
                           ->with('success', 'Data PBB berhasil ditambahkan');
                           
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal menambahkan data PBB: ' . $e->getMessage());
        }
    }

    public function show(Pbb $pbb)
    {
        $pbb->load(['pemilik', 'tagihan' => function($q) {
            $q->orderBy('tahun', 'desc');
        }]);
        
        $riwayatTagihan = $pbb->tagihan()->with('pembayaran')->get();
        
        return view('admin.pbb.show', compact('pbb', 'riwayatTagihan'));
    }

    public function edit(Pbb $pbb)
    {
        $penduduk = Penduduk::hidup()->orderBy('nama')->get();
        
        return view('admin.pbb.edit', compact('pbb', 'penduduk'));
    }

    public function update(Request $request, Pbb $pbb)
    {
        $request->validate([
            'nop' => 'required|string|size:18|unique:pbb,nop,' . $pbb->id,
            'nik_pemilik' => 'required|exists:penduduk,nik',
            'alamat_objek' => 'required|string|max:500',
            'rt' => 'required|string|max:3',
            'rw' => 'required|string|max:3',
            'dusun' => 'required|string|max:100',
            'luas_tanah' => 'required|numeric|min:0',
            'luas_bangunan' => 'nullable|numeric|min:0',
            'njop_tanah_m2' => 'required|numeric|min:0',
            'njop_bangunan_m2' => 'nullable|numeric|min:0',
            'kelas_tanah' => 'required|string|max:10',
            'kelas_bangunan' => 'nullable|string|max:10',
            'tahun_pendataan' => 'required|integer|min:1990|max:' . (now()->year + 1),
            'status' => 'required|in:Aktif,Tidak Aktif,Hapus'
        ]);

        $oldData = $pbb->toArray();
        
        DB::beginTransaction();
        try {
            $pbb->update($request->all());
            
            // Recalculate pajak untuk tahun ini jika ada perubahan NJOP
            if ($pbb->wasChanged(['njop_tanah_m2', 'njop_bangunan_m2', 'luas_tanah', 'luas_bangunan'])) {
                $pbb->hitungPajak(now()->year, true); // Force recalculate
            }
            
            DB::commit();
            
            // Log activity
            AuditLog::logDataChange('update', $pbb, $oldData, $request->all());
            
            return redirect()->route('admin.pbb.index')
                           ->with('success', 'Data PBB berhasil diperbarui');
                           
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal memperbarui data PBB: ' . $e->getMessage());
        }
    }

    public function destroy(Pbb $pbb)
    {
        $oldData = $pbb->toArray();
        
        DB::beginTransaction();
        try {
            // Soft delete - ubah status menjadi Hapus
            $pbb->update(['status' => 'Hapus']);
            
            DB::commit();
            
            // Log activity
            AuditLog::logDataChange('delete', $pbb, $oldData, ['status' => 'Hapus']);
            
            return redirect()->route('admin.pbb.index')
                           ->with('success', 'Data PBB berhasil dihapus');
                           
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->with('error', 'Gagal menghapus data PBB: ' . $e->getMessage());
        }
    }

    public function generateTagihan(Request $request)
    {
        $request->validate([
            'tahun' => 'required|integer|min:2020|max:' . (now()->year + 2),
            'pbb_ids' => 'nullable|array',
            'pbb_ids.*' => 'exists:pbb,id'
        ]);

        $tahun = $request->tahun;
        $query = Pbb::aktif();
        
        if ($request->filled('pbb_ids')) {
            $query->whereIn('id', $request->pbb_ids);
        }

        $pbbData = $query->get();
        $generated = 0;
        $errors = [];

        DB::beginTransaction();
        try {
            foreach ($pbbData as $pbb) {
                try {
                    $pbb->hitungPajak($tahun, true);
                    $generated++;
                } catch (\Exception $e) {
                    $errors[] = "NOP {$pbb->nop}: " . $e->getMessage();
                }
            }
            
            DB::commit();
            
            // Log activity
            AuditLog::logActivity([
                'action_type' => 'GENERATE_TAGIHAN',
                'module' => 'PBB',
                'description' => "Generate {$generated} tagihan PBB untuk tahun {$tahun}",
                'metadata' => [
                    'tahun' => $tahun,
                    'total_generated' => $generated,
                    'errors_count' => count($errors)
                ]
            ]);
            
            $message = "{$generated} tagihan berhasil dibuat untuk tahun {$tahun}";
            if (!empty($errors)) {
                $message .= ". Dengan " . count($errors) . " error.";
            }
            
            return redirect()->back()->with('success', $message);
            
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->with('error', 'Gagal generate tagihan: ' . $e->getMessage());
        }
    }

    public function cetakSppt(Pbb $pbb, $tahun)
    {
        $tagihan = $pbb->tagihan()->where('tahun', $tahun)->first();
        
        if (!$tagihan) {
            return redirect()->back()
                           ->with('error', 'Tagihan untuk tahun tersebut tidak ditemukan');
        }
        
        // Log activity
        AuditLog::logActivity([
            'action_type' => 'PRINT_SPPT',
            'module' => 'PBB',
            'description' => "Print SPPT NOP {$pbb->nop} tahun {$tahun}",
            'metadata' => [
                'nop' => $pbb->nop,
                'tahun' => $tahun,
                'tagihan_id' => $tagihan->id
            ]
        ]);
        
        return view('admin.pbb.sppt', compact('pbb', 'tagihan'));
    }

    public function statistik()
    {
        $stats = [
            'total_objek' => Pbb::aktif()->count(),
            'total_tagihan_tahun_ini' => TagihanPbb::where('tahun', now()->year)->sum('pajak_terhutang'),
            'total_terbayar' => TagihanPbb::where('tahun', now()->year)->sum('pajak_terbayar'),
            'objek_per_kelas' => Pbb::aktif()
                                   ->selectRaw('kelas_tanah, COUNT(*) as total')
                                   ->groupBy('kelas_tanah')
                                   ->pluck('total', 'kelas_tanah'),
            'tagihan_per_status' => TagihanPbb::where('tahun', now()->year)
                                              ->selectRaw('status_bayar, COUNT(*) as total, SUM(pajak_terhutang) as jumlah')
                                              ->groupBy('status_bayar')
                                              ->get(),
            'pembayaran_per_bulan' => TagihanPbb::where('tahun', now()->year)
                                                ->whereNotNull('tanggal_bayar')
                                                ->selectRaw('MONTH(tanggal_bayar) as bulan, COUNT(*) as total, SUM(pajak_terbayar) as jumlah')
                                                ->groupBy('bulan')
                                                ->get()
        ];
        
        return view('admin.pbb.statistik', compact('stats'));
    }

    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:excel,pdf',
            'tahun' => 'required|integer'
        ]);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'EXPORT',
            'module' => 'PBB',
            'description' => "Export data PBB format {$request->format} tahun {$request->tahun}",
            'metadata' => [
                'format' => $request->format,
                'tahun' => $request->tahun
            ]
        ]);

        // Implementation would depend on export library (PhpSpreadsheet, etc)
        return redirect()->back()
                       ->with('info', 'Fitur export akan segera tersedia');
    }
} 