<?php

namespace App\Http\Controllers;

use App\Models\PelayananSurat;
use App\Models\Penduduk;
use App\Models\TemplateSurat;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class PelayananSuratController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    public function index(Request $request)
    {
        $query = PelayananSurat::with(['penduduk', 'template']);

        // Filter berdasarkan status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan jenis surat
        if ($request->jenis_surat) {
            $query->where('jenis_surat', $request->jenis_surat);
        }

        // Filter berdasarkan tanggal
        if ($request->tanggal_dari && $request->tanggal_sampai) {
            $query->whereBetween('tanggal_pengajuan', [$request->tanggal_dari, $request->tanggal_sampai]);
        }

        // Pencarian
        if ($request->search) {
            $query->whereHas('penduduk', function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->search . '%')
                  ->orWhere('nik', 'like', '%' . $request->search . '%');
            });
        }

        // Handle export
        if ($request->export) {
            return $this->exportData($query, $request->export);
        }

        $pelayananSurat = $query->orderBy('tanggal_pengajuan', 'desc')->paginate(20);
        
        // Get penduduk data for the add modal
        $penduduk = Penduduk::orderBy('nama')->get();
        
        // Calculate statistics
        $stats = [
            'total' => PelayananSurat::count(),
            'menunggu' => PelayananSurat::where('status', 'Menunggu')->count(),
            'diproses' => PelayananSurat::where('status', 'Diproses')->count(),
            'selesai' => PelayananSurat::where('status', 'Selesai')->count(),
            'ditolak' => PelayananSurat::where('status', 'Ditolak')->count(),
        ];

        return view('admin.pelayanan-surat.index', compact('pelayananSurat', 'penduduk', 'stats'));
    }

    public function create()
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        $templates = TemplateSurat::aktif()->orderBy('nama_template')->get();
        return view('admin.pelayanan-surat.create', compact('penduduk', 'templates'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_surat' => 'required|string',
            'keperluan' => 'nullable|string',
            'tanggal_pengajuan' => 'required|date',
            'status' => 'required|in:Menunggu,Diproses,Selesai,Ditolak',
            'catatan' => 'nullable|string',
            'template_id' => 'nullable|exists:template_surat,id'
        ]);

        $pelayananSurat = PelayananSurat::create($validated);

        // If this is an AJAX request (from the modal)
        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Permohonan surat berhasil ditambahkan',
                'data' => $pelayananSurat->load(['penduduk', 'template'])
            ]);
        }

        return redirect()->route('admin.pelayanan-surat.index')
                        ->with('success', 'Permohonan surat berhasil ditambahkan');
    }

    public function show(PelayananSurat $pelayananSurat)
    {
        $pelayananSurat->load(['penduduk', 'template']);
        
        // Check if this is an AJAX request for JSON response
        if (request()->wantsJson() || request()->ajax()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $pelayananSurat->id,
                    'jenis_surat' => $pelayananSurat->jenis_surat,
                    'keperluan' => $pelayananSurat->keperluan,
                    'tanggal_pengajuan' => $pelayananSurat->tanggal_pengajuan->format('d/m/Y'),
                    'status' => $pelayananSurat->status,
                    'status_color' => $this->getStatusColor($pelayananSurat->status),
                    'catatan' => $pelayananSurat->catatan,
                    'file_pdf' => $pelayananSurat->file_pdf,
                    'tanggal_selesai' => $pelayananSurat->tanggal_selesai ? $pelayananSurat->tanggal_selesai->format('d/m/Y H:i') : null,
                    'template' => $pelayananSurat->template ? [
                        'id' => $pelayananSurat->template->id,
                        'nama_template' => $pelayananSurat->template->nama_template,
                        'jenis_surat' => $pelayananSurat->template->jenis_surat
                    ] : null,
                    'penduduk' => [
                        'nama' => $pelayananSurat->penduduk->nama,
                        'nik' => $pelayananSurat->penduduk->nik,
                        'alamat' => $pelayananSurat->penduduk->alamat,
                        'no_wa' => $pelayananSurat->penduduk->no_wa,
                    ]
                ]
            ]);
        }
        
        return view('admin.pelayanan-surat.show', compact('pelayananSurat'));
    }
    
    private function getStatusColor($status)
    {
        return match($status) {
            'Menunggu' => 'warning',
            'Diproses' => 'info',
            'Selesai' => 'success',
            'Ditolak' => 'danger',
            default => 'secondary'
        };
    }

    public function edit(PelayananSurat $pelayananSurat)
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        $templates = TemplateSurat::aktif()->orderBy('nama_template')->get();
        return view('admin.pelayanan-surat.edit', compact('pelayananSurat', 'penduduk', 'templates'));
    }

    public function update(Request $request, PelayananSurat $pelayananSurat)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_surat' => 'required|string',
            'keperluan' => 'nullable|string',
            'tanggal_pengajuan' => 'required|date',
            'status' => 'required|in:Menunggu,Diproses,Selesai,Ditolak',
            'catatan' => 'nullable|string',
            'template_id' => 'nullable|exists:template_surat,id'
        ]);

        $oldStatus = $pelayananSurat->status;
        $pelayananSurat->update($validated);

        // Kirim notifikasi WA jika status berubah ke Selesai
        if ($oldStatus !== 'Selesai' && $validated['status'] === 'Selesai') {
            $this->sendCompletionNotification($pelayananSurat);
        }

        return redirect()->route('admin.pelayanan-surat.index')
                        ->with('success', 'Data pelayanan surat berhasil diperbarui');
    }

    public function destroy(PelayananSurat $pelayananSurat)
    {
        // Hapus file PDF jika ada
        if ($pelayananSurat->file_pdf) {
            Storage::delete('public/surat/' . $pelayananSurat->file_pdf);
        }

        $pelayananSurat->delete();
        
        return redirect()->route('admin.pelayanan-surat.index')
                        ->with('success', 'Data pelayanan surat berhasil dihapus');
    }

    public function assignTemplate(Request $request, PelayananSurat $pelayananSurat)
    {
        $request->validate([
            'template_id' => 'required|exists:template_surat,id'
        ]);

        $pelayananSurat->update([
            'template_surat' => $request->template_id
        ]);

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Template berhasil ditugaskan'
            ]);
        }

        return redirect()->back()->with('success', 'Template berhasil ditugaskan');
    }

    public function generateSurat(Request $request, PelayananSurat $pelayananSurat)
    {
        if (!$pelayananSurat->template_surat) {
            return response()->json([
                'success' => false,
                'message' => 'Template belum dipilih'
            ], 400);
        }

        try {
            // Generate PDF from template
            $pdfPath = $pelayananSurat->generatePDF();
            
            // Update status and file path
            $pelayananSurat->update([
                'file_pdf' => $pdfPath,
                'status' => 'Selesai',
                'tanggal_selesai' => now()
            ]);

            // Send WhatsApp notification
            $this->sendCompletionNotification($pelayananSurat);

            if ($request->wantsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Surat berhasil digenerate',
                    'file_path' => asset('storage/surat/' . $pdfPath)
                ]);
            }

            return redirect()->back()->with('success', 'Surat berhasil digenerate');
        } catch (\Exception $e) {
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal generate surat: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()->with('error', 'Gagal generate surat: ' . $e->getMessage());
        }
    }

    public function uploadPdf(Request $request, PelayananSurat $pelayananSurat)
    {
        $request->validate([
            'file_pdf' => 'required|file|mimes:pdf|max:5120' // Max 5MB
        ]);

        // Hapus file lama jika ada
        if ($pelayananSurat->file_pdf) {
            Storage::delete('public/surat/' . $pelayananSurat->file_pdf);
        }

        // Upload file baru
        $fileName = time() . '_' . $pelayananSurat->id . '.pdf';
        $request->file('file_pdf')->storeAs('public/surat', $fileName);

        // Update database
        $pelayananSurat->update([
            'file_pdf' => $fileName,
            'status' => 'Selesai',
            'tanggal_selesai' => now()
        ]);

        // Kirim notifikasi WA
        $this->sendCompletionNotification($pelayananSurat);

        return redirect()->back()->with('success', 'File PDF berhasil diupload dan notifikasi telah dikirim');
    }

    public function downloadPdf(PelayananSurat $pelayananSurat)
    {
        if (!$pelayananSurat->file_pdf || !Storage::exists('public/surat/' . $pelayananSurat->file_pdf)) {
            return redirect()->back()->with('error', 'File PDF tidak ditemukan');
        }

        return response()->download(
            storage_path('app/public/surat/' . $pelayananSurat->file_pdf),
            $pelayananSurat->jenis_surat . '_' . $pelayananSurat->penduduk->nama . '.pdf'
        );
    }

    public function updateStatus(Request $request, PelayananSurat $pelayananSurat)
    {
        $request->validate([
            'status' => 'required|in:Menunggu,Diproses,Selesai,Ditolak',
            'catatan' => 'nullable|string'
        ]);

        $oldStatus = $pelayananSurat->status;
        
        $updateData = ['status' => $request->status];
        if ($request->catatan) {
            $updateData['catatan'] = $request->catatan;
        }
        
        if ($request->status === 'Selesai') {
            $updateData['tanggal_selesai'] = now();
        }

        $pelayananSurat->update($updateData);

        // Kirim notifikasi WA jika status berubah ke Selesai
        if ($oldStatus !== 'Selesai' && $request->status === 'Selesai') {
            $this->sendCompletionNotification($pelayananSurat);
        }

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Status berhasil diubah'
            ]);
        }

        return redirect()->back()->with('success', 'Status berhasil diubah');
    }

    private function sendCompletionNotification(PelayananSurat $pelayananSurat)
    {
        if (!$pelayananSurat->penduduk->no_wa) {
            return;
        }

        $message = "Yth. {$pelayananSurat->penduduk->nama}, \n\n";
        $message .= "Permohonan {$pelayananSurat->jenis_surat} Anda telah selesai diproses.\n\n";
        $message .= "Silakan ambil surat di kantor desa dengan membawa berkas persyaratan.\n\n";
        $message .= "Terima kasih.\n\n";
        $message .= "- Kantor Desa";

        try {
            $this->whatsappService->sendMessage($pelayananSurat->penduduk->no_wa, $message);
        } catch (\Exception $e) {
            \Log::error('WhatsApp notification failed: ' . $e->getMessage());
        }
    }

    private function exportData($query, $format)
    {
        $data = $query->get();
        $filename = 'pelayanan_surat_' . date('Y-m-d_H-i-s');

        switch ($format) {
            case 'pdf':
                // Implement PDF export
                return $this->exportToPDF($data, $filename);
            
            case 'excel':
                // Implement Excel export
                return $this->exportToExcel($data, $filename);
            
            case 'csv':
                // Implement CSV export
                return $this->exportToCSV($data, $filename);
            
            default:
                return redirect()->back()->with('error', 'Format export tidak didukung');
        }
    }

    private function exportToPDF($data, $filename)
    {
        // Implement PDF export logic
        return redirect()->back()->with('info', 'Export PDF belum tersedia');
    }

    private function exportToExcel($data, $filename)
    {
        // Implement Excel export logic
        return redirect()->back()->with('info', 'Export Excel belum tersedia');
    }

    private function exportToCSV($data, $filename)
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '.csv"',
        ];

        $callback = function() use ($data) {
            $file = fopen('php://output', 'w');
            
            // Header CSV
            fputcsv($file, [
                'No',
                'NIK',
                'Nama',
                'Jenis Surat',
                'Tanggal Pengajuan',
                'Status',
                'Keperluan',
                'Template',
                'File PDF'
            ]);

            // Data CSV
            foreach ($data as $index => $surat) {
                fputcsv($file, [
                    $index + 1,
                    $surat->penduduk->nik,
                    $surat->penduduk->nama,
                    $surat->jenis_surat,
                    $surat->tanggal_pengajuan->format('d/m/Y'),
                    $surat->status,
                    $surat->keperluan,
                    $surat->template ? $surat->template->nama_template : '-',
                    $surat->file_pdf ? 'Ada' : 'Belum ada'
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    // API Methods for mobile app
    public function apiIndex(Request $request)
    {
        $query = PelayananSurat::with('penduduk');
        
        if ($request->nik) {
            $query->whereHas('penduduk', function($q) use ($request) {
                $q->where('nik', $request->nik);
            });
        }

        $pelayananSurat = $query->orderBy('tanggal_pengajuan', 'desc')->paginate(10);

        return response()->json([
            'success' => true,
            'data' => $pelayananSurat
        ]);
    }

    public function apiStore(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_surat' => 'required|string',
            'keperluan' => 'nullable|string',
        ]);

        $validated['tanggal_pengajuan'] = now();
        $validated['status'] = 'Menunggu';

        $pelayananSurat = PelayananSurat::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Permohonan surat berhasil diajukan',
            'data' => $pelayananSurat->load('penduduk')
        ], 201);
    }

    public function apiShow(PelayananSurat $pelayananSurat)
    {
        $pelayananSurat->load(['penduduk', 'template']);
        
        return response()->json([
            'success' => true,
            'data' => $pelayananSurat
        ]);
    }

    public function sendWhatsApp(PelayananSurat $pelayananSurat)
    {
        if (!$pelayananSurat->penduduk->no_wa) {
            return response()->json([
                'success' => false,
                'message' => 'Nomor WhatsApp tidak tersedia'
            ], 400);
        }

        if ($pelayananSurat->status !== 'Selesai') {
            return response()->json([
                'success' => false,
                'message' => 'Surat belum selesai diproses'
            ], 400);
        }

        try {
            $this->sendCompletionNotification($pelayananSurat);
            
            return response()->json([
                'success' => true,
                'message' => 'Notifikasi WhatsApp berhasil dikirim'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengirim notifikasi: ' . $e->getMessage()
            ], 500);
        }
    }
} 
