<?php

namespace App\Http\Controllers;

use App\Models\Pengaduan;
use App\Models\Penduduk;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PengaduanController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    public function index(Request $request)
    {
        $query = Pengaduan::with('penduduk');

        // Filter berdasarkan status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan kategori
        if ($request->kategori) {
            $query->byKategori($request->kategori);
        }

        // Filter berdasarkan tanggal
        if ($request->tanggal_dari && $request->tanggal_sampai) {
            $query->whereBetween('created_at', [$request->tanggal_dari, $request->tanggal_sampai]);
        }

        // Pencarian
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('judul', 'like', '%' . $request->search . '%')
                  ->orWhere('isi_pengaduan', 'like', '%' . $request->search . '%')
                  ->orWhereHas('penduduk', function($subQ) use ($request) {
                      $subQ->where('nama', 'like', '%' . $request->search . '%');
                  });
            });
        }

        $pengaduan = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.pengaduan.index', compact('pengaduan'));
    }

    public function create()
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.pengaduan.create', compact('penduduk'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'judul' => 'required|string|max:255',
            'isi_pengaduan' => 'required|string',
            'kategori' => 'required|string',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180'
        ]);

        // Upload gambar jika ada
        if ($request->hasFile('gambar')) {
            $validated['gambar'] = $request->file('gambar')->store('pengaduan', 'public');
        }

        $pengaduan = Pengaduan::create($validated);

        return redirect()->route('admin.pengaduan.index')
                        ->with('success', 'Pengaduan berhasil ditambahkan');
    }

    public function show(Pengaduan $pengaduan)
    {
        $pengaduan->load('penduduk');
        return view('admin.pengaduan.show', compact('pengaduan'));
    }

    public function edit(Pengaduan $pengaduan)
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.pengaduan.edit', compact('pengaduan', 'penduduk'));
    }

    public function update(Request $request, Pengaduan $pengaduan)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'judul' => 'required|string|max:255',
            'isi_pengaduan' => 'required|string',
            'kategori' => 'required|string',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'status' => 'required|in:Baru,Diproses,Ditanggapi,Selesai'
        ]);

        // Upload gambar baru jika ada
        if ($request->hasFile('gambar')) {
            // Hapus gambar lama
            if ($pengaduan->gambar) {
                Storage::delete('public/' . $pengaduan->gambar);
            }
            $validated['gambar'] = $request->file('gambar')->store('pengaduan', 'public');
        }

        $pengaduan->update($validated);

        return redirect()->route('admin.pengaduan.index')
                        ->with('success', 'Pengaduan berhasil diperbarui');
    }

    public function destroy(Pengaduan $pengaduan)
    {
        // Hapus gambar jika ada
        if ($pengaduan->gambar) {
            Storage::delete('public/' . $pengaduan->gambar);
        }

        $pengaduan->delete();
        
        return redirect()->route('admin.pengaduan.index')
                        ->with('success', 'Pengaduan berhasil dihapus');
    }

    public function tanggapi(Request $request, Pengaduan $pengaduan)
    {
        $request->validate([
            'tanggapan' => 'required|string',
            'status' => 'required|in:Diproses,Ditanggapi,Selesai'
        ]);

        $pengaduan->update([
            'tanggapan' => $request->tanggapan,
            'status' => $request->status,
            'tanggal_tanggapan' => now()
        ]);

        // Kirim notifikasi WA jika status berubah ke Ditanggapi
        if ($request->status === 'Ditanggapi') {
            $this->sendResponseNotification($pengaduan);
        }

        return response()->json([
            'success' => true,
            'message' => 'Tanggapan berhasil disimpan'
        ]);
    }

    private function sendResponseNotification(Pengaduan $pengaduan)
    {
        $penduduk = $pengaduan->penduduk;
        
        if ($penduduk->no_wa) {
            $message = $this->whatsappService->templatePengaduanDitanggapi(
                $penduduk->nama,
                $pengaduan->judul,
                $pengaduan->tanggapan
            );

            $this->whatsappService->sendMessage(
                $penduduk->no_wa,
                $message,
                'pengaduan_ditanggapi',
                $pengaduan->id,
                Pengaduan::class
            );
        }
    }

    // API Methods untuk Flutter
    public function apiIndex(Request $request)
    {
        $nik = $request->user()->nik ?? $request->nik;
        
        $pengaduan = Pengaduan::where('nik', $nik)
                             ->orderBy('created_at', 'desc')
                             ->get()
                             ->map(function($item) {
                                 return [
                                     'id' => $item->id,
                                     'judul' => $item->judul,
                                     'isi_pengaduan' => $item->isi_pengaduan,
                                     'kategori' => $item->kategori,
                                     'status' => $item->status,
                                     'tanggapan' => $item->tanggapan,
                                     'gambar' => $item->gambar ? url('storage/' . $item->gambar) : null,
                                     'has_location' => $item->has_location,
                                     'latitude' => $item->latitude,
                                     'longitude' => $item->longitude,
                                     'tanggal_pengaduan' => $item->created_at->format('d/m/Y H:i'),
                                     'tanggal_tanggapan' => $item->tanggal_tanggapan ? $item->tanggal_tanggapan->format('d/m/Y H:i') : null
                                 ];
                             });

        return response()->json([
            'success' => true,
            'data' => $pengaduan
        ]);
    }

    public function apiStore(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'judul' => 'required|string|max:255',
            'isi_pengaduan' => 'required|string',
            'kategori' => 'required|string',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180'
        ]);

        // Upload gambar jika ada
        if ($request->hasFile('gambar')) {
            $validated['gambar'] = $request->file('gambar')->store('pengaduan', 'public');
        }

        $validated['status'] = 'Baru';

        $pengaduan = Pengaduan::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Pengaduan berhasil dikirim',
            'data' => [
                'id' => $pengaduan->id,
                'judul' => $pengaduan->judul,
                'status' => $pengaduan->status,
                'tanggal_pengaduan' => $pengaduan->created_at->format('d/m/Y H:i')
            ]
        ]);
    }

    public function apiShow(Pengaduan $pengaduan)
    {
        $pengaduan->load('penduduk');
        
        return response()->json([
            'success' => true,
            'data' => [
                'id' => $pengaduan->id,
                'judul' => $pengaduan->judul,
                'isi_pengaduan' => $pengaduan->isi_pengaduan,
                'kategori' => $pengaduan->kategori,
                'status' => $pengaduan->status,
                'tanggapan' => $pengaduan->tanggapan,
                'gambar' => $pengaduan->gambar ? url('storage/' . $pengaduan->gambar) : null,
                'has_location' => $pengaduan->has_location,
                'latitude' => $pengaduan->latitude,
                'longitude' => $pengaduan->longitude,
                'tanggal_pengaduan' => $pengaduan->created_at->format('d/m/Y H:i'),
                'tanggal_tanggapan' => $pengaduan->tanggal_tanggapan ? $pengaduan->tanggal_tanggapan->format('d/m/Y H:i') : null,
                'penduduk' => [
                    'nama' => $pengaduan->penduduk->nama,
                    'nik' => $pengaduan->penduduk->nik
                ]
            ]
        ]);
    }
} 
