<?php

namespace App\Http\Controllers;

use App\Models\PerangkatDesa;
use App\Models\Penduduk;
use App\Models\AuditLog;
use App\Services\FileUploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PerangkatDesaController extends Controller
{
    public function index()
    {
        try {
            // Ambil semua data perangkat desa dengan relasi
            $allPerangkat = PerangkatDesa::with('penduduk')
                                     ->orderByRaw("
                                         CASE jabatan
                                             WHEN 'Kepala Desa' THEN 1
                                             WHEN 'Sekretaris Desa' THEN 2
                                             WHEN 'Kasi Pemerintahan' THEN 3
                                             WHEN 'Kasi Kesejahteraan' THEN 4
                                             WHEN 'Kasi Kesra' THEN 5
                                             WHEN 'Kasi Pelayanan' THEN 6
                                             WHEN 'Kasi Pembangunan' THEN 7
                                             WHEN 'Kaur Umum' THEN 8
                                             WHEN 'Kaur Keuangan' THEN 9
                                             WHEN 'Kaur Perencanaan' THEN 10
                                             WHEN 'Kepala Dusun' THEN 11
                                             WHEN 'Ketua RT' THEN 12
                                             WHEN 'Ketua RW' THEN 13
                                             ELSE 99
                                         END, nama
                                     ")
                                     ->get();
            
            // Group untuk struktur organisasi
            $struktur = $allPerangkat->groupBy(function($item) {
                if (in_array($item->jabatan, ['Kepala Desa'])) return 0;
                if (in_array($item->jabatan, ['Sekretaris Desa'])) return 1;
                if (in_array($item->jabatan, ['Kasi Pemerintahan', 'Kasi Kesejahteraan', 'Kasi Kesra', 'Kasi Pelayanan', 'Kasi Pembangunan', 'Kaur Umum', 'Kaur Keuangan', 'Kaur Perencanaan', 'Kepala Dusun'])) return 2;
                if (in_array($item->jabatan, ['Ketua RT', 'Ketua RW'])) return 3;
                return 4;
            });
            
        } catch (\Exception $e) {
            // Fallback jika ada error
            $allPerangkat = PerangkatDesa::with('penduduk')->orderBy('jabatan')->get();
            $struktur = $allPerangkat->groupBy(function($item) {
                if (in_array($item->jabatan, ['Kepala Desa'])) return 0;
                if (in_array($item->jabatan, ['Sekretaris Desa'])) return 1;
                if (in_array($item->jabatan, ['Kasi Pemerintahan', 'Kasi Kesejahteraan', 'Kasi Kesra', 'Kasi Pelayanan', 'Kasi Pembangunan', 'Kaur Umum', 'Kaur Keuangan', 'Kaur Perencanaan', 'Kepala Dusun'])) return 2;
                if (in_array($item->jabatan, ['Ketua RT', 'Ketua RW'])) return 3;
                return 4;
            });
        }
        
        // Kirim data ke view
        // $perangkat = collection biasa untuk where() method di view
        // $struktur = grouped collection untuk struktur organisasi
        $perangkat = $allPerangkat;
        
        return view('admin.perangkat-desa.index', compact('perangkat', 'struktur'));
    }

    public function create()
    {
        $penduduk = Penduduk::hidup()->whereNotIn('nik', function($query) {
            $query->select('nik')->from('perangkat_desa')->where('status', 'Aktif');
        })->orderBy('nama')->get();
        
        $jabatan = [
            'Kepala Desa',
            'Sekretaris Desa',
            'Kasi Pemerintahan',
            'Kasi Kesejahteraan',
            'Kasi Kesra',
            'Kasi Pelayanan',
            'Kasi Pembangunan',
            'Kaur Umum',
            'Kaur Keuangan',
            'Kaur Perencanaan',
            'Kepala Dusun',
            'Ketua RT',
            'Ketua RW'
        ];
        
        return view('admin.perangkat-desa.create', compact('penduduk', 'jabatan'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nik' => 'required|exists:penduduk,nik|unique:perangkat_desa,nik,NULL,id,status,Aktif',
            'nama' => 'required|string|max:255',
            'nip' => 'nullable|string|max:30|unique:perangkat_desa,nip',
            'jabatan' => 'required|string|max:100',
            'bidang_tugas' => 'nullable|string|max:255',
            'tugas_tanggung_jawab' => 'nullable|string',
            'mulai_tugas' => 'required|date',
            'selesai_tugas' => 'nullable|date|after:mulai_tugas',
            'status' => 'required|in:Aktif,Tidak Aktif,Pensiun,Mutasi',
            'no_hp' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:perangkat_desa,email',
            'alamat' => 'nullable|string|max:500',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'file_sk' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
            'tanda_tangan' => 'nullable|image|mimes:png|max:1024',
            'wilayah_kerja' => 'nullable|string|max:255',
            'urutan' => 'nullable|integer|min:1',
            'level_hierarki' => 'nullable|integer|min:0|max:3'
        ]);

        $data = $request->except(['foto', 'file_sk', 'tanda_tangan']);

        // Handle file uploads
        $uploadService = new FileUploadService();
        
        if ($request->hasFile('foto')) {
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'perangkat');
            if ($uploadResult['success']) {
                $data['foto'] = $uploadResult['path'];
            }
        }

        if ($request->hasFile('file_sk')) {
            $uploadResult = $uploadService->uploadSingle($request->file('file_sk'), 'perangkat/sk');
            if ($uploadResult['success']) {
                $data['file_sk'] = $uploadResult['path'];
            }
        }

        if ($request->hasFile('tanda_tangan')) {
            $uploadResult = $uploadService->uploadSingle($request->file('tanda_tangan'), 'perangkat/ttd');
            if ($uploadResult['success']) {
                $data['tanda_tangan'] = $uploadResult['path'];
            }
        }

        $perangkat = PerangkatDesa::create($data);

        // Log activity
        AuditLog::logDataChange('create', $perangkat, null, $data);

        return redirect()->route('admin.perangkat-desa.index')
                        ->with('success', 'Data perangkat desa berhasil ditambahkan');
    }

    public function show(PerangkatDesa $perangkatDesa)
    {
        $perangkatDesa->load('penduduk');
        
        return view('admin.perangkat-desa.show', compact('perangkatDesa'));
    }

    public function edit(PerangkatDesa $perangkatDesa)
    {
        $penduduk = Penduduk::hidup()->where(function($query) use ($perangkatDesa) {
            $query->whereNotIn('nik', function($subQuery) {
                $subQuery->select('nik')->from('perangkat_desa')->where('status', 'Aktif');
            })->orWhere('nik', $perangkatDesa->nik);
        })->orderBy('nama')->get();
        
        $jabatan = [
            'Kepala Desa',
            'Sekretaris Desa',
            'Kasi Pemerintahan',
            'Kasi Kesejahteraan',
            'Kasi Kesra',
            'Kasi Pelayanan',
            'Kasi Pembangunan',
            'Kaur Umum',
            'Kaur Keuangan',
            'Kaur Perencanaan',
            'Kepala Dusun',
            'Ketua RT',
            'Ketua RW'
        ];
        
        return view('admin.perangkat-desa.edit', compact('perangkatDesa', 'penduduk', 'jabatan'));
    }

    public function update(Request $request, PerangkatDesa $perangkatDesa)
    {
        $request->validate([
            'nik' => 'required|exists:penduduk,nik|unique:perangkat_desa,nik,' . $perangkatDesa->id . ',id,status,Aktif',
            'nama' => 'required|string|max:255',
            'nip' => 'nullable|string|max:30|unique:perangkat_desa,nip,' . $perangkatDesa->id,
            'jabatan' => 'required|string|max:100',
            'bidang_tugas' => 'nullable|string|max:255',
            'tugas_tanggung_jawab' => 'nullable|string',
            'mulai_tugas' => 'required|date',
            'selesai_tugas' => 'nullable|date|after:mulai_tugas',
            'status' => 'required|in:Aktif,Tidak Aktif,Pensiun,Mutasi',
            'no_hp' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:perangkat_desa,email,' . $perangkatDesa->id,
            'alamat' => 'nullable|string|max:500',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'file_sk' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
            'tanda_tangan' => 'nullable|image|mimes:png|max:1024',
            'wilayah_kerja' => 'nullable|string|max:255',
            'urutan' => 'nullable|integer|min:1',
            'level_hierarki' => 'nullable|integer|min:0|max:3'
        ]);

        $oldData = $perangkatDesa->toArray();
        $data = $request->except(['foto', 'file_sk', 'tanda_tangan']);

        // Handle file uploads
        $uploadService = new FileUploadService();
        
        if ($request->hasFile('foto')) {
            // Delete old photo
            if ($perangkatDesa->foto) {
                Storage::disk('public')->delete($perangkatDesa->foto);
            }
            
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'perangkat');
            if ($uploadResult['success']) {
                $data['foto'] = $uploadResult['path'];
            }
        }

        if ($request->hasFile('file_sk')) {
            // Delete old SK file
            if ($perangkatDesa->file_sk) {
                Storage::disk('public')->delete($perangkatDesa->file_sk);
            }
            
            $uploadResult = $uploadService->uploadSingle($request->file('file_sk'), 'perangkat/sk');
            if ($uploadResult['success']) {
                $data['file_sk'] = $uploadResult['path'];
            }
        }

        if ($request->hasFile('tanda_tangan')) {
            // Delete old signature
            if ($perangkatDesa->tanda_tangan) {
                Storage::disk('public')->delete($perangkatDesa->tanda_tangan);
            }
            
            $uploadResult = $uploadService->uploadSingle($request->file('tanda_tangan'), 'perangkat/ttd');
            if ($uploadResult['success']) {
                $data['tanda_tangan'] = $uploadResult['path'];
            }
        }

        $perangkatDesa->update($data);

        // Log activity
        AuditLog::logDataChange('update', $perangkatDesa, $oldData, $data);

        return redirect()->route('admin.perangkat-desa.index')
                        ->with('success', 'Data perangkat desa berhasil diperbarui');
    }

    public function destroy(PerangkatDesa $perangkatDesa)
    {
        // Delete files
        if ($perangkatDesa->foto) {
            Storage::disk('public')->delete($perangkatDesa->foto);
        }
        if ($perangkatDesa->file_sk) {
            Storage::disk('public')->delete($perangkatDesa->file_sk);
        }
        if ($perangkatDesa->tanda_tangan) {
            Storage::disk('public')->delete($perangkatDesa->tanda_tangan);
        }

        $nama = $perangkatDesa->nama;
        $oldData = $perangkatDesa->toArray();
        $perangkatDesa->delete();

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'DELETE',
            'module' => 'PerangkatDesa',
            'table_name' => 'perangkat_desa',
            'record_id' => $oldData['id'] ?? null,
            'description' => "Delete perangkat desa: {$nama}",
            'old_values' => $oldData,
            'new_values' => [],
            'tags' => ['data_change']
        ]);

        return redirect()->route('admin.perangkat-desa.index')
                        ->with('success', 'Data perangkat desa berhasil dihapus');
    }

    public function struktur()
    {
        $kepalaDesa = PerangkatDesa::getKepalaDesa();
        $sekretaris = PerangkatDesa::getSekretarisDesa();
        $staf = PerangkatDesa::getByLevel(1);
        $kepalaDusun = PerangkatDesa::getByLevel(2);
        $rtRw = PerangkatDesa::getByLevel(3);
        
        return view('admin.perangkat-desa.struktur', compact(
            'kepalaDesa', 'sekretaris', 'staf', 'kepalaDusun', 'rtRw'
        ));
    }

    // Alias method untuk route yang menggunakan nama 'strukturOrganisasi'
    public function strukturOrganisasi()
    {
        return $this->struktur();
    }

    public function penandatangan()
    {
        $penandatangan = PerangkatDesa::getPenandatanganSurat();
        
        return view('admin.perangkat-desa.penandatangan', compact('penandatangan'));
    }

    public function cetakSk(PerangkatDesa $perangkatDesa)
    {
        if (!$perangkatDesa->file_sk) {
            return redirect()->back()
                           ->with('error', 'File SK tidak tersedia');
        }

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'DOWNLOAD',
            'module' => 'PerangkatDesa',
            'description' => 'Download SK perangkat: ' . $perangkatDesa->nama,
            'metadata' => [
                'perangkat_id' => $perangkatDesa->id,
                'file' => $perangkatDesa->file_sk
            ]
        ]);

        return response()->download(storage_path('app/public/' . $perangkatDesa->file_sk));
    }

    public function updateStatus(Request $request, PerangkatDesa $perangkatDesa)
    {
        $request->validate([
            'status' => 'required|in:Aktif,Tidak Aktif,Pensiun,Mutasi',
            'tanggal_selesai' => 'nullable|date'
        ]);

        $oldData = $perangkatDesa->toArray();
        $data = $request->only(['status']);
        
        if ($request->status !== 'Aktif' && $request->filled('tanggal_selesai')) {
            $data['selesai_tugas'] = $request->tanggal_selesai;
        }

        $perangkatDesa->update($data);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'UPDATE_STATUS',
            'module' => 'PerangkatDesa',
            'table_name' => 'perangkat_desa',
            'record_id' => $perangkatDesa->id,
            'description' => "Update status perangkat {$perangkatDesa->nama} to {$request->status}",
            'old_values' => $oldData,
            'new_values' => $data
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status berhasil diperbarui'
        ]);
    }

    public function reorder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:perangkat_desa,id',
            'items.*.urutan' => 'required|integer|min:1'
        ]);

        foreach ($request->items as $item) {
            PerangkatDesa::where('id', $item['id'])
                        ->update(['urutan' => $item['urutan']]);
        }

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'REORDER',
            'module' => 'PerangkatDesa',
            'description' => 'Reorder perangkat desa structure',
            'metadata' => ['items' => $request->items]
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Urutan berhasil diperbarui'
        ]);
    }
} 