<?php

namespace App\Http\Controllers;

use App\Models\TemplateSurat;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TemplateSuratController extends Controller
{
    public function index()
    {
        $templates = TemplateSurat::latest()->paginate(10);
        
        return view('admin.template-surat.index', compact('templates'));
    }

    public function create()
    {
        $jenisTemplate = TemplateSurat::JENIS_TEMPLATE;
        $kategoriTemplate = TemplateSurat::KATEGORI_TEMPLATE;
        
        return view('admin.template-surat.create', compact('jenisTemplate', 'kategoriTemplate'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_template' => 'required|string|max:255',
            'jenis_surat' => 'required|string|max:100',
            'kode_surat' => 'nullable|string|max:20',
            'format_nomor' => 'nullable|string|max:100',
            'template_content' => 'required|string',
            'variabel_tersedia' => 'nullable|array',
            'penandatangan_default' => 'nullable|string|max:255',
            'aktif' => 'required|boolean',
            'urutan' => 'nullable|integer',
            'keterangan' => 'nullable|string',
            'file_template' => 'nullable|file|mimes:docx,doc|max:2048'
        ]);

        $data = $request->except(['file_template']);

        // Handle file upload
        if ($request->hasFile('file_template')) {
            $file = $request->file('file_template');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('templates', $fileName, 'public');
            $data['file_template'] = $filePath;
        }

        $template = TemplateSurat::create($data);

        // Log activity
        AuditLog::logDataChange('create', $template, null, $data);

        return redirect()->route('admin.template-surat.index')
                        ->with('success', 'Template surat berhasil dibuat');
    }

    public function show(TemplateSurat $templateSurat)
    {
        return view('admin.template-surat.show', compact('templateSurat'));
    }

    public function edit(TemplateSurat $templateSurat)
    {
        $jenisTemplate = TemplateSurat::JENIS_TEMPLATE;
        $kategoriTemplate = TemplateSurat::KATEGORI_TEMPLATE;
        
        return view('admin.template-surat.edit', compact('templateSurat', 'jenisTemplate', 'kategoriTemplate'));
    }

    public function update(Request $request, TemplateSurat $templateSurat)
    {
        $request->validate([
            'nama_template' => 'required|string|max:255',
            'jenis_surat' => 'required|string|max:100',
            'kode_surat' => 'nullable|string|max:20',
            'format_nomor' => 'nullable|string|max:100',
            'template_content' => 'required|string',
            'variabel_tersedia' => 'nullable|array',
            'penandatangan_default' => 'nullable|string|max:255',
            'aktif' => 'required|boolean',
            'urutan' => 'nullable|integer',
            'keterangan' => 'nullable|string',
            'file_template' => 'nullable|file|mimes:docx,doc|max:2048'
        ]);

        $oldData = $templateSurat->toArray();
        $data = $request->except(['file_template']);

        // Handle file upload
        if ($request->hasFile('file_template')) {
            // Delete old file
            if ($templateSurat->file_template) {
                Storage::disk('public')->delete($templateSurat->file_template);
            }
            
            $file = $request->file('file_template');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('templates', $fileName, 'public');
            $data['file_template'] = $filePath;
        }

        $templateSurat->update($data);

        // Log activity
        AuditLog::logDataChange('update', $templateSurat, $oldData, $data);

        return redirect()->route('admin.template-surat.index')
                        ->with('success', 'Template surat berhasil diperbarui');
    }

    public function destroy(TemplateSurat $templateSurat)
    {
        $oldData = $templateSurat->toArray();
        
        // Delete file if exists
        if ($templateSurat->file_template) {
            Storage::disk('public')->delete($templateSurat->file_template);
        }

        $templateSurat->delete();

        // Log activity
        AuditLog::logDataChange('delete', $templateSurat, $oldData, null);

        return redirect()->route('admin.template-surat.index')
                        ->with('success', 'Template surat berhasil dihapus');
    }

    public function preview(TemplateSurat $templateSurat)
    {
        // Sample data untuk preview
        $sampleData = [
            'nama_pemohon' => 'John Doe',
            'nik' => '1234567890123456',
            'alamat' => 'Jl. Contoh No. 123, RT 01/RW 02, Dusun Contoh',
            'tempat_lahir' => 'Jakarta',
            'tanggal_lahir' => '01 Januari 1990',
            'jenis_kelamin' => 'Laki-laki',
            'agama' => 'Islam',
            'pekerjaan' => 'Wiraswasta',
            'keperluan' => 'Keperluan administrasi',
            'tanggal_surat' => now()->format('d F Y'),
            'nomor_surat' => '001/SK-DESA/I/2025'
        ];

        $previewContent = $templateSurat->replaceVariables($sampleData);
        
        return view('admin.template-surat.preview', compact('templateSurat', 'previewContent'));
    }

    public function duplicate(TemplateSurat $templateSurat)
    {
        $newTemplate = $templateSurat->replicate();
        $newTemplate->nama_template = $templateSurat->nama_template . ' (Copy)';
        $newTemplate->aktif = false;
        $newTemplate->save();

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'DUPLICATE',
            'module' => 'TemplateSurat',
            'table_name' => 'template_surat',
            'record_id' => $newTemplate->id,
            'description' => 'Duplicate template: ' . $templateSurat->nama_template,
            'metadata' => [
                'original_template_id' => $templateSurat->id,
                'new_template_id' => $newTemplate->id
            ]
        ]);

        return redirect()->route('admin.template-surat.edit', $newTemplate)
                        ->with('success', 'Template berhasil diduplikasi');
    }

    public function getByJenis(Request $request)
    {
        $jenisSurat = $request->get('jenis_surat');
        
        $templates = TemplateSurat::aktif()
                                 ->where('jenis_surat', $jenisSurat)
                                 ->get(['id', 'nama_template', 'template_content']);
        
        return response()->json($templates);
    }

    public function getVariables(TemplateSurat $templateSurat)
    {
        $variables = $templateSurat->extractVariables();
        
        return response()->json([
            'variables' => $variables,
            'available_variables' => TemplateSurat::getAvailableVariables()
        ]);
    }

    public function generateSurat(Request $request, TemplateSurat $templateSurat)
    {
        $request->validate([
            'variables' => 'required|array'
        ]);

        $content = $templateSurat->replaceVariables($request->variables);
        
        // Generate nomor surat jika tidak ada
        if (!isset($request->variables['nomor_surat'])) {
            $nomorSurat = $templateSurat->generateNomorSurat();
            $content = str_replace('{{nomor_surat}}', $nomorSurat, $content);
        }

        return response()->json([
            'success' => true,
            'content' => $content,
            'template_name' => $templateSurat->nama_template
        ]);
    }

    public function bulkUpdateStatus(Request $request)
    {
        $request->validate([
            'template_ids' => 'required|array',
            'template_ids.*' => 'exists:template_surat,id',
            'aktif' => 'required|boolean'
        ]);

        $updated = TemplateSurat::whereIn('id', $request->template_ids)
                                ->update(['aktif' => $request->aktif]);

        $status = $request->aktif ? 'aktif' : 'tidak aktif';

        // Log bulk activity
        AuditLog::logActivity([
            'action_type' => 'BULK_UPDATE',
            'module' => 'TemplateSurat',
            'description' => "Bulk update status to {$status} for {$updated} templates",
            'metadata' => [
                'template_ids' => $request->template_ids,
                'new_status' => $status,
                'affected_count' => $updated
            ]
        ]);

        return response()->json([
            'success' => true,
            'message' => "{$updated} template berhasil diperbarui"
        ]);
    }

    public function bulkActivate(Request $request)
    {
        $request->validate([
            'template_ids' => 'required|array',
            'template_ids.*' => 'exists:template_surat,id'
        ]);

        $updated = TemplateSurat::whereIn('id', $request->template_ids)
                                ->update(['aktif' => true]);

        // Log bulk activity
        AuditLog::logActivity([
            'action_type' => 'BULK_ACTIVATE',
            'module' => 'TemplateSurat',
            'description' => "Bulk activate {$updated} templates",
            'metadata' => [
                'template_ids' => $request->template_ids,
                'affected_count' => $updated
            ]
        ]);

        return response()->json([
            'success' => true,
            'message' => "{$updated} template berhasil diaktifkan"
        ]);
    }

    public function bulkDeactivate(Request $request)
    {
        $request->validate([
            'template_ids' => 'required|array',
            'template_ids.*' => 'exists:template_surat,id'
        ]);

        $updated = TemplateSurat::whereIn('id', $request->template_ids)
                                ->update(['aktif' => false]);

        // Log bulk activity
        AuditLog::logActivity([
            'action_type' => 'BULK_DEACTIVATE',
            'module' => 'TemplateSurat',
            'description' => "Bulk deactivate {$updated} templates",
            'metadata' => [
                'template_ids' => $request->template_ids,
                'affected_count' => $updated
            ]
        ]);

        return response()->json([
            'success' => true,
            'message' => "{$updated} template berhasil dinonaktifkan"
        ]);
    }

    // API methods for external access
    public function apiIndex(Request $request)
    {
        $query = TemplateSurat::query();

        if ($request->jenis_surat) {
            $query->where('jenis_surat', 'like', '%' . $request->jenis_surat . '%');
        }

        if ($request->aktif !== null) {
            $query->where('aktif', $request->aktif);
        }

        if ($request->only_active !== null) {
            $query->where('aktif', true);
        }

        $templates = $query->orderBy('nama_template')
                          ->get(['id', 'nama_template', 'jenis_surat', 'template_content', 'aktif']);

        return response()->json([
            'success' => true,
            'data' => $templates
        ]);
    }

    public function apiShow(TemplateSurat $templateSurat)
    {
        return response()->json([
            'success' => true,
            'data' => $templateSurat->only([
                'id', 'nama_template', 'jenis_surat', 'template_content', 'aktif'
            ])
        ]);
    }
}