<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class AdminSessionRefresh
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check if user is authenticated and can access admin
        if (Auth::check() && Auth::user()->canAccessAdmin()) {
            $session = $request->session();
            
            // Check if session is about to expire (within 30 minutes)
            $lastActivity = $session->get('last_activity', time());
            $sessionLifetime = config('session.lifetime', 1440) * 60; // Convert to seconds
            $timeUntilExpiry = $sessionLifetime - (time() - $lastActivity);
            
            // If session expires within 30 minutes, refresh it
            if ($timeUntilExpiry < 1800) { // 30 minutes
                $session->regenerate();
                $session->put('last_activity', time());
                
                Log::info('Admin session refreshed automatically', [
                    'user_id' => Auth::id(),
                    'time_until_expiry' => $timeUntilExpiry,
                    'new_session_id' => $session->getId()
                ]);
            } else {
                // Update last activity
                $session->put('last_activity', time());
            }
        }
        
        return $next($request);
    }
}
