<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class EnsureStorageDirectories
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        try {
            $this->ensureDirectoriesExist();
        } catch (\Exception $e) {
            Log::error('EnsureStorageDirectories error: ' . $e->getMessage());
            // Don't fail the request if storage directories can't be created
        }
        
        return $next($request);
    }

    /**
     * Ensure all required storage directories exist
     */
    private function ensureDirectoriesExist(): void
    {
        $directories = [
            'penduduk',
            'perangkat/foto',
            'perangkat/sk',
            'perangkat/ttd',
            'berita',
            'pengaduan',
            'produk',
            'produk/galeri',
            'wisata',
            'surat',
            'berkas-surat',
            'agenda/banner',
            'agenda/dokumentasi',
            'apbdes/bukti',
            'arsip-surat-keluar',
            'organisasi',
            'templates',
            'bukti_bayar',
            'qrcodes',
            'review',
            'checkin'
        ];

        foreach ($directories as $directory) {
            $this->createDirectoryIfNotExists($directory);
        }
    }

    /**
     * Create directory if it doesn't exist
     */
    private function createDirectoryIfNotExists(string $directory): void
    {
        $fullPath = Storage::disk('public')->path($directory);
        
        if (!file_exists($fullPath)) {
            try {
                mkdir($fullPath, 0755, true);
                
                // Create .gitkeep file to ensure directory is tracked
                $gitkeepPath = $fullPath . '/.gitkeep';
                if (!file_exists($gitkeepPath)) {
                    file_put_contents($gitkeepPath, '');
                }
                
            } catch (\Exception $e) {
                Log::warning("Failed to create storage directory: {$directory}. Error: " . $e->getMessage());
            }
        }
    }
} 