<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Symfony\Component\HttpFoundation\Response;

class SharedHostingStorageSync
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Only sync on successful responses
        if ($response->getStatusCode() === 200) {
            $this->syncStorageFiles();
        }

        return $response;
    }

    /**
     * Sync storage files to public directory for shared hosting
     */
    private function syncStorageFiles()
    {
        $sourcePath = storage_path('app/public');
        $targetPath = public_path('storage');

        if (!File::exists($sourcePath)) {
            return;
        }

        // Ensure target directory exists
        if (!File::exists($targetPath)) {
            File::makeDirectory($targetPath, 0755, true);
        }

        // Get all files in storage
        $files = File::allFiles($sourcePath);
        
        foreach ($files as $file) {
            $relativePath = str_replace($sourcePath . DIRECTORY_SEPARATOR, '', $file->getPathname());
            $targetFile = $targetPath . DIRECTORY_SEPARATOR . $relativePath;
            $targetDir = dirname($targetFile);

            // Create target directory if it doesn't exist
            if (!File::exists($targetDir)) {
                File::makeDirectory($targetDir, 0755, true);
            }

            // Copy file if it doesn't exist or is newer
            if (!File::exists($targetFile) || File::lastModified($file->getPathname()) > File::lastModified($targetFile)) {
                File::copy($file->getPathname(), $targetFile);
            }
        }
    }
}
