<?php

namespace App\Http\Middleware;

use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken as Middleware;

class VerifyCsrfToken extends Middleware
{
    /**
     * The URIs that should be excluded from CSRF verification.
     *
     * @var array<int, string>
     */
    protected $except = [
        // API routes that don't need CSRF protection
        'api/*',
        
        // Webhook endpoints
        'webhook/*',
        
        // Payment callback endpoints
        'payment/*/callback',
        'payment/*/notify',
        
        // Third-party integrations
        'integration/*',
        
        // Admin retry form route
        'admin/retry-form',
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle($request, \Closure $next)
    {
        try {
            return parent::handle($request, $next);
        } catch (\Illuminate\Session\TokenMismatchException $e) {
            // Log the CSRF mismatch for debugging
            \Illuminate\Support\Facades\Log::warning('CSRF token mismatch', [
                'url' => $request->url(),
                'method' => $request->method(),
                'user_id' => auth()->id(),
                'session_id' => $request->session()->getId(),
                'user_agent' => $request->userAgent(),
                'ip' => $request->ip()
            ]);
            
            // Handle CSRF token mismatch more gracefully
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sesi Anda telah berakhir. Silakan refresh halaman dan coba lagi.',
                    'error_type' => 'csrf_mismatch',
                    'requires_refresh' => true,
                    'redirect_url' => '/login'
                ], 419);
            }

            // For admin pages, redirect to login
            if ($request->is('admin*')) {
                return redirect()->route('login')
                    ->withErrors(['csrf' => 'Sesi Anda telah berakhir. Silakan login kembali.']);
            }

            // For other web requests, redirect back with error message
            return redirect()->back()
                ->withErrors(['csrf' => 'Sesi Anda telah berakhir. Silakan refresh halaman dan coba lagi.'])
                ->withInput($request->except(['_token', 'password', 'password_confirmation']));
        }
    }

    /**
     * Determine if the session and input CSRF tokens match.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return bool
     */
    protected function tokensMatch($request)
    {
        $token = $this->getTokenFromRequest($request);

        return is_string($request->session()->token()) &&
               is_string($token) &&
               hash_equals($request->session()->token(), $token);
    }

    /**
     * Get the CSRF token from the request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return string
     */
    protected function getTokenFromRequest($request)
    {
        $token = $request->input('_token') ?: $request->header('X-CSRF-TOKEN');

        if (!$token && $header = $request->header('X-XSRF-TOKEN')) {
            $token = $this->encrypter->decrypt($header, static::serialized());
        }

        return $token;
    }
}

