<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AuditLog extends Model
{
    use HasFactory;

    protected $table = 'audit_log';

    public $timestamps = false; // We only use created_at

    protected $fillable = [
        'user_id',
        'user_name',
        'user_role',
        'action_type',
        'module',
        'table_name',
        'record_id',
        'description',
        'old_values',
        'new_values',
        'metadata',
        'ip_address',
        'user_agent',
        'url',
        'method',
        'status',
        'error_message',
        'tags',
        'created_at'
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
        'metadata' => 'array',
        'tags' => 'array',
        'created_at' => 'datetime'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    // Scope methods
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeByAction($query, $action)
    {
        return $query->where('action_type', $action);
    }

    public function scopeByModule($query, $module)
    {
        return $query->where('module', $module);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeCritical($query)
    {
        return $query->whereJsonContains('tags', 'critical');
    }

    public function scopeDataChanges($query)
    {
        return $query->whereIn('action_type', ['CREATE', 'UPDATE', 'DELETE']);
    }

    public function scopeSecurity($query)
    {
        return $query->whereJsonContains('tags', 'security')
                    ->orWhereIn('action_type', ['LOGIN', 'LOGOUT'])
                    ->orWhere('status', 'FAILED');
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('created_at', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    public function scopeThisMonth($query)
    {
        return $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
    }

    // Static method untuk log aktivitas
    public static function logActivity($data)
    {
        // Auto-fill data dari request jika tidak disediakan
        $defaults = [
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name ?? 'System',
            'user_role' => auth()->user()->role ?? 'Guest',
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'url' => request()->fullUrl(),
            'method' => request()->method(),
            'status' => 'SUCCESS'
        ];

        // Only add created_at if not provided in data
        if (!isset($data['created_at'])) {
            $defaults['created_at'] = now();
        }

        $logData = array_merge($defaults, $data);

        return self::create($logData);
    }

    // Method untuk log login
    public static function logLogin($user, $success = true)
    {
        return self::logActivity([
            'user_id' => $user->id,
            'user_name' => $user->name,
            'user_role' => $user->role,
            'action_type' => 'LOGIN',
            'module' => 'Authentication',
            'description' => $success ? 'User logged in successfully' : 'Failed login attempt',
            'status' => $success ? 'SUCCESS' : 'FAILED',
            'tags' => ['security', 'authentication']
        ]);
    }

    // Method untuk log logout
    public static function logLogout($user)
    {
        return self::logActivity([
            'user_id' => $user->id,
            'user_name' => $user->name,
            'user_role' => $user->role,
            'action_type' => 'LOGOUT',
            'module' => 'Authentication',
            'description' => 'User logged out',
            'tags' => ['security', 'authentication']
        ]);
    }

    // Method untuk log data changes
    public static function logDataChange($action, $model, $oldValues = null, $newValues = null)
    {
        $tableName = $model->getTable();
        $recordId = $model->getKey();
        $modelName = class_basename($model);

        return self::logActivity([
            'action_type' => strtoupper($action),
            'module' => $modelName,
            'table_name' => $tableName,
            'record_id' => $recordId,
            'description' => "{$action} record in {$modelName}",
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'tags' => ['data_change']
        ]);
    }

    // Method untuk log file operations
    public static function logFileOperation($action, $fileName, $module = 'File')
    {
        return self::logActivity([
            'action_type' => strtoupper($action),
            'module' => $module,
            'description' => "{$action} file: {$fileName}",
            'metadata' => ['file_name' => $fileName],
            'tags' => ['file_operation']
        ]);
    }

    // Method untuk log system events
    public static function logSystemEvent($event, $description, $metadata = [])
    {
        return self::logActivity([
            'user_id' => null,
            'user_name' => 'System',
            'user_role' => 'System',
            'action_type' => strtoupper($event),
            'module' => 'System',
            'description' => $description,
            'metadata' => $metadata,
            'tags' => ['system']
        ]);
    }

    // Method untuk get activity summary
    public static function getActivitySummary($period = 'today')
    {
        $query = self::query();

        switch ($period) {
            case 'today':
                $query->today();
                break;
            case 'week':
                $query->thisWeek();
                break;
            case 'month':
                $query->thisMonth();
                break;
        }

        return [
            'total_activities' => $query->count(),
            'by_action' => $query->selectRaw('action_type, COUNT(*) as count')
                                ->groupBy('action_type')
                                ->pluck('count', 'action_type'),
            'by_module' => $query->selectRaw('module, COUNT(*) as count')
                                ->groupBy('module')
                                ->pluck('count', 'module'),
            'by_status' => $query->selectRaw('status, COUNT(*) as count')
                                ->groupBy('status')
                                ->pluck('count', 'status'),
            'failed_activities' => $query->where('status', 'FAILED')->count(),
            'critical_activities' => $query->whereJsonContains('tags', 'critical')->count()
        ];
    }

    // Method untuk get security events
    public static function getSecurityEvents($limit = 50)
    {
        return self::security()
                  ->orderBy('created_at', 'desc')
                  ->limit($limit)
                  ->get();
    }

    // Method untuk cleanup old logs
    public static function cleanup($daysToKeep = 90)
    {
        $cutoffDate = now()->subDays($daysToKeep);
        
        return self::where('created_at', '<', $cutoffDate)->delete();
    }

    // Simple method to create audit log without worrying about created_at
    public static function createLog($data)
    {
        // Remove created_at from data if present, let database handle it
        unset($data['created_at']);
        
        // Auto-fill basic data if not provided
        $defaults = [
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name ?? 'System',
            'user_role' => auth()->user()->role ?? 'Guest',
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'url' => request()->fullUrl(),
            'method' => request()->method(),
            'status' => 'SUCCESS'
        ];

        $logData = array_merge($defaults, $data);

        return self::create($logData);
    }

    // Attribute untuk badge color
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'SUCCESS' => 'success',
            'FAILED' => 'danger',
            'WARNING' => 'warning',
            default => 'secondary'
        };
    }

    public function getActionBadgeColorAttribute()
    {
        return match($this->action_type) {
            'CREATE' => 'success',
            'UPDATE' => 'warning',
            'DELETE' => 'danger',
            'LOGIN' => 'info',
            'LOGOUT' => 'secondary',
            'EXPORT', 'DOWNLOAD' => 'primary',
            default => 'secondary'
        };
    }

    // Method untuk check if critical
    public function isCritical()
    {
        return in_array('critical', $this->tags ?? []);
    }

    // Method untuk check if security related
    public function isSecurityRelated()
    {
        return in_array('security', $this->tags ?? []) ||
               in_array($this->action_type, ['LOGIN', 'LOGOUT']) ||
               $this->status === 'FAILED';
    }
} 