<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;

class Bug extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description', 
        'severity',
        'status',
        'type',
        'module',
        'url',
        'user_agent',
        'ip_address',
        'user_id',
        'error_data',
        'environment_data',
        'error_code',
        'resolved_at',
        'resolved_by',
        'resolution_notes',
        'occurrence_count',
        'last_occurred_at'
    ];

    protected $casts = [
        'error_data' => 'array',
        'environment_data' => 'array',
        'resolved_at' => 'datetime',
        'last_occurred_at' => 'datetime'
    ];

    // Constants
    const SEVERITY_LEVELS = [
        'low' => 'Low',
        'medium' => 'Medium', 
        'high' => 'High',
        'critical' => 'Critical'
    ];

    const STATUS_OPTIONS = [
        'open' => 'Open',
        'in_progress' => 'In Progress',
        'resolved' => 'Resolved',
        'closed' => 'Closed'
    ];

    const TYPE_OPTIONS = [
        'bug' => 'Bug',
        'error' => 'Error',
        'warning' => 'Warning',
        'info' => 'Info'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function resolver()
    {
        return $this->belongsTo(User::class, 'resolved_by');
    }

    // Scopes
    public function scopeOpen($query)
    {
        return $query->where('status', 'open');
    }

    public function scopeCritical($query)
    {
        return $query->where('severity', 'critical');
    }

    public function scopeHigh($query)
    {
        return $query->where('severity', 'high');
    }

    public function scopeByModule($query, $module)
    {
        return $query->where('module', $module);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // Mutators & Accessors
    public function getSeverityBadgeColorAttribute()
    {
        return match($this->severity) {
            'low' => 'success',
            'medium' => 'warning', 
            'high' => 'danger',
            'critical' => 'dark',
            default => 'secondary'
        };
    }

    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'open' => 'danger',
            'in_progress' => 'warning',
            'resolved' => 'success',
            'closed' => 'secondary',
            default => 'secondary'
        };
    }

    public function getTypeBadgeColorAttribute()
    {
        return match($this->type) {
            'bug' => 'danger',
            'error' => 'danger',
            'warning' => 'warning',
            'info' => 'info',
            default => 'secondary'
        };
    }

    // Methods
    public function markAsResolved($notes = null)
    {
        $this->update([
            'status' => 'resolved',
            'resolved_at' => now(),
            'resolved_by' => Auth::id(),
            'resolution_notes' => $notes
        ]);
    }

    public function incrementOccurrence()
    {
        $this->increment('occurrence_count');
        $this->update(['last_occurred_at' => now()]);
    }

    public static function createFromException(\Throwable $exception, $request = null, $module = null)
    {
        $errorData = [
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString(),
            'code' => $exception->getCode()
        ];

        if ($request) {
            $errorData['request'] = [
                'method' => $request->method(),
                'url' => $request->fullUrl(),
                'parameters' => $request->except(['password', 'password_confirmation', '_token']),
                'headers' => $request->headers->all()
            ];
        }

        $environmentData = [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true)
        ];

        // Check if similar bug exists
        $existingBug = self::where('title', 'like', '%' . class_basename($exception) . '%')
                          ->where('module', $module)
                          ->where('status', '!=', 'closed')
                          ->first();

        if ($existingBug) {
            $existingBug->incrementOccurrence();
            return $existingBug;
        }

        return self::create([
            'title' => class_basename($exception) . ' in ' . ($module ?? 'Unknown Module'),
            'description' => $exception->getMessage(),
            'severity' => self::determineSeverity($exception),
            'type' => 'error',
            'module' => $module,
            'url' => $request ? $request->fullUrl() : null,
            'user_agent' => $request ? $request->userAgent() : null,
            'ip_address' => $request ? $request->ip() : null,
            'user_id' => Auth::id(),
            'error_data' => $errorData,
            'environment_data' => $environmentData,
            'error_code' => $exception->getCode(),
            'last_occurred_at' => now()
        ]);
    }

    private static function determineSeverity(\Throwable $exception)
    {
        $criticalErrors = [
            'FatalError',
            'ParseError', 
            'TypeError',
            'Error'
        ];

        $highErrors = [
            'InvalidArgumentException',
            'BadMethodCallException',
            'RuntimeException'
        ];

        $exceptionClass = class_basename($exception);

        if (in_array($exceptionClass, $criticalErrors)) {
            return 'critical';
        }

        if (in_array($exceptionClass, $highErrors)) {
            return 'high';
        }

        return 'medium';
    }

    // Statistics methods
    public static function getStatistics($period = 'week')
    {
        $query = self::query();

        switch ($period) {
            case 'today':
                $query->whereDate('created_at', today());
                break;
            case 'week':
                $query->where('created_at', '>=', now()->subWeek());
                break;
            case 'month':
                $query->where('created_at', '>=', now()->subMonth());
                break;
        }

        return [
            'total' => $query->count(),
            'open' => $query->where('status', 'open')->count(),
            'critical' => $query->where('severity', 'critical')->count(),
            'by_module' => $query->groupBy('module')->selectRaw('module, COUNT(*) as count')->pluck('count', 'module'),
            'by_type' => $query->groupBy('type')->selectRaw('type, COUNT(*) as count')->pluck('count', 'type')
        ];
    }
} 