<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class KegiatanPembangunan extends Model
{
    use HasFactory;

    protected $table = 'kegiatan_pembangunan';

    protected $fillable = [
        'nama_kegiatan',
        'deskripsi',
        'lokasi',
        'alamat_lengkap',
        'latitude',
        'longitude',
        'jenis_kegiatan',
        'sumber_dana',
        'total_anggaran',
        'realisasi_anggaran',
        'sisa_anggaran',
        'tanggal_mulai',
        'tanggal_selesai_rencana',
        'tanggal_selesai_aktual',
        'status',
        'progres_fisik',
        'progres_keuangan',
        'pelaksana_kegiatan',
        'penanggung_jawab',
        'manfaat_kegiatan',
        'kendala',
        'dokumentasi',
        'user_id'
    ];

    protected $casts = [
        'total_anggaran' => 'decimal:2',
        'realisasi_anggaran' => 'decimal:2',
        'sisa_anggaran' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'tanggal_mulai' => 'date',
        'tanggal_selesai_rencana' => 'date',
        'tanggal_selesai_aktual' => 'date',
        'progres_fisik' => 'integer',
        'progres_keuangan' => 'integer',
        'dokumentasi' => 'array'
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->whereIn('status', ['perencanaan', 'berjalan']);
    }

    public function scopeSelesai($query)
    {
        return $query->where('status', 'selesai');
    }

    public function scopeBerjalan($query)
    {
        return $query->where('status', 'berjalan');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_kegiatan', $jenis);
    }

    public function scopeBySumberDana($query, $sumber)
    {
        return $query->where('sumber_dana', $sumber);
    }

    public function scopeByTahun($query, $tahun)
    {
        return $query->whereYear('tanggal_mulai', $tahun);
    }

    // Accessors
    public function getFormattedTotalAnggaranAttribute()
    {
        return 'Rp ' . number_format($this->total_anggaran, 0, ',', '.');
    }

    public function getFormattedRealisasiAnggaranAttribute()
    {
        return 'Rp ' . number_format($this->realisasi_anggaran, 0, ',', '.');
    }

    public function getFormattedSisaAnggaranAttribute()
    {
        return 'Rp ' . number_format($this->sisa_anggaran, 0, ',', '.');
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'perencanaan' => '<span class="badge bg-info">Perencanaan</span>',
            'berjalan' => '<span class="badge bg-primary">Berjalan</span>',
            'selesai' => '<span class="badge bg-success">Selesai</span>',
            'tertunda' => '<span class="badge bg-warning">Tertunda</span>',
            'dibatalkan' => '<span class="badge bg-danger">Dibatalkan</span>'
        ];

        return $badges[$this->status] ?? '';
    }

    public function getProgresBarAttribute()
    {
        $width = $this->progres_fisik;
        $colorClass = 'bg-primary';
        
        if ($width >= 75) {
            $colorClass = 'bg-success';
        } elseif ($width >= 50) {
            $colorClass = 'bg-info';
        } elseif ($width >= 25) {
            $colorClass = 'bg-warning';
        }

        return '<div class="progress">
                    <div class="progress-bar ' . $colorClass . '" style="width: ' . $width . '%">
                        ' . $width . '%
                    </div>
                </div>';
    }

    public function getDurasiHariAttribute()
    {
        if ($this->tanggal_selesai_aktual) {
            return $this->tanggal_mulai->diffInDays($this->tanggal_selesai_aktual);
        }
        return $this->tanggal_mulai->diffInDays($this->tanggal_selesai_rencana);
    }

    public function getPersentaseRealisasiKeuanganAttribute()
    {
        if ($this->total_anggaran > 0) {
            return round(($this->realisasi_anggaran / $this->total_anggaran) * 100, 2);
        }
        return 0;
    }

    public function getIsOnTimeAttribute()
    {
        if ($this->status === 'selesai' && $this->tanggal_selesai_aktual) {
            return $this->tanggal_selesai_aktual <= $this->tanggal_selesai_rencana;
        }
        return now()->toDateString() <= $this->tanggal_selesai_rencana;
    }

    // Methods
    public function updateProgres($fisik = null, $keuangan = null)
    {
        $updates = [];
        
        if ($fisik !== null) {
            $updates['progres_fisik'] = max(0, min(100, $fisik));
        }
        
        if ($keuangan !== null) {
            $updates['progres_keuangan'] = max(0, min(100, $keuangan));
            $updates['realisasi_anggaran'] = ($keuangan / 100) * $this->total_anggaran;
            $updates['sisa_anggaran'] = $this->total_anggaran - $updates['realisasi_anggaran'];
        }
        
        if (!empty($updates)) {
            $this->update($updates);
        }
    }

    public function selesaikan($tanggalSelesai = null)
    {
        $this->update([
            'status' => 'selesai',
            'tanggal_selesai_aktual' => $tanggalSelesai ?? now()->toDateString(),
            'progres_fisik' => 100,
            'progres_keuangan' => $this->persentase_realisasi_keuangan
        ]);
    }

    public function addDokumentasi($filePath)
    {
        $dokumentasi = $this->dokumentasi ?? [];
        $dokumentasi[] = $filePath;
        $this->update(['dokumentasi' => $dokumentasi]);
    }

    public static function getJenisKegiatanOptions()
    {
        return [
            'fisik' => 'Pembangunan Fisik',
            'non_fisik' => 'Pembangunan Non-Fisik',
            'infrastruktur' => 'Infrastruktur',
            'pemberdayaan' => 'Pemberdayaan Masyarakat',
            'kesehatan' => 'Kesehatan',
            'pendidikan' => 'Pendidikan',
            'ekonomi' => 'Ekonomi'
        ];
    }

    public static function getSumberDanaOptions()
    {
        return [
            'dana_desa' => 'Dana Desa',
            'add' => 'Alokasi Dana Desa (ADD)',
            'pad' => 'Pendapatan Asli Desa (PAD)',
            'apbd_kabupaten' => 'APBD Kabupaten',
            'apbd_provinsi' => 'APBD Provinsi',
            'apbn' => 'APBN',
            'swadaya' => 'Swadaya Masyarakat',
            'hibah' => 'Hibah',
            'csr' => 'Corporate Social Responsibility'
        ];
    }

    public static function getStatusOptions()
    {
        return [
            'perencanaan' => 'Perencanaan',
            'berjalan' => 'Sedang Berjalan',
            'selesai' => 'Selesai',
            'tertunda' => 'Tertunda',
            'dibatalkan' => 'Dibatalkan'
        ];
    }

    public static function getTotalAnggaranByJenis($jenis = null)
    {
        $query = self::query();
        if ($jenis) {
            $query->byJenis($jenis);
        }
        return $query->sum('total_anggaran');
    }

    public static function getStatistikProgres()
    {
        return [
            'total_kegiatan' => self::count(),
            'berjalan' => self::berjalan()->count(),
            'selesai' => self::selesai()->count(),
            'rata_rata_progres' => self::avg('progres_fisik'),
            'total_anggaran' => self::sum('total_anggaran'),
            'realisasi_anggaran' => self::sum('realisasi_anggaran')
        ];
    }
} 