<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Polling extends Model
{
    use HasFactory;

    protected $table = 'polling';

    // Constants untuk jenis polling
    const JENIS_POLLING = [
        'Aspirasi' => 'Aspirasi Masyarakat',
        'Voting' => 'Voting/Pemilihan',
        'Survey' => 'Survey Pendapat',
        'Polling' => 'Polling Umum'
    ];

    const TARGET_PESERTA = [
        'Semua Warga' => 'Seluruh Warga Desa',
        'RT Tertentu' => 'RT Tertentu',
        'RW Tertentu' => 'RW Tertentu', 
        'Dusun Tertentu' => 'Dusun Tertentu',
        'Kelompok Usia' => 'Kelompok Usia Tertentu'
    ];

    protected $fillable = [
        'judul',
        'deskripsi',
        'jenis',
        'mulai_voting',
        'selesai_voting',
        'multi_pilihan',
        'anonim',
        'target_voter',
        'filter_voter',
        'status',
        'total_voter',
        'total_suara',
        'banner',
        'hasil_keputusan',
        'created_by'
    ];

    protected $casts = [
        'mulai_voting' => 'datetime',
        'selesai_voting' => 'datetime',
        'multi_pilihan' => 'boolean',
        'anonim' => 'boolean',
        'filter_voter' => 'array'
    ];

    // Relationships
    public function opsi()
    {
        return $this->hasMany(PollingOpsi::class);
    }

    public function votes()
    {
        return $this->hasMany(PollingVote::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scope methods
    public function scopeAktif($query)
    {
        return $query->where('status', 'Aktif');
    }

    public function scopeSedangBerlangsung($query)
    {
        return $query->where('status', 'Aktif')
                    ->where('mulai_voting', '<=', now())
                    ->where('selesai_voting', '>=', now());
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis', $jenis);
    }

    // Method untuk cek apakah user bisa vote
    public function canUserVote($nik = null)
    {
        // Cek status polling
        if ($this->status !== 'Aktif') {
            return false;
        }

        // Cek waktu voting
        if (now()->isBefore($this->mulai_voting) || now()->isAfter($this->selesai_voting)) {
            return false;
        }

        // Cek apakah sudah vote (jika tidak anonim)
        if (!$this->anonim && $nik) {
            $existingVote = $this->votes()->where('nik', $nik)->exists();
            if ($existingVote) {
                return false;
            }
        }

        // Cek filter target voter
        if ($this->target_voter !== 'Semua Warga' && $nik) {
            return $this->checkTargetVoter($nik);
        }

        return true;
    }

    // Method untuk validasi target voter
    private function checkTargetVoter($nik)
    {
        if (!$this->filter_voter) {
            return true;
        }

        $penduduk = Penduduk::where('nik', $nik)->first();
        if (!$penduduk) {
            return false;
        }

        switch ($this->target_voter) {
            case 'RT Tertentu':
                return in_array($penduduk->rt, $this->filter_voter);
            
            case 'RW Tertentu':
                return in_array($penduduk->rw, $this->filter_voter);
            
            case 'Dusun Tertentu':
                return in_array($penduduk->dusun, $this->filter_voter);
            
            case 'Kelompok Usia':
                $umur = $penduduk->umur;
                $minUmur = $this->filter_voter['min_umur'] ?? 0;
                $maxUmur = $this->filter_voter['max_umur'] ?? 100;
                return $umur >= $minUmur && $umur <= $maxUmur;
            
            default:
                return true;
        }
    }

    // Method untuk submit vote
    public function submitVote($opsiId, $nik = null, $namaVoter = null, $alasan = null, $ipAddress = null)
    {
        if (!$this->canUserVote($nik)) {
            throw new \Exception('Anda tidak dapat memberikan suara pada polling ini.');
        }

        $opsi = $this->opsi()->find($opsiId);
        if (!$opsi) {
            throw new \Exception('Opsi voting tidak valid.');
        }

        // Simpan vote
        $vote = PollingVote::create([
            'polling_id' => $this->id,
            'opsi_id' => $opsiId,
            'nik' => $this->anonim ? null : $nik,
            'nama_voter' => $this->anonim ? null : $namaVoter,
            'alasan' => $alasan,
            'ip_address' => $ipAddress ?: request()->ip()
        ]);

        // Update counter
        $opsi->increment('jumlah_suara');
        $this->increment('total_suara');

        return $vote;
    }

    // Method untuk get hasil voting
    public function getHasilVoting()
    {
        $hasil = $this->opsi()->with('votes')->get()->map(function($opsi) {
            $persentase = $this->total_suara > 0 ? ($opsi->jumlah_suara / $this->total_suara) * 100 : 0;
            
            return [
                'id' => $opsi->id,
                'judul' => $opsi->judul_opsi,
                'deskripsi' => $opsi->deskripsi_opsi,
                'jumlah_suara' => $opsi->jumlah_suara,
                'persentase' => round($persentase, 2),
                'estimasi_biaya' => $opsi->estimasi_biaya
            ];
        });

        return $hasil->sortByDesc('jumlah_suara')->values();
    }

    // Method untuk auto close polling jika sudah selesai
    public function autoUpdateStatus()
    {
        if ($this->status === 'Aktif' && now()->isAfter($this->selesai_voting)) {
            $this->update(['status' => 'Selesai']);
        }
    }

    // Attribute untuk status badge
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'Draft' => 'secondary',
            'Aktif' => 'primary',
            'Selesai' => 'success',
            'Ditutup' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk cek apakah sedang berlangsung
    public function isSedangBerlangsung()
    {
        return $this->status === 'Aktif' && 
               now()->isBetween($this->mulai_voting, $this->selesai_voting);
    }

    // Method untuk durasi voting
    public function getDurasiVoting()
    {
        return $this->selesai_voting->diff($this->mulai_voting);
    }
} 