<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TemplateSurat extends Model
{
    use HasFactory;

    protected $table = 'template_surat';

    // Constants untuk jenis template
    const JENIS_TEMPLATE = [
        'Surat Keterangan Domisili',
        'Surat Keterangan Kelahiran',
        'Surat Keterangan Kematian',
        'Surat Keterangan Usaha',
        'Surat Keterangan Penghasilan',
        'Surat Keterangan Catatan Kepolisian',
        'Surat Pengantar Nikah',
        'Surat Keterangan Tidak Mampu',
        'Surat Rekomendasi',
        'Surat Undangan',
        'Surat Edaran',
        'Surat Keputusan'
    ];

    const KATEGORI_TEMPLATE = [
        'Umum' => 'Template untuk keperluan umum',
        'Khusus' => 'Template untuk keperluan khusus',
        'Resmi' => 'Template untuk keperluan resmi'
    ];

    protected $fillable = [
        'nama_template',
        'jenis_surat',
        'kode_surat',
        'format_nomor',
        'template_content',
        'variabel_tersedia',
        'penandatangan_default',
        'aktif',
        'urutan',
        'keterangan'
    ];

    protected $casts = [
        'variabel_tersedia' => 'array',
        'aktif' => 'boolean'
    ];

    // Scope untuk template aktif
    public function scopeAktif($query)
    {
        return $query->where('aktif', true);
    }

    // Scope berdasarkan jenis surat
    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_surat', $jenis);
    }

    // Method untuk replace variabel dalam template
    public function replaceVariables($data)
    {
        $content = $this->template_content;
        
        // Daftar variabel yang bisa digunakan
        $variables = [
            '{nama}' => $data['nama'] ?? '',
            '{nik}' => $data['nik'] ?? '',
            '{alamat}' => $data['alamat'] ?? '',
            '{tempat_lahir}' => $data['tempat_lahir'] ?? '',
            '{tanggal_lahir}' => $data['tanggal_lahir'] ?? '',
            '{jenis_kelamin}' => $data['jenis_kelamin'] ?? '',
            '{agama}' => $data['agama'] ?? '',
            '{pekerjaan}' => $data['pekerjaan'] ?? '',
            '{no_kk}' => $data['no_kk'] ?? '',
            '{dusun}' => $data['dusun'] ?? '',
            '{rt}' => $data['rt'] ?? '',
            '{rw}' => $data['rw'] ?? '',
            '{tanggal_sekarang}' => now()->format('d F Y'),
            '{keperluan}' => $data['keperluan'] ?? '',
            '{berlaku_sampai}' => $data['berlaku_sampai'] ?? ''
        ];

        // Replace semua variabel
        foreach ($variables as $placeholder => $value) {
            $content = str_replace($placeholder, $value, $content);
        }

        return $content;
    }

    // Method untuk generate nomor surat
    public function generateNomorSurat($counter = null)
    {
        if (!$counter) {
            $counter = PelayananSurat::where('jenis_surat', $this->jenis_surat)
                                   ->whereYear('created_at', now()->year)
                                   ->count() + 1;
        }

        $format = $this->format_nomor;
        $variables = [
            '{counter}' => str_pad($counter, 3, '0', STR_PAD_LEFT),
            '{kode}' => $this->kode_surat,
            '{bulan}' => now()->format('m'),
            '{tahun}' => now()->format('Y'),
            '{tahun_pendek}' => now()->format('y')
        ];

        foreach ($variables as $placeholder => $value) {
            $format = str_replace($placeholder, $value, $format);
        }

        return $format;
    }

    // Method untuk extract variables dari template
    public function extractVariables()
    {
        preg_match_all('/\{\{([^}]+)\}\}/', $this->template_content, $matches);
        return array_unique($matches[1]);
    }

    // Method untuk get available variables
    public static function getAvailableVariables()
    {
        return [
            'nama_pemohon' => 'Nama Pemohon',
            'nik' => 'NIK',
            'alamat' => 'Alamat',
            'tempat_lahir' => 'Tempat Lahir',
            'tanggal_lahir' => 'Tanggal Lahir',
            'jenis_kelamin' => 'Jenis Kelamin',
            'agama' => 'Agama',
            'pekerjaan' => 'Pekerjaan',
            'status_perkawinan' => 'Status Perkawinan',
            'rt' => 'RT',
            'rw' => 'RW',
            'dusun' => 'Dusun',
            'keperluan' => 'Keperluan',
            'tanggal_surat' => 'Tanggal Surat',
            'nomor_surat' => 'Nomor Surat'
        ];
    }

    // Relationship dengan pelayanan surat
    public function pelayananSurat()
    {
        return $this->hasMany(PelayananSurat::class, 'jenis_surat', 'jenis_surat');
    }
} 