<?php

namespace App\Services;

use Gregwar\Captcha\CaptchaBuilder;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class CaptchaService
{
    /**
     * Generate CAPTCHA image and store phrase in session
     * 
     * @return array
     */
    public static function generate()
    {
        $builder = new CaptchaBuilder();
        $builder->build();
        
        // Store CAPTCHA phrase in session
        Session::put('captcha_phrase', $builder->getPhrase());
        
        return [
            'image' => $builder->inline()
        ];
    }
    
    /**
     * Validate CAPTCHA input
     * 
     * @param string $input
     * @return bool
     */
    public static function validate($input)
    {
        $phrase = session('captcha_phrase');
        
        if (!$phrase) {
            Log::warning('CAPTCHA Validation Failed - No phrase in session');
            return false;
        }
        
        // Case-insensitive comparison
        $isValid = strtolower($input) === strtolower($phrase);
        
        Log::info('CAPTCHA Validation', [
            'input' => $input,
            'phrase' => $phrase,
            'valid' => $isValid
        ]);
        
        // Clear captcha from session after validation
        session()->forget('captcha_phrase');
        
        return $isValid;
    }
    
    /**
     * Generate CAPTCHA for API endpoint
     * 
     * @return \Illuminate\Http\Response
     */
    public static function getCaptchaImage()
    {
        try {
            $builder = new CaptchaBuilder();
            $builder->build();
            
            // Store CAPTCHA phrase in session
            Session::put('captcha_phrase', $builder->getPhrase());
            
            Log::info('CAPTCHA Generated', ['phrase' => $builder->getPhrase()]);
            
            return response($builder->get(), 200, [
                'Content-Type' => 'image/jpeg',
                'Cache-Control' => 'no-cache, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0'
            ]);
        } catch (\Exception $e) {
            Log::error('CAPTCHA Generation Failed', ['error' => $e->getMessage()]);
            throw $e;
        }
    }
}
