<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class DukcapilService
{
    protected $apiKey;
    protected $baseUrl;
    protected $timeout;

    public function __construct()
    {
        $this->apiKey = config('services.dukcapil.api_key');
        $this->baseUrl = config('services.dukcapil.base_url', 'https://apimws.dukcapil.kemendagri.go.id');
        $this->timeout = config('services.dukcapil.timeout', 30);
    }

    /**
     * Verifikasi NIK via API Dukcapil
     */
    public function verifyNik($nik)
    {
        try {
            // Validasi format NIK
            if (!$this->isValidNikFormat($nik)) {
                return [
                    'success' => false,
                    'message' => 'Format NIK tidak valid',
                    'data' => null
                ];
            }

            // Check cache first
            $cacheKey = 'dukcapil_nik_' . $nik;
            if (Cache::has($cacheKey)) {
                $cachedData = Cache::get($cacheKey);
                Log::info('NIK verification from cache', ['nik' => $nik]);
                return $cachedData;
            }

            // Call Dukcapil API
            $response = Http::timeout($this->timeout)
                ->withHeaders([
                    'X-API-KEY' => $this->apiKey,
                    'Content-Type' => 'application/json'
                ])
                ->post($this->baseUrl . '/api/v1/verify-nik', [
                    'nik' => $nik
                ]);

            if ($response->successful()) {
                $data = $response->json();
                
                if ($data['status'] === 'success' && isset($data['data'])) {
                    $result = [
                        'success' => true,
                        'message' => 'NIK valid dan terdaftar di Dukcapil',
                        'data' => [
                            'nik' => $data['data']['nik'],
                            'nama' => $data['data']['nama_lengkap'],
                            'tempat_lahir' => $data['data']['tempat_lahir'],
                            'tanggal_lahir' => $data['data']['tanggal_lahir'],
                            'jenis_kelamin' => $data['data']['jenis_kelamin'],
                            'alamat' => $data['data']['alamat'],
                            'rt' => $data['data']['rt'] ?? null,
                            'rw' => $data['data']['rw'] ?? null,
                            'provinsi' => $data['data']['provinsi'],
                            'kabupaten' => $data['data']['kabupaten'],
                            'kecamatan' => $data['data']['kecamatan'],
                            'kelurahan' => $data['data']['kelurahan'],
                            'agama' => $data['data']['agama'] ?? null,
                            'status_kawin' => $data['data']['status_kawin'] ?? null,
                            'pekerjaan' => $data['data']['pekerjaan'] ?? null,
                            'verified_at' => now()
                        ]
                    ];

                    // Cache for 24 hours
                    Cache::put($cacheKey, $result, now()->addHours(24));
                    
                    Log::info('NIK verification successful', ['nik' => $nik]);
                    return $result;
                } else {
                    $result = [
                        'success' => false,
                        'message' => $data['message'] ?? 'NIK tidak terdaftar di Dukcapil',
                        'data' => null
                    ];
                    
                    // Cache negative result for 1 hour
                    Cache::put($cacheKey, $result, now()->addHour());
                    return $result;
                }
            }

            Log::error('Dukcapil API Error', [
                'nik' => $nik,
                'status' => $response->status(),
                'response' => $response->body()
            ]);

            return [
                'success' => false,
                'message' => 'Gagal terhubung ke layanan Dukcapil',
                'data' => null
            ];

        } catch (\Exception $e) {
            Log::error('Dukcapil Service Exception', [
                'nik' => $nik,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat verifikasi NIK',
                'data' => null,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Verifikasi format NIK
     */
    protected function isValidNikFormat($nik)
    {
        // NIK harus 16 digit
        if (strlen($nik) !== 16 || !is_numeric($nik)) {
            return false;
        }

        // Validasi kode wilayah (6 digit pertama)
        $kodeWilayah = substr($nik, 0, 6);
        
        // Validasi tanggal lahir (6 digit: DDMMYY)
        $tanggalLahir = substr($nik, 6, 6);
        $tanggal = (int)substr($tanggalLahir, 0, 2);
        $bulan = (int)substr($tanggalLahir, 2, 2);
        $tahun = (int)substr($tanggalLahir, 4, 2);

        // Untuk perempuan, tanggal ditambah 40
        if ($tanggal > 40) {
            $tanggal -= 40;
        }

        // Validasi tanggal
        if ($tanggal < 1 || $tanggal > 31 || $bulan < 1 || $bulan > 12) {
            return false;
        }

        return true;
    }

    /**
     * Get status layanan Dukcapil
     */
    public function getServiceStatus()
    {
        try {
            $response = Http::timeout(10)
                ->withHeaders([
                    'X-API-KEY' => $this->apiKey
                ])
                ->get($this->baseUrl . '/api/v1/status');

            if ($response->successful()) {
                return [
                    'success' => true,
                    'status' => 'online',
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'status' => 'offline',
                'message' => 'Service unavailable'
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Get quotas penggunaan API
     */
    public function getQuotaUsage()
    {
        try {
            $response = Http::timeout(10)
                ->withHeaders([
                    'X-API-KEY' => $this->apiKey
                ])
                ->get($this->baseUrl . '/api/v1/quota');

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to get quota information'
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Bulk verify NIK
     */
    public function bulkVerifyNik(array $nikList)
    {
        $results = [];
        $batchSize = 50; // Process in batches

        $batches = array_chunk($nikList, $batchSize);

        foreach ($batches as $batch) {
            try {
                $response = Http::timeout($this->timeout)
                    ->withHeaders([
                        'X-API-KEY' => $this->apiKey,
                        'Content-Type' => 'application/json'
                    ])
                    ->post($this->baseUrl . '/api/v1/bulk-verify-nik', [
                        'nik_list' => $batch
                    ]);

                if ($response->successful()) {
                    $batchResults = $response->json();
                    $results = array_merge($results, $batchResults['data'] ?? []);
                } else {
                    // If batch fails, process individually
                    foreach ($batch as $nik) {
                        $results[] = $this->verifyNik($nik);
                        sleep(1); // Rate limiting
                    }
                }

                // Rate limiting between batches
                sleep(2);

            } catch (\Exception $e) {
                Log::error('Bulk NIK verification failed', [
                    'batch' => $batch,
                    'error' => $e->getMessage()
                ]);

                // Fallback to individual verification
                foreach ($batch as $nik) {
                    $results[] = $this->verifyNik($nik);
                }
            }
        }

        return $results;
    }

    /**
     * Clear cache for specific NIK
     */
    public function clearNikCache($nik)
    {
        $cacheKey = 'dukcapil_nik_' . $nik;
        Cache::forget($cacheKey);
    }

    /**
     * Clear all NIK verification cache
     */
    public function clearAllCache()
    {
        Cache::flush();
    }
} 