<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Exception;
// use Intervention\Image\Facades\Image;

class FileUploadService
{
    /**
     * Upload single file with image resize for photos
     */
    public function uploadSingle(
        UploadedFile $file, 
        string $directory, 
        ?string $filename = null,
        string $disk = 'public',
        array $resizeOptions = []
    ): array {
        try {
            // Validate file
            $this->validateFile($file);
            
            // Generate filename if not provided
            if (!$filename) {
                $filename = $this->generateFilename($file);
            }
            
            // Ensure directory exists
            $this->ensureDirectoryExists($directory, $disk);
            
            // Handle image resize if it's an image and resize options are provided
            if (str_starts_with($file->getMimeType(), 'image/') && !empty($resizeOptions)) {
                $path = $this->uploadAndResizeImage($file, $directory, $filename, $disk, $resizeOptions);
            } else {
                // Store file normally
                $path = $file->storeAs($directory, $filename, $disk);
            }
            
            if (!$path) {
                throw new Exception('Failed to store file');
            }
            
            return [
                'success' => true,
                'path' => $path,
                'filename' => $filename,
                'original_name' => $file->getClientOriginalName(),
                'size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'url' => $disk === 'public' ? asset('storage/' . $path) : null
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Upload multiple files
     */
    public function uploadMultiple(
        array $files, 
        string $directory, 
        string $disk = 'public'
    ): array {
        $results = [];
        $successful = 0;
        $failed = 0;
        
        foreach ($files as $file) {
            if ($file instanceof UploadedFile) {
                $result = $this->uploadSingle($file, $directory, null, $disk);
                $results[] = $result;
                
                if ($result['success']) {
                    $successful++;
                } else {
                    $failed++;
                }
            }
        }
        
        return [
            'results' => $results,
            'summary' => [
                'total' => count($files),
                'successful' => $successful,
                'failed' => $failed
            ]
        ];
    }
    
    /**
     * Delete file
     */
    public function deleteFile(string $path, string $disk = 'public'): bool
    {
        try {
            if (Storage::disk($disk)->exists($path)) {
                return Storage::disk($disk)->delete($path);
            }
            return true; // File doesn't exist, consider as deleted
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Delete multiple files
     */
    public function deleteMultiple(array $paths, string $disk = 'public'): array
    {
        $results = [];
        foreach ($paths as $path) {
            $results[$path] = $this->deleteFile($path, $disk);
        }
        return $results;
    }
    
    /**
     * Validate uploaded file
     */
    private function validateFile(UploadedFile $file): void
    {
        // Check if file is valid
        if (!$file->isValid()) {
            throw new Exception('Invalid file upload');
        }
        
        // Check file size (default max 2MB)
        $maxSize = config('app.max_upload_size', 2048) * 1024; // Convert KB to bytes
        if ($file->getSize() > $maxSize) {
            throw new Exception('File size exceeds maximum allowed size');
        }
        
        // Check if it's an image for image uploads
        if (str_starts_with($file->getMimeType(), 'image/')) {
            $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
            if (!in_array($file->getMimeType(), $allowedMimes)) {
                throw new Exception('Invalid image format. Allowed: JPEG, PNG, JPG, GIF, WebP');
            }
            
            // Additional security checks for images
            $this->validateImageSecurity($file);
        }
        
        // Check for dangerous file extensions
        $this->validateFileExtension($file);
    }
    
    /**
     * Additional security validation for images
     */
    private function validateImageSecurity(UploadedFile $file): void
    {
        try {
            // Check file signature (magic bytes) matches MIME type
            if (function_exists('finfo_open')) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $detectedMime = finfo_file($finfo, $file->getPathname());
                finfo_close($finfo);
                
                if ($detectedMime !== $file->getMimeType()) {
                    throw new Exception('File type mismatch detected');
                }
            }
            
            // Scan for suspicious content in image files
            $content = file_get_contents($file->getPathname());
            if ($content !== false) {
                $this->scanForMaliciousContent($content);
            }
            
            // Validate image dimensions and properties
            $imageInfo = getimagesize($file->getPathname());
            if ($imageInfo === false) {
                throw new Exception('Invalid image file');
            }
            
            // Check for reasonable image dimensions (prevent DoS attacks)
            // Allow up to configured maximum pixels for modern high-resolution photos
            // The image will be resized to 400x400 anyway
            $maxDimensions = config('app.max_image_dimensions', 8000);
            if ($imageInfo[0] > $maxDimensions || $imageInfo[1] > $maxDimensions) {
                throw new Exception("Image dimensions too large. Maximum allowed: {$maxDimensions}x{$maxDimensions} pixels");
            }
            
            // Also check for extremely small images
            if ($imageInfo[0] < 10 || $imageInfo[1] < 10) {
                throw new Exception('Image dimensions too small. Minimum: 10x10 pixels');
            }
            
        } catch (Exception $e) {
            throw new Exception('Image validation failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Validate file extension against whitelist
     */
    private function validateFileExtension(UploadedFile $file): void
    {
        $extension = strtolower($file->getClientOriginalExtension());
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx'];
        
        if (!in_array($extension, $allowedExtensions)) {
            throw new Exception('File extension not allowed: ' . $extension);
        }
    }
    
    /**
     * Scan file content for malicious patterns
     */
    private function scanForMaliciousContent(string $content): void
    {
        $maliciousPatterns = [
            '/<\?php/i',
            '/<script/i',
            '/eval\s*\(/i',
            '/base64_decode\s*\(/i',
            '/exec\s*\(/i',
            '/shell_exec\s*\(/i',
            '/system\s*\(/i',
            '/passthru\s*\(/i',
            '/file_get_contents\s*\(/i',
            '/file_put_contents\s*\(/i',
            '/\$_GET/i',
            '/\$_POST/i',
            '/\$_REQUEST/i',
            '/\$_COOKIE/i',
            '/\$_SESSION/i'
        ];
        
        foreach ($maliciousPatterns as $pattern) {
            if (preg_match($pattern, $content)) {
                throw new Exception('Potentially malicious content detected in file');
            }
        }
    }
    
    /**
     * Generate unique filename
     */
    private function generateFilename(UploadedFile $file): string
    {
        $timestamp = now()->format('YmdHis');
        $random = Str::random(8);
        $extension = $file->getClientOriginalExtension();
        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        
        // Clean original name
        $cleanName = Str::slug($originalName);
        if (strlen($cleanName) > 20) {
            $cleanName = substr($cleanName, 0, 20);
        }
        
        return "{$timestamp}_{$random}_{$cleanName}.{$extension}";
    }
    
    /**
     * Ensure directory exists
     */
    private function ensureDirectoryExists(string $directory, string $disk = 'public'): void
    {
        $fullPath = Storage::disk($disk)->path($directory);
        if (!file_exists($fullPath)) {
            mkdir($fullPath, 0755, true);
        }
    }
    
    /**
     * Get file info
     */
    public function getFileInfo(string $path, string $disk = 'public'): ?array
    {
        try {
            if (!Storage::disk($disk)->exists($path)) {
                return null;
            }
            
            return [
                'path' => $path,
                'size' => Storage::disk($disk)->size($path),
                'last_modified' => Storage::disk($disk)->lastModified($path),
                'url' => $disk === 'public' ? asset('storage/' . $path) : null,
                'exists' => true
            ];
        } catch (Exception $e) {
            return null;
        }
    }
    
    /**
     * Move uploaded file with better error handling
     */
    public function moveUploadedFile(
        UploadedFile $file, 
        string $directory, 
        ?string $filename = null,
        string $disk = 'public'
    ): array {
        try {
            // Create directory if it doesn't exist
            $storagePath = Storage::disk($disk)->path($directory);
            if (!file_exists($storagePath)) {
                if (!mkdir($storagePath, 0755, true)) {
                    throw new Exception("Failed to create directory: {$directory}");
                }
            }
            
            // Generate filename if not provided
            if (!$filename) {
                $filename = $this->generateFilename($file);
            }
            
            // Move file
            $targetPath = $storagePath . DIRECTORY_SEPARATOR . $filename;
            if (!$file->move($storagePath, $filename)) {
                throw new Exception("Failed to move uploaded file");
            }
            
            // Verify file was moved successfully
            if (!file_exists($targetPath)) {
                throw new Exception("File was not properly saved");
            }
            
            $relativePath = $directory . '/' . $filename;
            
            return [
                'success' => true,
                'path' => $relativePath,
                'filename' => $filename,
                'full_path' => $targetPath,
                'url' => $disk === 'public' ? asset('storage/' . $relativePath) : null,
                'size' => filesize($targetPath)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Upload and resize image specifically for photos
     */
    public function uploadPhotoWithResize(
        UploadedFile $file, 
        string $directory, 
        ?string $filename = null,
        string $disk = 'public'
    ): array {
        $resizeOptions = [
            'width' => 400,
            'height' => 400,
            'quality' => 85,
            'maintain_aspect_ratio' => true
        ];
        
        return $this->uploadSingle($file, $directory, $filename, $disk, $resizeOptions);
    }

    /**
     * Upload and resize image using GD extension (fallback for Intervention Image)
     */
    private function uploadAndResizeImage(
        UploadedFile $file, 
        string $directory, 
        string $filename, 
        string $disk,
        array $options
    ): string {
        try {
            // First upload the original file
            $path = $file->storeAs($directory, $filename, $disk);
            
            // Try to resize using GD if available
            if (extension_loaded('gd') && str_starts_with($file->getMimeType(), 'image/')) {
                $this->resizeImageWithGD($path, $disk, $options);
            }
            
            return $path;
            
        } catch (Exception $e) {
            // Fallback to normal upload if anything fails
            Log::warning('Image upload/resize failed: ' . $e->getMessage());
            // Still return the path as the file was uploaded
            return $file->storeAs($directory, $filename, $disk);
        }
    }

    /**
     * Resize image using GD extension
     */
    private function resizeImageWithGD(string $path, string $disk, array $options): void
    {
        try {
            $width = $options['width'] ?? 400;
            $height = $options['height'] ?? 400;
            $quality = $options['quality'] ?? 85;
            
            $fullPath = Storage::disk($disk)->path($path);
            
            if (!file_exists($fullPath)) {
                return;
            }
            
            // Get image info
            $imageInfo = getimagesize($fullPath);
            if ($imageInfo === false) {
                return;
            }
            
            // Create image resource based on type
            $sourceImage = null;
            switch ($imageInfo[2]) {
                case IMAGETYPE_JPEG:
                    $sourceImage = imagecreatefromjpeg($fullPath);
                    break;
                case IMAGETYPE_PNG:
                    $sourceImage = imagecreatefrompng($fullPath);
                    break;
                case IMAGETYPE_GIF:
                    $sourceImage = imagecreatefromgif($fullPath);
                    break;
                default:
                    return; // Unsupported format
            }
            
            if (!$sourceImage) {
                return;
            }
            
            // Calculate new dimensions maintaining aspect ratio
            $originalWidth = $imageInfo[0];
            $originalHeight = $imageInfo[1];
            
            $ratio = min($width / $originalWidth, $height / $originalHeight);
            $newWidth = round($originalWidth * $ratio);
            $newHeight = round($originalHeight * $ratio);
            
            // Create new image
            $newImage = imagecreatetruecolor($newWidth, $newHeight);
            
            // Preserve transparency for PNG
            if ($imageInfo[2] == IMAGETYPE_PNG) {
                imagealphablending($newImage, false);
                imagesavealpha($newImage, true);
                $transparent = imagecolorallocatealpha($newImage, 255, 255, 255, 127);
                imagefill($newImage, 0, 0, $transparent);
            }
            
            // Resize
            imagecopyresampled(
                $newImage, $sourceImage,
                0, 0, 0, 0,
                $newWidth, $newHeight,
                $originalWidth, $originalHeight
            );
            
            // Save resized image
            switch ($imageInfo[2]) {
                case IMAGETYPE_JPEG:
                    imagejpeg($newImage, $fullPath, $quality);
                    break;
                case IMAGETYPE_PNG:
                    imagepng($newImage, $fullPath);
                    break;
                case IMAGETYPE_GIF:
                    imagegif($newImage, $fullPath);
                    break;
            }
            
            // Clean up memory
            imagedestroy($sourceImage);
            imagedestroy($newImage);
            
        } catch (Exception $e) {
            Log::warning('GD image resize failed: ' . $e->getMessage());
        }
    }
} 