<?php

namespace App\Services;

class InputSanitizer
{
    /**
     * Sanitize HTML content while preserving safe tags
     */
    public static function sanitizeHtml(string $content): string
    {
        // Allow only safe HTML tags
        $allowedTags = '<p><br><strong><em><u><ol><ul><li><h1><h2><h3><h4><h5><h6>';
        
        // Strip potentially dangerous tags
        $cleaned = strip_tags($content, $allowedTags);
        
        // Remove potentially dangerous attributes
        $cleaned = preg_replace('/on\w+="[^"]*"/i', '', $cleaned);
        $cleaned = preg_replace('/on\w+=\'[^\']*\'/i', '', $cleaned);
        $cleaned = preg_replace('/javascript:/i', '', $cleaned);
        $cleaned = preg_replace('/vbscript:/i', '', $cleaned);
        $cleaned = preg_replace('/data:/i', '', $cleaned);
        
        return trim($cleaned);
    }
    
    /**
     * Sanitize user input for safe database storage
     */
    public static function sanitizeInput(string $input): string
    {
        // Remove null bytes
        $input = str_replace(chr(0), '', $input);
        
        // Trim whitespace
        $input = trim($input);
        
        // Convert special characters to HTML entities
        $input = htmlspecialchars($input, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        return $input;
    }
    
    /**
     * Validate and sanitize NIK (Indonesian National ID)
     */
    public static function sanitizeNik(string $nik): string
    {
        // Remove any non-digit characters
        $nik = preg_replace('/[^0-9]/', '', $nik);
        
        // Ensure exactly 16 digits
        if (strlen($nik) !== 16) {
            throw new \InvalidArgumentException('NIK harus terdiri dari 16 digit');
        }
        
        return $nik;
    }
    
    /**
     * Sanitize phone number
     */
    public static function sanitizePhone(string $phone): string
    {
        // Remove any non-digit characters except + and -
        $phone = preg_replace('/[^0-9+\-]/', '', $phone);
        
        // Convert Indonesian format
        if (str_starts_with($phone, '08')) {
            $phone = '+62' . substr($phone, 1);
        }
        
        return $phone;
    }
    
    /**
     * Sanitize email
     */
    public static function sanitizeEmail(string $email): string
    {
        $email = trim(strtolower($email));
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new \InvalidArgumentException('Format email tidak valid');
        }
        
        return $email;
    }
    
    /**
     * Remove potentially dangerous file extensions from filename
     */
    public static function sanitizeFilename(string $filename): string
    {
        // Remove path traversal attempts
        $filename = basename($filename);
        
        // Remove dangerous characters
        $filename = preg_replace('/[^a-zA-Z0-9._\-]/', '', $filename);
        
        // Prevent double extensions and dangerous extensions
        $dangerousExtensions = [
            'php', 'php3', 'php4', 'php5', 'phtml', 'asp', 'aspx', 
            'jsp', 'js', 'vbs', 'exe', 'bat', 'com', 'scr', 'sh'
        ];
        
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if (in_array($extension, $dangerousExtensions)) {
            $filename = pathinfo($filename, PATHINFO_FILENAME) . '_safe.txt';
        }
        
        return $filename;
    }
}
