<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class TripayService
{
    private $apiKey;
    private $privateKey;
    private $merchantCode;
    private $baseUrl;
    private $isProduction;

    public function __construct()
    {
        $this->apiKey = config('tripay.api_key');
        $this->privateKey = config('tripay.private_key');
        $this->merchantCode = config('tripay.merchant_code');
        $this->isProduction = config('tripay.is_production', false);
        $this->baseUrl = $this->isProduction 
            ? 'https://tripay.co.id/api' 
            : 'https://tripay.co.id/api-sandbox';
    }

    /**
     * Get available payment channels
     */
    public function getPaymentChannels()
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->get($this->baseUrl . '/merchant/payment-channel');

            if ($response->successful()) {
                return $response->json();
            }

            return ['success' => false, 'message' => 'Failed to get payment channels'];
        } catch (\Exception $e) {
            Log::error('Tripay get payment channels error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Create payment transaction
     */
    public function createTransaction($data)
    {
        try {
            $merchantRef = $data['merchant_ref'];
            $amount = $data['amount'];
            $customerName = $data['customer_name'];
            $customerEmail = $data['customer_email'];
            $customerPhone = $data['customer_phone'];
            $orderItems = $data['order_items'];
            $returnUrl = $data['return_url'] ?? '';
            $expiredTime = $data['expired_time'] ?? (time() + (24 * 60 * 60)); // 24 hours
            $signature = $this->generateSignature($merchantRef, $amount);

            $payload = [
                'method' => $data['payment_method'],
                'merchant_ref' => $merchantRef,
                'amount' => $amount,
                'customer_name' => $customerName,
                'customer_email' => $customerEmail,
                'customer_phone' => $customerPhone,
                'order_items' => $orderItems,
                'return_url' => $returnUrl,
                'expired_time' => $expiredTime,
                'signature' => $signature
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->post($this->baseUrl . '/transaction/create', $payload);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'success' => true,
                    'data' => $result['data'],
                    'reference' => $result['data']['reference'],
                    'checkout_url' => $result['data']['checkout_url'],
                    'qr_url' => $result['data']['qr_url'] ?? null,
                    'pay_code' => $result['data']['pay_code'] ?? null,
                ];
            }

            return [
                'success' => false, 
                'message' => $response->json()['message'] ?? 'Transaction creation failed'
            ];

        } catch (\Exception $e) {
            Log::error('Tripay create transaction error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Get transaction detail
     */
    public function getTransaction($reference)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->get($this->baseUrl . '/transaction/detail', [
                'reference' => $reference
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return ['success' => false, 'message' => 'Transaction not found'];
        } catch (\Exception $e) {
            Log::error('Tripay get transaction error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Validate callback signature
     */
    public function validateCallback($callbackSignature, $payload)
    {
        $json = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $signature = hash_hmac('sha256', $json, $this->privateKey);
        
        return hash_equals($signature, $callbackSignature);
    }

    /**
     * Handle payment callback
     */
    public function handleCallback($payload)
    {
        try {
            $reference = $payload['reference'];
            $status = $payload['status'];
            $merchantRef = $payload['merchant_ref'];
            $totalAmount = $payload['total_amount'];
            
            Log::info('Tripay callback received', [
                'reference' => $reference,
                'status' => $status,
                'merchant_ref' => $merchantRef,
                'amount' => $totalAmount
            ]);

            return [
                'success' => true,
                'reference' => $reference,
                'status' => $status,
                'merchant_ref' => $merchantRef,
                'amount' => $totalAmount,
                'paid_at' => isset($payload['paid_at']) ? Carbon::createFromTimestamp($payload['paid_at']) : null
            ];

        } catch (\Exception $e) {
            Log::error('Tripay handle callback error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Get fee calculation
     */
    public function getFeeCalculation($amount, $paymentMethod)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->get($this->baseUrl . '/merchant/fee-calculator', [
                'amount' => $amount,
                'code' => $paymentMethod
            ]);

            if ($response->successful()) {
                $data = $response->json()['data'];
                return [
                    'success' => true,
                    'total_fee' => $data['total_fee'],
                    'amount' => $amount,
                    'total_amount' => $amount + $data['total_fee']
                ];
            }

            return ['success' => false, 'message' => 'Failed to calculate fee'];
        } catch (\Exception $e) {
            Log::error('Tripay fee calculation error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Generate signature for transaction
     */
    private function generateSignature($merchantRef, $amount)
    {
        $payload = $this->merchantCode . $merchantRef . $amount;
        return hash_hmac('sha256', $payload, $this->privateKey);
    }

    /**
     * Generate merchant reference
     */
    public function generateMerchantRef($prefix = 'TRX')
    {
        return $prefix . '-' . time() . '-' . strtoupper(substr(md5(uniqid()), 0, 6));
    }

    /**
     * Check if payment method is available
     */
    public function isPaymentMethodAvailable($code)
    {
        $channels = $this->getPaymentChannels();
        
        if (!$channels['success']) {
            return false;
        }

        foreach ($channels['data'] as $channel) {
            if ($channel['code'] === $code && $channel['active']) {
                return true;
            }
        }

        return false;
    }

    /**
     * Format order items for Tripay
     */
    public function formatOrderItems($items)
    {
        $formatted = [];
        
        foreach ($items as $item) {
            $formatted[] = [
                'sku' => $item['sku'] ?? '',
                'name' => $item['name'],
                'price' => $item['price'],
                'quantity' => $item['quantity'],
                'product_url' => $item['product_url'] ?? '',
                'image_url' => $item['image_url'] ?? ''
            ];
        }

        return $formatted;
    }

    /**
     * Get transaction status mapping
     */
    public function getStatusMapping()
    {
        return [
            'UNPAID' => 'pending',
            'PAID' => 'paid',
            'SETTLED' => 'settled',
            'EXPIRED' => 'expired',
            'FAILED' => 'failed',
            'REFUND' => 'refunded'
        ];
    }

    /**
     * Initiate refund
     */
    public function initiateRefund($reference, $amount = null, $reason = '')
    {
        try {
            $payload = [
                'reference' => $reference,
                'reason' => $reason
            ];

            if ($amount) {
                $payload['amount'] = $amount;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
            ])->post($this->baseUrl . '/transaction/refund', $payload);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()['data']
                ];
            }

            return [
                'success' => false, 
                'message' => $response->json()['message'] ?? 'Refund failed'
            ];

        } catch (\Exception $e) {
            Log::error('Tripay refund error: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
} 