<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Optimize penduduk table indexes
        Schema::table('penduduk', function (Blueprint $table) {
            // Check if indexes don't exist before creating
            if (!$this->indexExists('penduduk', 'idx_status_gender')) {
                $table->index(['status_keluarga', 'jenis_kelamin'], 'idx_status_gender');
            }
            if (!$this->indexExists('penduduk', 'idx_agama_dusun')) {
                $table->index(['agama', 'dusun'], 'idx_agama_dusun');
            }
            if (!$this->indexExists('penduduk', 'idx_pekerjaan_pendidikan')) {
                $table->index(['pekerjaan', 'pendidikan'], 'idx_pekerjaan_pendidikan');
            }
            if (!$this->indexExists('penduduk', 'idx_timestamps')) {
                $table->index(['created_at', 'updated_at'], 'idx_timestamps');
            }
            if (!$this->indexExists('penduduk', 'idx_nama_nik_search')) {
                $table->index(['nama', 'nik'], 'idx_nama_nik_search');
            }
        });

        // Optimize berita table indexes
        if (Schema::hasTable('berita')) {
            Schema::table('berita', function (Blueprint $table) {
                if (!$this->indexExists('berita', 'idx_berita_display')) {
                    $table->index(['status', 'is_featured', 'tanggal_terbit'], 'idx_berita_display');
                }
                if (!$this->indexExists('berita', 'idx_kategori_status')) {
                    $table->index(['kategori', 'status'], 'idx_kategori_status');
                }
                if (!$this->indexExists('berita', 'idx_popularity')) {
                    $table->index(['views', 'likes_count'], 'idx_popularity');
                }
            });
        }

        // Optimize pengaduan table indexes
        if (Schema::hasTable('pengaduan')) {
            Schema::table('pengaduan', function (Blueprint $table) {
                if (!$this->indexExists('pengaduan', 'idx_status_kategori')) {
                    $table->index(['status', 'kategori'], 'idx_status_kategori');
                }
                if (!$this->indexExists('pengaduan', 'idx_timeline_status')) {
                    $table->index(['created_at', 'status'], 'idx_timeline_status');
                }
            });
        }

        // Optimize pelayanan_surat table indexes
        if (Schema::hasTable('pelayanan_surat')) {
            Schema::table('pelayanan_surat', function (Blueprint $table) {
                if (!$this->indexExists('pelayanan_surat', 'idx_status_jenis')) {
                    $table->index(['status', 'jenis_surat'], 'idx_status_jenis');
                }
                if (!$this->indexExists('pelayanan_surat', 'idx_created_status')) {
                    $table->index(['created_at', 'status'], 'idx_created_status');
                }
                if (!$this->indexExists('pelayanan_surat', 'idx_nik_status')) {
                    $table->index(['nik', 'status'], 'idx_nik_status');
                }
            });
        }

        // Optimize transaksi table indexes if exists
        if (Schema::hasTable('transaksi')) {
            Schema::table('transaksi', function (Blueprint $table) {
                $table->index(['status', 'payment_method'], 'idx_payment_status');
                $table->index(['created_at', 'status'], 'idx_created_payment');
                $table->index(['user_id', 'status'], 'idx_user_payment');
            });
        }

        // Optimize audit_log table for performance
        if (Schema::hasTable('audit_log')) {
            Schema::table('audit_log', function (Blueprint $table) {
                if (!$this->indexExists('audit_log', 'idx_module_action_time')) {
                    $table->index(['module', 'action_type', 'created_at'], 'idx_module_action_time');
                }
                if (!$this->indexExists('audit_log', 'idx_user_action')) {
                    $table->index(['user_id', 'action_type'], 'idx_user_action');
                }
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('penduduk', function (Blueprint $table) {
            $table->dropIndex('idx_status_gender');
            $table->dropIndex('idx_agama_dusun');
            $table->dropIndex('idx_pekerjaan_pendidikan');
            $table->dropIndex('idx_timestamps');
            $table->dropIndex('idx_nama_nik_search');
        });

        Schema::table('berita', function (Blueprint $table) {
            $table->dropIndex('idx_berita_display');
            $table->dropIndex('idx_kategori_status');
            $table->dropIndex('idx_popularity');
        });

        Schema::table('pengaduan', function (Blueprint $table) {
            $table->dropIndex('idx_status_kategori');
            $table->dropIndex('idx_timeline_status');
        });

        Schema::table('pelayanan_surat', function (Blueprint $table) {
            $table->dropIndex('idx_status_jenis');
            $table->dropIndex('idx_created_status');
            $table->dropIndex('idx_nik_status');
        });

        if (Schema::hasTable('transaksi')) {
            Schema::table('transaksi', function (Blueprint $table) {
                $table->dropIndex('idx_payment_status');
                $table->dropIndex('idx_created_payment');
                $table->dropIndex('idx_user_payment');
            });
        }

        Schema::table('audit_log', function (Blueprint $table) {
            $table->dropIndex('idx_module_action_time');
            $table->dropIndex('idx_user_action');
        });
    }

    /**
     * Check if index exists
     */
    private function indexExists($table, $indexName)
    {
        try {
            $indexes = DB::select("SHOW INDEX FROM {$table}");
            foreach ($indexes as $index) {
                if ($index->Key_name === $indexName) {
                    return true;
                }
            }
            return false;
        } catch (\Exception $e) {
            return false;
        }
    }
}; 