<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('transaksi', function (Blueprint $table) {
            $table->id();
            $table->string('kode_transaksi', 30)->unique(); // TRX-20250612-001
            $table->string('merchant_ref', 50)->unique(); // Reference untuk Tripay
            $table->string('tripay_reference', 50)->nullable()->unique(); // Reference dari Tripay
            
            // Jenis Transaksi
            $table->enum('jenis_transaksi', [
                'tiket_wisata',
                'produk_umkm',
                'marketplace',
                'donasi',
                'layanan_desa',
                'lainnya'
            ]);
            $table->morphs('transactionable'); // Polymorphic relation (tiket_wisata, produk_umkm, dll)
            
            // Data Customer
            $table->string('customer_name');
            $table->string('customer_email');
            $table->string('customer_phone', 15);
            $table->text('customer_address')->nullable();
            
            // Detail Pembayaran
            $table->decimal('amount', 15, 2); // Total amount
            $table->decimal('fee_merchant', 10, 2)->default(0); // Fee dari Tripay
            $table->decimal('fee_customer', 10, 2)->default(0); // Fee yang dibebankan ke customer
            $table->decimal('total_fee', 10, 2)->default(0); // Total fee
            $table->decimal('amount_received', 15, 2)->nullable(); // Amount yang diterima merchant
            
            // Metode Pembayaran
            $table->string('payment_method', 50); // BRIVA, BCAVA, OVO, DANA, dll
            $table->string('payment_method_name')->nullable(); // Nama lengkap metode pembayaran
            $table->string('payment_name')->nullable(); // Nama bank/ewallet
            $table->string('payment_code')->nullable(); // Kode pembayaran (VA number, dll)
            $table->string('checkout_url')->nullable(); // URL checkout Tripay
            $table->string('qr_url')->nullable(); // URL QR code untuk QRIS
            
            // Status Transaksi
            $table->enum('status', [
                'UNPAID',      // Belum dibayar
                'PAID',        // Sudah dibayar
                'SETTLED',     // Sudah settled
                'EXPIRED',     // Expired
                'FAILED',      // Gagal
                'REFUND',      // Refund
                'CANCELLED'    // Dibatalkan
            ])->default('UNPAID');
            
            // Waktu & Expired
            $table->timestamp('paid_at')->nullable();
            $table->timestamp('expired_time')->nullable();
            $table->timestamp('settled_at')->nullable();
            $table->integer('payment_timeout')->default(86400); // 24 jam dalam detik
            
            // Callback & Webhook
            $table->json('callback_data')->nullable(); // Data dari callback Tripay
            $table->timestamp('callback_received_at')->nullable();
            $table->string('callback_signature')->nullable();
            $table->boolean('is_callback_verified')->default(false);
            
            // Instructions & Details
            $table->json('payment_instructions')->nullable(); // Instruksi pembayaran dari Tripay
            $table->json('order_items')->nullable(); // Detail item yang dibeli
            $table->text('note')->nullable(); // Catatan transaksi
            
            // Refund & Cancellation
            $table->text('cancel_reason')->nullable();
            $table->decimal('refund_amount', 15, 2)->nullable();
            $table->timestamp('refund_at')->nullable();
            $table->string('refund_reference')->nullable();
            $table->text('refund_reason')->nullable();
            
            // Tracking & Analytics
            $table->string('user_agent')->nullable();
            $table->string('ip_address', 45)->nullable();
            $table->string('source', 50)->default('website'); // website, mobile_app, admin
            $table->json('metadata')->nullable(); // Data tambahan
            
            // Notification & Communication
            $table->boolean('email_sent')->default(false);
            $table->boolean('whatsapp_sent')->default(false);
            $table->timestamp('notification_sent_at')->nullable();
            $table->integer('notification_attempts')->default(0);
            
            // Admin & Management
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null'); // Admin yang handle
            $table->text('admin_notes')->nullable();
            $table->timestamp('verified_at')->nullable();
            $table->foreignId('verified_by')->nullable()->constrained('users')->onDelete('set null');
            
            $table->timestamps();
            
            // Indexes
            $table->index(['status', 'jenis_transaksi']);
            $table->index(['customer_email', 'customer_phone']);
            $table->index(['payment_method', 'status']);
            $table->index(['merchant_ref']);
            $table->index(['tripay_reference']);
            $table->index(['expired_time', 'status']);
            $table->index(['paid_at', 'status']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('transaksi');
    }
};
