<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Bantuan;
use App\Models\Penduduk;

class BantuanSeeder extends Seeder
{
    public function run()
    {
        echo "Creating sample Bantuan data...\n";
        
        // Get some penduduk data for bantuan recipients
        $pendudukList = Penduduk::limit(20)->get();
        
        if ($pendudukList->isEmpty()) {
            echo "❌ No penduduk data found. Cannot create bantuan.\n";
            return;
        }
        
        $jenisBantuan = [
            'PKH',
            'Bansos Pangan', 
            'BLT Dana Desa',
            'BPNT',
            'Bantuan Pendidikan',
            'Bantuan Kesehatan',
            'Bantuan Perumahan',
            'Kartu Prakerja',
            'Lainnya'
        ];
        
        $statusList = ['Diajukan', 'Diverifikasi', 'Diterima', 'Ditolak'];
        
        $bantuanData = [];
        $totalCreated = 0;
        
        foreach ($pendudukList as $index => $penduduk) {
            // Create 1-2 bantuan records per penduduk
            $bantuanCount = rand(1, 2);
            
            for ($i = 0; $i < $bantuanCount; $i++) {
                $jenis = $jenisBantuan[array_rand($jenisBantuan)];
                $status = $statusList[array_rand($statusList)];
                $nominal = $this->getNominalByJenis($jenis);
                
                $bantuanData[] = [
                    'nik' => $penduduk->nik,
                    'jenis_bantuan' => $jenis,
                    'deskripsi' => $this->getDeskripsiByJenis($jenis),
                    'periode' => date('Y') . '-' . str_pad(rand(1, 12), 2, '0', STR_PAD_LEFT),
                    'nominal' => $nominal,
                    'status' => $status,
                    'tanggal_distribusi' => $status === 'Diterima' ? now()->subDays(rand(1, 10)) : null,
                    'keterangan' => $this->getCatatanByStatus($status),
                    'created_at' => now(),
                    'updated_at' => now()
                ];
                
                $totalCreated++;
                
                // Avoid too many records
                if ($totalCreated >= 30) break 2;
            }
        }
        
        // Insert in chunks for better performance
        $chunks = array_chunk($bantuanData, 10);
        foreach ($chunks as $chunk) {
            Bantuan::insert($chunk);
        }
        
        echo "✅ Successfully created {$totalCreated} Bantuan records\n";
        echo "Total bantuan: " . Bantuan::count() . "\n";
        echo "Bantuan diterima: " . Bantuan::where('status', 'Diterima')->count() . "\n";
        echo "Bantuan pending: " . Bantuan::where('status', 'Diajukan')->count() . "\n";
    }
    
    private function getNominalByJenis($jenis)
    {
        $nominalMap = [
            'PKH' => rand(500000, 1000000),
            'Bansos Pangan' => rand(150000, 300000),
            'BLT Dana Desa' => rand(300000, 600000),
            'BPNT' => rand(200000, 400000),
            'Bantuan Pendidikan' => rand(1000000, 2000000),
            'Bantuan Kesehatan' => rand(500000, 1500000),
            'Bantuan Perumahan' => rand(2000000, 5000000),
            'Kartu Prakerja' => rand(1000000, 3000000),
            'Lainnya' => rand(100000, 500000)
        ];
        
        return $nominalMap[$jenis] ?? rand(100000, 500000);
    }
    
    private function getDeskripsiByJenis($jenis)
    {
        $deskripsiMap = [
            'PKH' => 'Program Keluarga Harapan - Bantuan tunai bersyarat untuk keluarga kurang mampu',
            'Bansos Pangan' => 'Bantuan sosial berupa paket sembako dan bahan pokok',
            'BLT Dana Desa' => 'Bantuan Langsung Tunai dari Dana Desa untuk masyarakat terdampak',
            'BPNT' => 'Bantuan Pangan Non Tunai melalui kartu elektronik',
            'Bantuan Pendidikan' => 'Bantuan biaya sekolah dan perlengkapan pendidikan',
            'Bantuan Kesehatan' => 'Bantuan biaya pengobatan dan perawatan kesehatan',
            'Bantuan Perumahan' => 'Bantuan perbaikan rumah tidak layak huni',
            'Kartu Prakerja' => 'Program pelatihan dan bantuan upah untuk pencari kerja',
            'Lainnya' => 'Bantuan sosial lainnya sesuai kebutuhan masyarakat'
        ];
        
        return $deskripsiMap[$jenis] ?? 'Bantuan sosial untuk masyarakat kurang mampu';
    }
    
    private function getCatatanByStatus($status)
    {
        switch ($status) {
            case 'Diajukan':
                return 'Pengajuan sedang dalam proses verifikasi';
            case 'Diverifikasi':
                return 'Data telah diverifikasi, menunggu persetujuan';
            case 'Diterima':
                return 'Bantuan telah disalurkan kepada penerima';
            case 'Ditolak':
                return 'Pengajuan ditolak karena tidak memenuhi syarat';
            default:
                return null;
        }
    }
} 
