<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Bug;
use App\Models\User;

class BugSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $users = User::all();
        $firstUser = $users->first();

        // Sample bug data for demonstration
        $bugs = [
            [
                'title' => 'Database Connection Timeout',
                'description' => 'Database connection mengalami timeout saat melakukan query besar pada tabel penduduk.',
                'severity' => 'critical',
                'status' => 'open',
                'type' => 'error',
                'module' => 'PendudukController',
                'url' => '/admin/penduduk',
                'user_id' => $firstUser?->id,
                'error_data' => [
                    'message' => 'SQLSTATE[HY000]: General error: 2006 MySQL server has gone away',
                    'file' => '/app/Http/Controllers/PendudukController.php',
                    'line' => 45,
                    'trace' => 'Stack trace here...'
                ],
                'environment_data' => [
                    'php_version' => PHP_VERSION,
                    'laravel_version' => app()->version(),
                    'memory_usage' => 134217728,
                    'server' => 'Apache/2.4.54'
                ],
                'occurrence_count' => 5,
                'last_occurred_at' => now()->subMinutes(30),
                'created_at' => now()->subHours(2)
            ],
            [
                'title' => 'Template Surat Generation Failed',
                'description' => 'Gagal generate PDF surat karena template tidak ditemukan atau rusak.',
                'severity' => 'high',
                'status' => 'in_progress',
                'type' => 'bug',
                'module' => 'PelayananSuratController',
                'url' => '/admin/pelayanan-surat/create',
                'user_id' => $firstUser?->id,
                'error_data' => [
                    'message' => 'Template file not found: template_surat_keterangan.html',
                    'file' => '/app/Http/Controllers/PelayananSuratController.php',
                    'line' => 156,
                    'template_id' => 5
                ],
                'environment_data' => [
                    'php_version' => PHP_VERSION,
                    'laravel_version' => app()->version(),
                    'storage_writable' => true
                ],
                'occurrence_count' => 12,
                'last_occurred_at' => now()->subMinutes(15),
                'created_at' => now()->subHours(4)
            ],
            [
                'title' => 'WhatsApp API Rate Limit Exceeded',
                'description' => 'API WhatsApp mencapai batas rate limit saat mengirim notifikasi massal.',
                'severity' => 'medium',
                'status' => 'resolved',
                'type' => 'warning',
                'module' => 'WhatsAppService',
                'url' => '/admin/notifikasi-wa',
                'user_id' => $firstUser?->id,
                'error_data' => [
                    'message' => 'Rate limit exceeded: 1000 messages per hour',
                    'api_response' => ['error' => 'rate_limit_exceeded'],
                    'pending_messages' => 45
                ],
                'environment_data' => [
                    'api_endpoint' => 'https://api.whatsapp.com/send',
                    'rate_limit' => '1000/hour',
                    'current_usage' => 1000
                ],
                'occurrence_count' => 3,
                'resolved_at' => now()->subHours(1),
                'resolved_by' => $firstUser?->id,
                'resolution_notes' => 'Implemented queue system to respect rate limits',
                'last_occurred_at' => now()->subHours(6),
                'created_at' => now()->subHours(8)
            ],
            [
                'title' => 'File Upload Size Limit',
                'description' => 'User tidak dapat upload file berkas pendukung karena melebihi batas ukuran.',
                'severity' => 'low',
                'status' => 'open',
                'type' => 'info',
                'module' => 'FileUploadController',
                'url' => '/admin/pelayanan-surat/create',
                'user_id' => $firstUser?->id,
                'error_data' => [
                    'message' => 'File size exceeds maximum allowed: 2MB',
                    'file_size' => 3145728,
                    'max_size' => 2097152,
                    'file_type' => 'application/pdf'
                ],
                'environment_data' => [
                    'upload_max_filesize' => '2M',
                    'post_max_size' => '8M',
                    'max_execution_time' => 30
                ],
                'occurrence_count' => 8,
                'last_occurred_at' => now()->subMinutes(45),
                'created_at' => now()->subHours(12)
            ]
        ];

        foreach ($bugs as $bugData) {
            Bug::create($bugData);
        }

        $this->command->info('Bug demo data seeded successfully!');
    }
}