<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\TemplateSurat;
use App\Models\Pbb;
use App\Models\TagihanPbb;
use App\Models\Polling;
use App\Models\PollingOpsi;
use App\Models\PerangkatDesa;
use App\Models\AgendaDesa;
use App\Models\InventarisDesa;
use App\Models\ArsipSuratKeluar;

class EnhancedVillageSystemSeeder extends Seeder
{
    public function run()
    {
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');

        // Seed Users and Roles
        $this->seedUsersAndRoles();
        
        // Seed Template Surat
        $this->seedTemplateSurat();
        
        // Seed PBB Data
        $this->seedPbbData();
        
        // Seed Polling Data
        $this->seedPollingData();
        
        // Seed Perangkat Desa
        $this->seedPerangkatDesa();
        
        // Seed Agenda Desa
        $this->seedAgendaDesa();
        
        // Seed Inventaris Desa
        $this->seedInventarisDesa();
        
        // Seed Arsip Surat Keluar
        $this->seedArsipSuratKeluar();

        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
    }

    private function seedUsersAndRoles()
    {
        // Create admin user if not exists
        if (!User::where('email', 'admin@desa.id')->exists()) {
            User::create([
                'name' => 'Administrator Desa',
                'email' => 'admin@desa.id',
                'password' => Hash::make('admin123'),
                'is_admin' => true,
                'email_verified_at' => now()
            ]);
        }

        // Create operator user
        if (!User::where('email', 'operator@desa.id')->exists()) {
            User::create([
                'name' => 'Operator Desa',
                'email' => 'operator@desa.id',
                'password' => Hash::make('operator123'),
                'is_admin' => true,
                'email_verified_at' => now()
            ]);
        }
    }

    private function seedTemplateSurat()
    {
        $templates = [
            [
                'nama_template' => 'Surat Keterangan Domisili',
                'jenis_surat' => 'Surat Keterangan',
                'deskripsi' => 'Template surat keterangan domisili penduduk',
                'variabel_tersedia' => json_encode([
                    '{{nama}}', '{{nik}}', '{{alamat}}', '{{rt}}', '{{rw}}', 
                    '{{desa}}', '{{kecamatan}}', '{{kabupaten}}'
                ]),
                'konten_template' => 'SURAT KETERANGAN DOMISILI
Nomor: {{nomor_surat}}

Yang bertanda tangan di bawah ini, Kepala Desa {{nama_desa}}, menerangkan bahwa:

Nama          : {{nama}}
NIK           : {{nik}}
Alamat        : {{alamat}}, RT {{rt}}/RW {{rw}}
Desa/Kelurahan: {{desa}}
Kecamatan     : {{kecamatan}}
Kabupaten     : {{kabupaten}}

Adalah benar-benar berdomisili di alamat tersebut di atas.

Demikian surat keterangan ini dibuat untuk dapat digunakan sebagaimana mestinya.',
                'format_nomor' => 'DOM/{{urut}}/{{bulan_romawi}}/{{tahun}}',
                'is_active' => true,
                'created_by' => 1
            ],
            [
                'nama_template' => 'Surat Keterangan Tidak Mampu',
                'jenis_surat' => 'Surat Keterangan',
                'deskripsi' => 'Template surat keterangan tidak mampu',
                'variabel_tersedia' => json_encode([
                    '{{nama}}', '{{nik}}', '{{alamat}}', '{{pekerjaan}}', 
                    '{{penghasilan}}', '{{keperluan}}'
                ]),
                'konten_template' => 'SURAT KETERANGAN TIDAK MAMPU
Nomor: {{nomor_surat}}

Yang bertanda tangan di bawah ini, Kepala Desa {{nama_desa}}, menerangkan bahwa:

Nama          : {{nama}}
NIK           : {{nik}}
Alamat        : {{alamat}}
Pekerjaan     : {{pekerjaan}}
Penghasilan   : {{penghasilan}}

Adalah benar-benar warga yang tidak mampu secara ekonomi.

Surat keterangan ini dibuat untuk keperluan: {{keperluan}}',
                'format_nomor' => 'TM/{{urut}}/{{bulan_romawi}}/{{tahun}}',
                'is_active' => true,
                'created_by' => 1
            ],
            [
                'nama_template' => 'Surat Pengantar SKCK',
                'jenis_surat' => 'Surat Pengantar',
                'deskripsi' => 'Template surat pengantar untuk pembuatan SKCK',
                'variabel_tersedia' => json_encode([
                    '{{nama}}', '{{nik}}', '{{tempat_lahir}}', '{{tanggal_lahir}}',
                    '{{alamat}}', '{{pekerjaan}}', '{{keperluan}}'
                ]),
                'konten_template' => 'SURAT PENGANTAR SKCK
Nomor: {{nomor_surat}}

Yang bertanda tangan di bawah ini, Kepala Desa {{nama_desa}}, dengan ini menerangkan bahwa:

Nama           : {{nama}}
NIK            : {{nik}}
Tempat/Tgl Lahir: {{tempat_lahir}}, {{tanggal_lahir}}
Alamat         : {{alamat}}
Pekerjaan      : {{pekerjaan}}

Adalah benar warga desa kami yang berkelakuan baik dan tidak pernah terlibat masalah hukum.

Surat pengantar ini digunakan untuk keperluan: {{keperluan}}',
                'format_nomor' => 'SKCK/{{urut}}/{{bulan_romawi}}/{{tahun}}',
                'is_active' => true,
                'created_by' => 1
            ]
        ];

        foreach ($templates as $template) {
            TemplateSurat::create($template);
        }
    }

    private function seedPbbData()
    {
        $pbbData = [
            [
                'nop' => '3201012001001001',
                'nama_wajib_pajak' => 'Budi Santoso',
                'alamat_wajib_pajak' => 'Jl. Merdeka No. 123',
                'alamat_objek_pajak' => 'Jl. Merdeka No. 123 RT 01/RW 01',
                'luas_tanah' => 200,
                'luas_bangunan' => 150,
                'nilai_tanah_per_m2' => 500000,
                'nilai_bangunan_per_m2' => 750000,
                'njop_tanah' => 100000000,
                'njop_bangunan' => 112500000,
                'njop_total' => 212500000,
                'njoptkp' => 12000000,
                'njop_kena_pajak' => 200500000,
                'tarif_pajak' => 0.5,
                'pbb_terutang' => 1002500,
                'is_active' => true,
                'created_by' => 1
            ],
            [
                'nop' => '3201012001001002',
                'nama_wajib_pajak' => 'Siti Aminah',
                'alamat_wajib_pajak' => 'Jl. Kemerdekaan No. 45',
                'alamat_objek_pajak' => 'Jl. Kemerdekaan No. 45 RT 02/RW 01',
                'luas_tanah' => 150,
                'luas_bangunan' => 100,
                'nilai_tanah_per_m2' => 450000,
                'nilai_bangunan_per_m2' => 700000,
                'njop_tanah' => 67500000,
                'njop_bangunan' => 70000000,
                'njop_total' => 137500000,
                'njoptkp' => 12000000,
                'njop_kena_pajak' => 125500000,
                'tarif_pajak' => 0.5,
                'pbb_terutang' => 627500,
                'is_active' => true,
                'created_by' => 1
            ]
        ];

        foreach ($pbbData as $data) {
            $pbb = Pbb::create($data);
            
            // Create tagihan for current year
            TagihanPbb::create([
                'pbb_id' => $pbb->id,
                'tahun_pajak' => now()->year,
                'jumlah_pajak' => $pbb->pbb_terutang,
                'denda' => 0,
                'total_tagihan' => $pbb->pbb_terutang,
                'status_bayar' => 'belum_bayar',
                'created_by' => 1
            ]);
        }
    }

    private function seedPollingData()
    {
        $polling = Polling::create([
            'judul' => 'Pilihan Prioritas Pembangunan Desa 2024',
            'deskripsi' => 'Voting untuk menentukan prioritas pembangunan desa tahun 2024',
            'mulai_voting' => now(),
            'selesai_voting' => now()->addDays(30),
            'jenis' => 'Usulan Pembangunan',
            'status' => 'Aktif',
            'anonim' => true,
            'created_by' => 1
        ]);

        $options = [
            'Perbaikan Jalan Desa',
            'Pembangunan Balai Desa',
            'Rehabilitasi Irigasi',
            'Program Bantuan UMKM',
            'Pembangunan Posyandu'
        ];

        foreach ($options as $option) {
            PollingOpsi::create([
                'polling_id' => $polling->id,
                'judul_opsi' => $option,
                'urutan' => array_search($option, $options) + 1
            ]);
        }
    }

    private function seedPerangkatDesa()
    {
        $perangkat = [
            [
                'nama' => 'H. Ahmad Wijaya, S.AP',
                'jabatan' => 'Kepala Desa',
                'level_jabatan' => 1,
                'nip' => '196505151990031002',
                'alamat' => 'Jl. Raya Desa No. 1',
                'no_hp' => '081234567890',
                'email' => 'kades@desa.id',
                'status' => 'aktif',
                'masa_jabatan_mulai' => '2019-08-17',
                'masa_jabatan_selesai' => '2025-08-17',
                'created_by' => 1
            ],
            [
                'nama' => 'Drs. Bambang Sutrisno',
                'jabatan' => 'Sekretaris Desa',
                'level_jabatan' => 2,
                'nip' => '197203121995031001',
                'alamat' => 'Jl. Mawar No. 15',
                'no_hp' => '081234567891',
                'email' => 'sekdes@desa.id',
                'status' => 'aktif',
                'masa_jabatan_mulai' => '2020-01-01',
                'created_by' => 1
            ],
            [
                'nama' => 'Siti Fatimah, S.Pd',
                'jabatan' => 'Kepala Urusan Umum',
                'level_jabatan' => 3,
                'alamat' => 'Jl. Melati No. 8',
                'no_hp' => '081234567892',
                'email' => 'kaur.umum@desa.id',
                'status' => 'aktif',
                'masa_jabatan_mulai' => '2020-02-01',
                'created_by' => 1
            ],
            [
                'nama' => 'Muhammad Fauzi',
                'jabatan' => 'Kepala Urusan Keuangan',
                'level_jabatan' => 3,
                'alamat' => 'Jl. Anggrek No. 12',
                'no_hp' => '081234567893',
                'email' => 'kaur.keuangan@desa.id',
                'status' => 'aktif',
                'masa_jabatan_mulai' => '2020-03-01',
                'created_by' => 1
            ]
        ];

        foreach ($perangkat as $data) {
            PerangkatDesa::create($data);
        }
    }

    private function seedAgendaDesa()
    {
        $agenda = [
            [
                'judul' => 'Rapat Bulanan BPD',
                'deskripsi' => 'Rapat koordinasi bulanan dengan Badan Permusyawaratan Desa',
                'tanggal_mulai' => now()->addDays(7)->setTime(9, 0),
                'tanggal_selesai' => now()->addDays(7)->setTime(12, 0),
                'lokasi' => 'Balai Desa',
                'alamat_lengkap' => 'Balai Desa, Jl. Raya Desa No. 1',
                'kategori' => 'Rapat',
                'status' => 'dijadwalkan',
                'is_reminder_active' => true,
                'reminder_waktu' => now()->addDays(6)->setTime(9, 0),
                'created_by' => 1
            ],
            [
                'judul' => 'Gotong Royong Pembersihan Selokan',
                'deskripsi' => 'Kegiatan gotong royong membersihkan selokan di sepanjang jalan utama desa',
                'tanggal_mulai' => now()->addDays(14)->setTime(7, 0),
                'tanggal_selesai' => now()->addDays(14)->setTime(11, 0),
                'lokasi' => 'Jalan Utama Desa',
                'kategori' => 'Gotong Royong',
                'status' => 'dijadwalkan',
                'is_reminder_active' => true,
                'reminder_waktu' => now()->addDays(13)->setTime(19, 0),
                'created_by' => 1
            ],
            [
                'judul' => 'Pelatihan UMKM',
                'deskripsi' => 'Pelatihan kewirausahaan dan manajemen usaha untuk pelaku UMKM desa',
                'tanggal_mulai' => now()->addDays(21)->setTime(13, 0),
                'tanggal_selesai' => now()->addDays(21)->setTime(17, 0),
                'lokasi' => 'Balai Desa',
                'alamat_lengkap' => 'Balai Desa, Jl. Raya Desa No. 1',
                'kategori' => 'Pelatihan',
                'status' => 'dijadwalkan',
                'is_reminder_active' => true,
                'reminder_waktu' => now()->addDays(20)->setTime(13, 0),
                'created_by' => 1
            ]
        ];

        foreach ($agenda as $data) {
            AgendaDesa::create($data);
        }
    }

    private function seedInventarisDesa()
    {
        $inventaris = [
            [
                'nama_barang' => 'Komputer Desktop Dell Optiplex',
                'kategori' => 'Elektronik',
                'deskripsi' => 'Komputer untuk administrasi desa dengan spesifikasi Intel Core i5, RAM 8GB',
                'merk_tipe' => 'Dell Optiplex 3070',
                'tahun_perolehan' => 2022,
                'harga_perolehan' => 8500000,
                'nilai_sekarang' => 7500000,
                'sumber_dana' => 'APBDes',
                'kondisi' => 'Baik',
                'status_pemakaian' => 'Digunakan',
                'lokasi_penyimpanan' => 'Kantor Desa - Ruang Admin',
                'penanggung_jawab' => 'Sekretaris Desa',
                'jabatan_pj' => 'Sekretaris Desa',
                'created_by' => 1
            ],
            [
                'nama_barang' => 'Printer Canon PIXMA G3010',
                'kategori' => 'Elektronik',
                'deskripsi' => 'Printer multifungsi untuk kebutuhan cetak dokumen desa',
                'merk_tipe' => 'Canon PIXMA G3010',
                'tahun_perolehan' => 2023,
                'harga_perolehan' => 2500000,
                'nilai_sekarang' => 2200000,
                'sumber_dana' => 'APBDes',
                'kondisi' => 'Baik',
                'status_pemakaian' => 'Digunakan',
                'lokasi_penyimpanan' => 'Kantor Desa - Ruang Admin',
                'penanggung_jawab' => 'Kepala Urusan Umum',
                'jabatan_pj' => 'Kaur Umum',
                'created_by' => 1
            ],
            [
                'nama_barang' => 'Meja Kerja Kayu Jati',
                'kategori' => 'Furniture',
                'deskripsi' => 'Meja kerja untuk pegawai desa ukuran 120x60 cm',
                'tahun_perolehan' => 2021,
                'harga_perolehan' => 1500000,
                'nilai_sekarang' => 1200000,
                'sumber_dana' => 'APBDes',
                'kondisi' => 'Baik',
                'status_pemakaian' => 'Digunakan',
                'lokasi_penyimpanan' => 'Kantor Desa - Ruang Sekretaris',
                'penanggung_jawab' => 'Kepala Urusan Umum',
                'created_by' => 1
            ],
            [
                'nama_barang' => 'Motor Dinas Honda Beat',
                'kategori' => 'Kendaraan',
                'deskripsi' => 'Sepeda motor untuk keperluan dinas luar kantor',
                'merk_tipe' => 'Honda Beat FI 2022',
                'tahun_perolehan' => 2022,
                'harga_perolehan' => 18000000,
                'nilai_sekarang' => 15000000,
                'sumber_dana' => 'Dana Desa',
                'kondisi' => 'Baik',
                'status_pemakaian' => 'Digunakan',
                'lokasi_penyimpanan' => 'Garasi Kantor Desa',
                'penanggung_jawab' => 'Sekretaris Desa',
                'created_by' => 1
            ]
        ];

        foreach ($inventaris as $data) {
            $item = InventarisDesa::create($data);
            $item->kode_inventaris = InventarisDesa::generateKodeInventaris($data['kategori']);
            $item->save();
        }
    }

    private function seedArsipSuratKeluar()
    {
        $suratKeluar = [
            [
                'tipe_surat' => 'Surat Undangan',
                'perihal' => 'Undangan Rapat Koordinasi Program Pembangunan Desa',
                'tujuan' => 'Kepala Dinas PMD Kabupaten',
                'tanggal_surat' => now()->subDays(10),
                'keterangan' => 'Undangan rapat pembahasan program pembangunan tahun 2024',
                'created_by' => 1
            ],
            [
                'tipe_surat' => 'Surat Permohonan',
                'perihal' => 'Permohonan Bantuan Material Pembangunan Jalan',
                'tujuan' => 'Kepala Dinas PUPR Kabupaten',
                'tanggal_surat' => now()->subDays(5),
                'keterangan' => 'Permohonan bantuan material untuk perbaikan jalan desa',
                'created_by' => 1
            ],
            [
                'tipe_surat' => 'Surat Keterangan',
                'perihal' => 'Keterangan Pelaksanaan Program BLT DD',
                'tujuan' => 'Camat Setempat',
                'tanggal_surat' => now()->subDays(3),
                'keterangan' => 'Laporan pelaksanaan program Bantuan Langsung Tunai Dana Desa',
                'created_by' => 1
            ]
        ];

        foreach ($suratKeluar as $data) {
            $arsip = ArsipSuratKeluar::create($data);
            $arsip->nomor_surat = ArsipSuratKeluar::generateNomorSurat($data['tipe_surat']);
            $arsip->save();
        }
    }
} 