<?php
/**
 * Script untuk memperbaiki file UMKM yang sudah ada dengan nama bermasalah
 */

require_once 'vendor/autoload.php';

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\Umkm;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

echo "=== Fixing Existing UMKM Files ===\n";

try {
    // Get all UMKM records
    $umkmRecords = Umkm::whereNotNull('logo')->orWhereNotNull('galeri_foto')->get();
    
    echo "Found " . $umkmRecords->count() . " UMKM records with files\n\n";
    
    foreach ($umkmRecords as $umkm) {
        echo "Processing UMKM: " . $umkm->nama_umkm . "\n";
        $updated = false;
        
        // Fix logo
        if ($umkm->logo) {
            echo "  Checking logo: " . $umkm->logo . "\n";
            
            // Check if file exists with current name
            $currentPath = 'public/umkm/logo/' . $umkm->logo;
            if (Storage::exists($currentPath)) {
                // Generate clean filename
                $extension = pathinfo($umkm->logo, PATHINFO_EXTENSION);
                $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($umkm->logo, PATHINFO_FILENAME));
                $newFilename = time() . '_logo_' . $cleanName . '.' . $extension;
                
                // Only rename if different
                if ($umkm->logo !== $newFilename) {
                    $newPath = 'public/umkm/logo/' . $newFilename;
                    
                    if (Storage::move($currentPath, $newPath)) {
                        $umkm->logo = $newFilename;
                        $updated = true;
                        echo "    ✓ Renamed logo: " . $umkm->logo . " -> " . $newFilename . "\n";
                    } else {
                        echo "    ✗ Failed to rename logo\n";
                    }
                } else {
                    echo "    ✓ Logo filename is already clean\n";
                }
            } else {
                echo "    ! Logo file not found: " . $currentPath . "\n";
            }
        }
        
        // Fix gallery photos
        if ($umkm->galeri_foto && is_array($umkm->galeri_foto)) {
            echo "  Checking " . count($umkm->galeri_foto) . " gallery photos\n";
            
            $newGalleryFiles = [];
            foreach ($umkm->galeri_foto as $index => $filename) {
                echo "    Checking gallery photo " . ($index + 1) . ": " . $filename . "\n";
                
                $currentPath = 'public/umkm/galeri/' . $filename;
                if (Storage::exists($currentPath)) {
                    // Generate clean filename
                    $extension = pathinfo($filename, PATHINFO_EXTENSION);
                    $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($filename, PATHINFO_FILENAME));
                    $newFilename = time() . '_galeri_' . $cleanName . '_' . $index . '.' . $extension;
                    
                    // Only rename if different
                    if ($filename !== $newFilename) {
                        $newPath = 'public/umkm/galeri/' . $newFilename;
                        
                        if (Storage::move($currentPath, $newPath)) {
                            $newGalleryFiles[] = $newFilename;
                            $updated = true;
                            echo "      ✓ Renamed gallery photo: " . $filename . " -> " . $newFilename . "\n";
                        } else {
                            echo "      ✗ Failed to rename gallery photo\n";
                            $newGalleryFiles[] = $filename; // Keep original name
                        }
                    } else {
                        echo "      ✓ Gallery photo filename is already clean\n";
                        $newGalleryFiles[] = $filename;
                    }
                } else {
                    echo "      ! Gallery photo file not found: " . $currentPath . "\n";
                    $newGalleryFiles[] = $filename; // Keep original name
                }
            }
            
            $umkm->galeri_foto = $newGalleryFiles;
        }
        
        // Save if updated
        if ($updated) {
            $umkm->save();
            echo "  ✓ Updated UMKM record\n";
        } else {
            echo "  ✓ No changes needed\n";
        }
        
        echo "\n";
    }
    
    echo "=== Fix Complete ===\n";
    echo "All UMKM files have been processed!\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>
