import 'package:flutter/material.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';

class MapProvider extends ChangeNotifier {
  GoogleMapController? _mapController;
  final Set<Marker> _markers = {};
  final Set<Polygon> _polygons = {};
  LatLng _center = const LatLng(-6.200000, 106.816635);
  double _zoom = 15.0;

  // Getters
  GoogleMapController? get mapController => _mapController;
  Set<Marker> get markers => _markers;
  Set<Polygon> get polygons => _polygons;
  LatLng get center => _center;
  double get zoom => _zoom;

  void setMapController(GoogleMapController controller) {
    _mapController = controller;
    notifyListeners();
  }

  void updateCenter(LatLng newCenter) {
    _center = newCenter;
    notifyListeners();
  }

  void updateZoom(double newZoom) {
    _zoom = newZoom;
    notifyListeners();
  }

  void addMarker(Marker marker) {
    _markers.add(marker);
    notifyListeners();
  }

  void removeMarker(MarkerId markerId) {
    _markers.removeWhere((marker) => marker.markerId == markerId);
    notifyListeners();
  }

  void clearMarkers() {
    _markers.clear();
    notifyListeners();
  }

  void addPolygon(Polygon polygon) {
    _polygons.add(polygon);
    notifyListeners();
  }

  void removePolygon(PolygonId polygonId) {
    _polygons.removeWhere((polygon) => polygon.polygonId == polygonId);
    notifyListeners();
  }

  void clearPolygons() {
    _polygons.clear();
    notifyListeners();
  }

  void animateToLocation(LatLng location, {double? zoom}) {
    _mapController?.animateCamera(
      CameraUpdate.newLatLngZoom(
        location,
        zoom ?? _zoom,
      ),
    );
  }

  void animateToBounds(LatLngBounds bounds) {
    _mapController?.animateCamera(
      CameraUpdate.newLatLngBounds(bounds, 100.0),
    );
  }

  // Create village boundary polygon
  void createVillageBoundary(List<LatLng> points) {
    final polygon = Polygon(
      polygonId: const PolygonId('village_boundary'),
      points: points,
      fillColor: Colors.blue.withValues(alpha: 0.2),
      strokeColor: Colors.blue,
      strokeWidth: 2,
    );
    
    clearPolygons();
    addPolygon(polygon);
  }

  // Create RT/RW boundaries
  void createRtRwBoundaries(Map<String, List<LatLng>> boundaries) {
    clearPolygons();
    
    boundaries.forEach((key, points) {
      final polygon = Polygon(
        polygonId: PolygonId(key),
        points: points,
        fillColor: Colors.green.withValues(alpha: 0.3),
        strokeColor: Colors.green,
        strokeWidth: 1,
      );
      addPolygon(polygon);
    });
  }

  // Add population markers
  void addPopulationMarkers(List<Map<String, dynamic>> populationData) {
    clearMarkers();
    
    for (var data in populationData) {
      final marker = Marker(
        markerId: MarkerId(data['id'].toString()),
        position: LatLng(data['latitude'], data['longitude']),
        infoWindow: InfoWindow(
          title: data['nama'],
          snippet: data['alamat'],
        ),
        icon: BitmapDescriptor.defaultMarkerWithHue(
          data['jenis_kelamin'] == 'L' 
            ? BitmapDescriptor.hueBlue 
            : BitmapDescriptor.hueRed,
        ),
      );
      addMarker(marker);
    }
  }
}