import 'package:flutter/foundation.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';

import '../models/news.dart';
import '../services/api_service.dart';
import '../core/constants/app_constants.dart';

class NewsProvider extends ChangeNotifier {
  List<News> _newsList = [];
  bool _isLoading = false;
  String? _errorMessage;
  int _currentPage = 1;
  bool _hasNextPage = true;
  bool _isLoadingMore = false;

  // Getters
  List<News> get newsList => _newsList;
  bool get isLoading => _isLoading;
  bool get isLoadingMore => _isLoadingMore;
  String? get errorMessage => _errorMessage;
  bool get hasError => _errorMessage != null;
  bool get hasData => _newsList.isNotEmpty;
  bool get hasNextPage => _hasNextPage;
  int get currentPage => _currentPage;

  // Get latest news (first 3 items)
  List<News> get latestNews => _newsList.take(3).toList();

  NewsProvider() {
    _loadCachedData();
    loadNewsList();
  }

  Future<void> _loadCachedData() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      final cachedData = prefs.getString(AppConstants.newsDataKey);
      
      if (cachedData != null) {
        final jsonData = json.decode(cachedData);
        final newsJsonList = jsonData['data'] as List;
        _newsList = newsJsonList.map((json) => News.fromJson(json)).toList();
        
        // Check if cache is expired
        final cacheTime = DateTime.parse(jsonData['cached_at']);
        final now = DateTime.now();
        final difference = now.difference(cacheTime).inHours;
        
        if (difference < AppConstants.cacheNewsDuration) {
          notifyListeners();
          return;
        }
      }
    } catch (e) {
      debugPrint('Error loading cached news data: $e');
    }
  }

  Future<void> loadNewsList({bool forceRefresh = false}) async {
    if (_isLoading && !forceRefresh) return;

    _setLoading(true);
    _clearError();
    _currentPage = 1;
    _hasNextPage = true;

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.getNewsList(
          page: _currentPage,
          limit: AppConstants.newsPageSize,
          status: 'published',
        ),
      );

      if (response.success && response.data != null) {
        _newsList = response.data!;
        _updatePagination(response.meta);
        await _cacheData();
        _setLoading(false);
      } else {
        _setError(response.message ?? 'Gagal memuat data berita');
      }
    } catch (e) {
      _setError(e.toString());
    }
  }

  Future<void> loadMoreNews() async {
    if (_isLoadingMore || !_hasNextPage) return;

    _isLoadingMore = true;
    notifyListeners();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.getNewsList(
          page: _currentPage + 1,
          limit: AppConstants.newsPageSize,
          status: 'published',
        ),
      );

      if (response.success && response.data != null) {
        _newsList.addAll(response.data!);
        _currentPage++;
        _updatePagination(response.meta);
        await _cacheData();
      }
    } catch (e) {
      debugPrint('Error loading more news: $e');
    } finally {
      _isLoadingMore = false;
      notifyListeners();
    }
  }

  void _updatePagination(dynamic meta) {
    if (meta != null) {
      _hasNextPage = meta.hasNextPage ?? false;
      _currentPage = meta.currentPage ?? _currentPage;
    }
  }

  Future<void> _cacheData() async {
    if (_newsList.isEmpty) return;
    
    try {
      final prefs = await SharedPreferences.getInstance();
      final cacheData = {
        'data': _newsList.map((news) => news.toJson()).toList(),
        'cached_at': DateTime.now().toIso8601String(),
      };
      
      await prefs.setString(
        AppConstants.newsDataKey,
        json.encode(cacheData),
      );
    } catch (e) {
      debugPrint('Error caching news data: $e');
    }
  }

  // Get news by slug
  Future<News?> getNewsDetail(String slug) async {
    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.getNewsDetail(slug),
      );

      if (response.success && response.data != null) {
        return response.data;
      }
    } catch (e) {
      debugPrint('Error getting news detail: $e');
    }
    return null;
  }

  void _setLoading(bool loading) {
    _isLoading = loading;
    notifyListeners();
  }

  void _setError(String error) {
    _errorMessage = error;
    _isLoading = false;
    notifyListeners();
  }

  void _clearError() {
    _errorMessage = null;
    notifyListeners();
  }

  // Refresh data
  Future<void> refresh() async {
    await loadNewsList(forceRefresh: true);
  }

  // Clear cache
  Future<void> clearCache() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      await prefs.remove(AppConstants.newsDataKey);
      _newsList.clear();
      _currentPage = 1;
      _hasNextPage = true;
      notifyListeners();
    } catch (e) {
      debugPrint('Error clearing news cache: $e');
    }
  }

}
