<?php
/**
 * Check Laravel Errors and Debug Info
 */

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Check Laravel Errors</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .log { background: #f1f1f1; padding: 10px; border-radius: 4px; font-family: monospace; font-size: 12px; max-height: 400px; overflow-y: auto; white-space: pre-wrap; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Check Laravel Errors & Debug Info</h1>
        <p><strong>Time:</strong> <?= date('Y-m-d H:i:s') ?></p>
        
        <!-- Laravel Log Check -->
        <div class="section">
            <h2>📄 Laravel Log Check</h2>
            
            <?php
            $logPath = __DIR__ . '/../storage/logs/laravel.log';
            
            if (file_exists($logPath)) {
                $logSize = filesize($logPath);
                echo "<div class='info'>📁 Log file exists: " . number_format($logSize) . " bytes</div>";
                
                if ($logSize > 0) {
                    // Read last 50 lines
                    $lines = file($logPath);
                    $totalLines = count($lines);
                    $lastLines = array_slice($lines, -50);
                    
                    echo "<div class='info'>📊 Total lines: {$totalLines} | Showing last 50 lines</div>";
                    echo "<div class='log'>" . htmlspecialchars(implode('', $lastLines)) . "</div>";
                } else {
                    echo "<div class='info'>📝 Log file is empty</div>";
                }
            } else {
                echo "<div class='warning'>⚠️ Laravel log file not found</div>";
            }
            ?>
        </div>
        
        <!-- Error Log Check -->
        <div class="section">
            <h2>🚨 PHP Error Log Check</h2>
            
            <?php
            $errorLog = ini_get('error_log');
            echo "<div class='info'><strong>PHP Error Log Path:</strong> " . ($errorLog ?: 'Not set') . "</div>";
            
            // Check common error log locations
            $possibleLogs = [
                __DIR__ . '/error_log',
                __DIR__ . '/../storage/logs/php_errors.log',
                '/var/log/apache2/error.log',
                '/var/log/nginx/error.log'
            ];
            
            foreach ($possibleLogs as $logFile) {
                if (file_exists($logFile)) {
                    $size = filesize($logFile);
                    echo "<div class='info'>📁 Found error log: {$logFile} (" . number_format($size) . " bytes)</div>";
                    
                    if ($size > 0 && $size < 1000000) { // Only show if less than 1MB
                        $content = file_get_contents($logFile);
                        $lines = explode("\n", $content);
                        $recentLines = array_slice($lines, -20);
                        
                        echo "<div class='log'>" . htmlspecialchars(implode("\n", $recentLines)) . "</div>";
                    }
                }
            }
            ?>
        </div>
        
        <!-- PHP Configuration -->
        <div class="section">
            <h2>⚙️ PHP Configuration</h2>
            
            <?php
            $configs = [
                'PHP Version' => PHP_VERSION,
                'Display Errors' => ini_get('display_errors') ? 'ON' : 'OFF',
                'Log Errors' => ini_get('log_errors') ? 'ON' : 'OFF',
                'Error Reporting' => error_reporting(),
                'Upload Max Filesize' => ini_get('upload_max_filesize'),
                'Post Max Size' => ini_get('post_max_size'),
                'Max Execution Time' => ini_get('max_execution_time') . 's',
                'Memory Limit' => ini_get('memory_limit'),
                'File Uploads' => ini_get('file_uploads') ? 'ON' : 'OFF'
            ];
            
            foreach ($configs as $key => $value) {
                echo "<div class='info'><strong>{$key}:</strong> {$value}</div>";
            }
            ?>
        </div>
        
        <!-- Server Information -->
        <div class="section">
            <h2>🖥️ Server Information</h2>
            
            <?php
            $serverInfo = [
                'Server Software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                'Document Root' => $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown',
                'Script Name' => $_SERVER['SCRIPT_NAME'] ?? 'Unknown',
                'HTTP Host' => $_SERVER['HTTP_HOST'] ?? 'Unknown',
                'Request URI' => $_SERVER['REQUEST_URI'] ?? 'Unknown',
                'User Agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
            ];
            
            foreach ($serverInfo as $key => $value) {
                echo "<div class='info'><strong>{$key}:</strong> " . htmlspecialchars($value) . "</div>";
            }
            ?>
        </div>
        
        <!-- Environment Check -->
        <div class="section">
            <h2>🌍 Environment Check</h2>
            
            <?php
            // Try to load Laravel
            try {
                require_once __DIR__ . '/../vendor/autoload.php';
                $app = require_once __DIR__ . '/../bootstrap/app.php';
                $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
                
                echo "<div class='success'>✅ Laravel loaded successfully</div>";
                
                // Check environment
                $env = app()->environment();
                echo "<div class='info'><strong>Environment:</strong> {$env}</div>";
                
                // Check database connection
                try {
                    \Illuminate\Support\Facades\DB::connection()->getPdo();
                    echo "<div class='success'>✅ Database connection OK</div>";
                } catch (Exception $e) {
                    echo "<div class='error'>❌ Database connection failed: " . $e->getMessage() . "</div>";
                }
                
                // Check storage permissions
                $storagePaths = [
                    'storage/app' => storage_path('app'),
                    'storage/logs' => storage_path('logs'),
                    'storage/framework' => storage_path('framework'),
                    'public/storage' => public_path('storage')
                ];
                
                foreach ($storagePaths as $name => $path) {
                    $exists = is_dir($path);
                    $writable = $exists && is_writable($path);
                    
                    echo "<div class='" . ($writable ? 'success' : 'error') . "'>";
                    echo "<strong>{$name}:</strong> ";
                    echo ($exists ? '📁 EXISTS' : '❌ NOT FOUND') . " ";
                    echo ($writable ? '✅ WRITABLE' : '❌ NOT WRITABLE');
                    echo "</div>";
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Laravel failed to load: " . $e->getMessage() . "</div>";
                echo "<div class='log'>" . $e->getTraceAsString() . "</div>";
            }
            ?>
        </div>
        
        <!-- File Permissions -->
        <div class="section">
            <h2>🔐 File Permissions Check</h2>
            
            <?php
            $checkPaths = [
                'Project Root' => __DIR__ . '/..',
                'Storage Directory' => __DIR__ . '/../storage',
                'Public Directory' => __DIR__,
                'Config Directory' => __DIR__ . '/../config',
                '.env File' => __DIR__ . '/../.env'
            ];
            
            foreach ($checkPaths as $name => $path) {
                if (file_exists($path)) {
                    $perms = fileperms($path);
                    $permsOctal = sprintf('%o', $perms);
                    $readable = is_readable($path);
                    $writable = is_writable($path);
                    
                    echo "<div class='info'>";
                    echo "<strong>{$name}:</strong> {$permsOctal} | ";
                    echo ($readable ? '📖 READABLE' : '❌ NOT READABLE') . " | ";
                    echo ($writable ? '✅ WRITABLE' : '❌ NOT WRITABLE');
                    echo "</div>";
                } else {
                    echo "<div class='error'><strong>{$name}:</strong> ❌ NOT FOUND</div>";
                }
            }
            ?>
        </div>
        
        <!-- Recent Files -->
        <div class="section">
            <h2>📁 Recent Files Check</h2>
            
            <?php
            $directories = [
                'storage/app/public/wisata' => __DIR__ . '/../storage/app/public/wisata',
                'public/storage/wisata' => __DIR__ . '/storage/wisata'
            ];
            
            foreach ($directories as $name => $dir) {
                echo "<div class='info'><strong>{$name}:</strong></div>";
                
                if (is_dir($dir)) {
                    $files = glob($dir . '/*');
                    $files = array_slice($files, -10); // Last 10 files
                    
                    if (count($files) > 0) {
                        foreach ($files as $file) {
                            $filename = basename($file);
                            $size = filesize($file);
                            $modified = date('Y-m-d H:i:s', filemtime($file));
                            
                            echo "<div class='info'>📄 {$filename} (" . number_format($size) . " bytes) - {$modified}</div>";
                        }
                    } else {
                        echo "<div class='warning'>⚠️ No files found</div>";
                    }
                } else {
                    echo "<div class='error'>❌ Directory not found</div>";
                }
            }
            ?>
        </div>
        
        <div class="section">
            <h2>🎯 Next Steps</h2>
            <ol>
                <li><strong>Check Debug Script:</strong> <a href="https://susukannet.id/debug-edit-update.php" target="_blank">Run Debug Script</a></li>
                <li><strong>Test Admin Login:</strong> <a href="https://susukannet.id/login" target="_blank">Login Page</a></li>
                <li><strong>Test Edit Form:</strong> <a href="https://susukannet.id/admin/objek-wisata/1/edit" target="_blank">Edit Tourism</a></li>
                <li><strong>Check Laravel Logs:</strong> Monitor storage/logs/laravel.log for real-time errors</li>
                <li><strong>Check Network Tab:</strong> Use browser dev tools to see actual HTTP requests/responses</li>
            </ol>
        </div>
    </div>
</body>
</html>
