<?php
/**
 * Check View Synchronization - Shared Hosting
 * This script checks if view files are properly synced and routes are working
 */

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Check View Sync - Shared Hosting</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        .step { margin: 15px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
        pre { background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto; font-size: 12px; }
        .status-box { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .status-success { background: #d4edda; border-color: #c3e6cb; color: #155724; }
        .status-error { background: #f8d7da; border-color: #f5c6cb; color: #721c24; }
        .status-warning { background: #fff3cd; border-color: #ffeaa7; color: #856404; }
        .btn { padding: 8px 16px; margin: 5px; background: #007bff; color: white; text-decoration: none; border-radius: 4px; display: inline-block; }
        .btn:hover { background: #0056b3; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Check View Synchronization - Tourism Module</h1>
        <p>Checking if view files and routes are properly synchronized on shared hosting.</p>

        <?php
        $baseDir = dirname(__DIR__);
        
        echo "<h2>📁 File System Check</h2>";
        
        // Check if key files exist
        $files = [
            'Controller' => $baseDir . '/app/Http/Controllers/Admin/ObjekWisataController.php',
            'Show View' => $baseDir . '/resources/views/admin/objek-wisata/show.blade.php', 
            'Edit View' => $baseDir . '/resources/views/admin/objek-wisata/edit.blade.php',
            'Index View' => $baseDir . '/resources/views/admin/objek-wisata/index.blade.php',
            'Routes File' => $baseDir . '/routes/web.php',
            'Model' => $baseDir . '/app/Models/ObjekWisata.php',
        ];
        
        foreach ($files as $name => $path) {
            if (file_exists($path)) {
                $modTime = date('Y-m-d H:i:s', filemtime($path));
                echo "<div class='status-box status-success'>✅ $name: EXISTS (Modified: $modTime)</div>";
            } else {
                echo "<div class='status-box status-error'>❌ $name: NOT FOUND</div>";
            }
        }

        echo "<h2>🌐 Laravel App Check</h2>";
        
        try {
            if (file_exists($baseDir . '/vendor/autoload.php')) {
                require_once $baseDir . '/vendor/autoload.php';
                
                // Try to load Laravel app
                $app = require_once $baseDir . '/bootstrap/app.php';
                $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
                
                echo "<div class='status-box status-success'>✅ Laravel App: Loaded successfully</div>";
                
                // Check if ObjekWisata model can be loaded
                try {
                    $objekCount = \App\Models\ObjekWisata::count();
                    echo "<div class='status-box status-success'>✅ ObjekWisata Model: Working ($objekCount records)</div>";
                } catch (Exception $e) {
                    echo "<div class='status-box status-error'>❌ ObjekWisata Model Error: " . $e->getMessage() . "</div>";
                }
                
                // Check routes
                echo "<h3>🛣️ Route Check</h3>";
                
                try {
                    $router = $app->make('router');
                    $routes = $router->getRoutes()->getRoutes();
                    
                    $targetRoutes = [
                        'admin.objek-wisata.index',
                        'admin.objek-wisata.show', 
                        'admin.objek-wisata.edit',
                        'admin.objek-wisata.create',
                        'admin.objek-wisata.store',
                        'admin.objek-wisata.update',
                        'admin.objek-wisata.destroy'
                    ];
                    
                    foreach ($targetRoutes as $routeName) {
                        try {
                            $route = $router->getRoutes()->getByName($routeName);
                            if ($route) {
                                $uri = $route->uri();
                                $methods = implode('|', $route->methods());
                                echo "<div class='status-box status-success'>✅ Route '$routeName': $methods $uri</div>";
                            } else {
                                echo "<div class='status-box status-error'>❌ Route '$routeName': NOT FOUND</div>";
                            }
                        } catch (Exception $e) {
                            echo "<div class='status-box status-error'>❌ Route '$routeName': ERROR - " . $e->getMessage() . "</div>";
                        }
                    }
                    
                } catch (Exception $e) {
                    echo "<div class='status-box status-error'>❌ Route Loading Error: " . $e->getMessage() . "</div>";
                }
                
            } else {
                echo "<div class='status-box status-error'>❌ Laravel autoloader not found</div>";
            }
        } catch (Exception $e) {
            echo "<div class='status-box status-error'>❌ Laravel App Error: " . $e->getMessage() . "</div>";
        }

        // Check file permissions
        echo "<h2>🔐 Permission Check</h2>";
        
        $permissionPaths = [
            'storage/' => $baseDir . '/storage',
            'bootstrap/cache/' => $baseDir . '/bootstrap/cache',  
            'resources/views/' => $baseDir . '/resources/views',
        ];
        
        foreach ($permissionPaths as $name => $path) {
            if (is_dir($path)) {
                $perms = substr(sprintf('%o', fileperms($path)), -4);
                $writable = is_writable($path) ? 'Writable' : 'Not Writable';
                echo "<div class='status-box status-success'>✅ $name: $perms ($writable)</div>";
            } else {
                echo "<div class='status-box status-error'>❌ $name: Directory not found</div>";
            }
        }

        // Check cache status
        echo "<h2>💾 Cache Status</h2>";
        
        $cacheFiles = [
            'Route Cache' => $baseDir . '/bootstrap/cache/routes-v7.php',
            'Config Cache' => $baseDir . '/bootstrap/cache/config.php',
            'Views Cache' => $baseDir . '/storage/framework/views',
        ];
        
        foreach ($cacheFiles as $name => $path) {
            if (file_exists($path)) {
                if (is_file($path)) {
                    $modTime = date('Y-m-d H:i:s', filemtime($path));
                    echo "<div class='status-box status-warning'>⚠️ $name: EXISTS (Modified: $modTime) - Should be cleared</div>";
                } else {
                    $files = glob($path . '/*');
                    $count = count($files);
                    echo "<div class='status-box status-warning'>⚠️ $name: $count cached files - Should be cleared</div>";
                }
            } else {
                echo "<div class='status-box status-success'>✅ $name: CLEARED</div>";
            }
        }

        echo "<h2>🧪 Direct Test Links</h2>";
        
        $baseUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
        
        $testLinks = [
            'Admin Tourism Index' => $baseUrl . '/admin/objek-wisata',
            'Admin Tourism Show (ID:1)' => $baseUrl . '/admin/objek-wisata/1', 
            'Admin Tourism Edit (ID:1)' => $baseUrl . '/admin/objek-wisata/1/edit',
            'Admin Tourism Create' => $baseUrl . '/admin/objek-wisata/create',
            'Frontend Tourism' => $baseUrl . '/wisata',
        ];
        
        foreach ($testLinks as $name => $url) {
            echo "<div class='step'>";
            echo "<strong>$name:</strong><br>";
            echo "<a href='$url' target='_blank' class='btn'>$url</a>";
            echo "</div>";
        }

        echo "<h2>🔧 Troubleshooting Actions</h2>";
        
        echo "<div class='step'>";
        echo "<h4>1. Clear All Caches (Via SSH/Terminal)</h4>";
        echo "<pre>cd " . $baseDir . "
php artisan optimize:clear
php artisan view:clear
php artisan config:clear
php artisan route:clear
php artisan cache:clear</pre>";
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h4>2. Check File Permissions</h4>";
        echo "<pre>chmod -R 755 storage/
chmod -R 755 bootstrap/cache/
chmod -R 644 resources/views/</pre>";
        echo "</div>";

        echo "<div class='step'>";
        echo "<h4>3. Force File Sync (If using file sync tools)</h4>";
        echo "<p>Make sure all files are uploaded and synchronized:</p>";
        echo "<ul>";
        echo "<li>app/Http/Controllers/Admin/ObjekWisataController.php</li>";
        echo "<li>resources/views/admin/objek-wisata/*.blade.php</li>";
        echo "<li>routes/web.php</li>";
        echo "<li>app/Models/ObjekWisata.php</li>";
        echo "</ul>";
        echo "</div>";

        echo "<div class='step'>";
        echo "<h4>4. Browser Cache</h4>";
        echo "<p>Clear browser cache and try accessing the URLs in incognito/private mode.</p>";
        echo "</div>";
        ?>
    </div>
</body>
</html>
