<?php
/**
 * Debug Edit & Update Process for Tourism Images
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use App\Models\User;
use App\Services\FileUploadService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Debug Edit & Update Process</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .debug-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; white-space: pre-wrap; word-break: break-all; }
        .form-test { margin: 20px 0; padding: 15px; border: 2px solid #007bff; border-radius: 8px; background: #f8f9ff; }
        .form-control { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; margin: 5px 0; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; margin: 5px; }
        .btn-danger { background: #dc3545; }
        .test-image { max-width: 150px; height: 100px; object-fit: cover; border: 2px solid #ddd; margin: 5px; border-radius: 4px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Debug Edit & Update Process</h1>
        
        <?php
        $errors = [];
        $debugInfo = [];
        
        // Handle test form submission
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            echo "<div class='debug-section'>";
            echo "<h2>📤 Form Submission Debug</h2>";
            
            try {
                echo "<div class='info'><strong>POST Data:</strong></div>";
                echo "<div class='code'>" . print_r($_POST, true) . "</div>";
                
                echo "<div class='info'><strong>FILES Data:</strong></div>";
                echo "<div class='code'>" . print_r($_FILES, true) . "</div>";
                
                // Simulate controller validation
                $validator = Validator::make($_POST + $_FILES, [
                    'nama_wisata' => 'required|string|max:255',
                    'kategori_wisata' => 'required|in:wisata_alam,wisata_budaya,wisata_sejarah,wisata_religi,wisata_kuliner,wisata_edukasi,wisata_adventure,wisata_air,wisata_pegunungan,agrowisata,ekowisata,wisata_buatan,lainnya',
                    'deskripsi' => 'required|string',
                    'alamat_lengkap' => 'required|string',
                    'harga_tiket_dewasa' => 'required|numeric|min:0',
                    'foto_utama' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                    'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                ]);
                
                if ($validator->fails()) {
                    echo "<div class='error'><strong>Validation Errors:</strong></div>";
                    foreach ($validator->errors()->all() as $error) {
                        echo "<div class='error'>❌ {$error}</div>";
                    }
                } else {
                    echo "<div class='success'>✅ Validation passed</div>";
                }
                
                // Test file upload if present
                if (isset($_FILES['foto_utama']) && $_FILES['foto_utama']['error'] === UPLOAD_ERR_OK) {
                    echo "<div class='info'><strong>Testing file upload...</strong></div>";
                    
                    $uploadedFile = new \Illuminate\Http\UploadedFile(
                        $_FILES['foto_utama']['tmp_name'],
                        $_FILES['foto_utama']['name'],
                        $_FILES['foto_utama']['type'],
                        $_FILES['foto_utama']['error'],
                        true
                    );
                    
                    $uploadService = new FileUploadService();
                    $uploadResult = $uploadService->uploadPhotoWithResize($uploadedFile, 'wisata');
                    
                    if ($uploadResult['success']) {
                        echo "<div class='success'>✅ File upload successful: {$uploadResult['filename']}</div>";
                        
                        // Test updating tourism object
                        if (isset($_POST['tourism_id'])) {
                            $tourism = ObjekWisata::find($_POST['tourism_id']);
                            if ($tourism) {
                                $oldPhoto = $tourism->foto_utama;
                                $tourism->update([
                                    'nama_wisata' => $_POST['nama_wisata'],
                                    'kategori_wisata' => $_POST['kategori_wisata'],
                                    'deskripsi' => $_POST['deskripsi'],
                                    'alamat' => $_POST['alamat_lengkap'],
                                    'harga_tiket_dewasa' => $_POST['harga_tiket_dewasa'],
                                    'foto_utama' => $uploadResult['filename']
                                ]);
                                
                                echo "<div class='success'>✅ Tourism updated successfully</div>";
                                echo "<div class='info'>Old photo: " . ($oldPhoto ?: 'NULL') . "</div>";
                                echo "<div class='info'>New photo: {$uploadResult['filename']}</div>";
                                
                                // Test URL generation
                                $newUrl = $tourism->fresh()->foto_utama_url;
                                echo "<div class='info'>New URL: {$newUrl}</div>";
                                echo "<img src='{$newUrl}?v=" . time() . "' class='test-image' alt='Updated'>";
                            }
                        }
                        
                    } else {
                        echo "<div class='error'>❌ File upload failed: {$uploadResult['error']}</div>";
                    }
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
                echo "<div class='code'>" . $e->getTraceAsString() . "</div>";
            }
            echo "</div>";
        }
        
        // Debug 1: Check current tourism data
        echo "<div class='debug-section'>";
        echo "<h2>🎯 Debug 1: Current Tourism Data</h2>";
        
        try {
            $tourism = ObjekWisata::first();
            
            if ($tourism) {
                echo "<div class='success'>✅ Found tourism: {$tourism->nama_wisata}</div>";
                echo "<div class='info'><strong>ID:</strong> {$tourism->id}</div>";
                echo "<div class='info'><strong>Current foto_utama:</strong> " . ($tourism->foto_utama ?: 'NULL') . "</div>";
                echo "<div class='info'><strong>Generated URL:</strong> {$tourism->foto_utama_url}</div>";
                
                if ($tourism->foto_utama) {
                    echo "<img src='{$tourism->foto_utama_url}?v=" . time() . "' class='test-image' alt='Current'>";
                }
                
                $debugInfo['tourism_id'] = $tourism->id;
                $debugInfo['current_photo'] = $tourism->foto_utama;
                
            } else {
                echo "<div class='error'>❌ No tourism data found</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Tourism query error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Debug 2: Check authentication
        echo "<div class='debug-section'>";
        echo "<h2>👤 Debug 2: Authentication Check</h2>";
        
        try {
            $adminUsers = User::where('role', 'admin')->get();
            echo "<div class='info'>📊 Admin users: " . $adminUsers->count() . "</div>";
            
            foreach ($adminUsers->take(2) as $user) {
                echo "<div class='info'>• {$user->name} ({$user->email})</div>";
            }
            
            // Check current auth status
            if (Auth::check()) {
                $currentUser = Auth::user();
                echo "<div class='success'>✅ Currently authenticated as: {$currentUser->name}</div>";
            } else {
                echo "<div class='warning'>⚠️ Not authenticated (normal for this script)</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Auth check error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Debug 3: Check controller methods
        echo "<div class='debug-section'>";
        echo "<h2>🎮 Debug 3: Controller Methods Check</h2>";
        
        try {
            $controller = new \App\Http\Controllers\Admin\ObjekWisataController();
            echo "<div class='success'>✅ Controller loaded successfully</div>";
            
            $methods = ['edit', 'update', 'show'];
            foreach ($methods as $method) {
                if (method_exists($controller, $method)) {
                    echo "<div class='info'>📋 Method {$method}: ✅ EXISTS</div>";
                    
                    // Get method reflection for parameters
                    $reflection = new ReflectionMethod($controller, $method);
                    $params = $reflection->getParameters();
                    $paramNames = array_map(function($p) { return $p->getName(); }, $params);
                    echo "<div class='code'>Parameters: " . implode(', ', $paramNames) . "</div>";
                } else {
                    echo "<div class='error'>📋 Method {$method}: ❌ NOT FOUND</div>";
                }
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Controller check error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Debug 4: Routes check
        echo "<div class='debug-section'>";
        echo "<h2>🔗 Debug 4: Routes Check</h2>";
        
        try {
            $routes = [
                'admin.objek-wisata.index' => route('admin.objek-wisata.index'),
                'admin.objek-wisata.create' => route('admin.objek-wisata.create'),
            ];
            
            if (isset($debugInfo['tourism_id'])) {
                $routes['admin.objek-wisata.show'] = route('admin.objek-wisata.show', $debugInfo['tourism_id']);
                $routes['admin.objek-wisata.edit'] = route('admin.objek-wisata.edit', $debugInfo['tourism_id']);
                $routes['admin.objek-wisata.update'] = route('admin.objek-wisata.update', $debugInfo['tourism_id']);
            }
            
            foreach ($routes as $name => $url) {
                echo "<div class='info'><strong>{$name}:</strong> {$url}</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Routes check error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Debug 5: File system check
        echo "<div class='debug-section'>";
        echo "<h2>📁 Debug 5: File System Check</h2>";
        
        $directories = [
            'storage/app/public/wisata' => storage_path('app/public/wisata'),
            'public/storage/wisata' => public_path('storage/wisata'),
            'public/images' => public_path('images')
        ];
        
        foreach ($directories as $name => $path) {
            $exists = is_dir($path);
            $writable = $exists && is_writable($path);
            $fileCount = $exists ? count(glob($path . '/*')) : 0;
            
            echo "<div class='" . ($exists ? 'success' : 'error') . "'>";
            echo "<strong>{$name}:</strong> " . ($exists ? '✅ EXISTS' : '❌ NOT FOUND');
            if ($exists) {
                echo " | " . ($writable ? '✅ WRITABLE' : '❌ NOT WRITABLE') . " | 📁 {$fileCount} files";
            }
            echo "</div>";
        }
        echo "</div>";
        
        // Debug 6: PHP settings
        echo "<div class='debug-section'>";
        echo "<h2>⚙️ Debug 6: PHP Upload Settings</h2>";
        
        $settings = [
            'file_uploads' => ini_get('file_uploads') ? '✅ ON' : '❌ OFF',
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'max_execution_time' => ini_get('max_execution_time') . 's',
            'memory_limit' => ini_get('memory_limit'),
            'tmp_upload_dir' => ini_get('upload_tmp_dir') ?: sys_get_temp_dir()
        ];
        
        foreach ($settings as $name => $value) {
            echo "<div class='info'><strong>{$name}:</strong> {$value}</div>";
        }
        echo "</div>";
        ?>
        
        <!-- Test Form -->
        <?php if (isset($debugInfo['tourism_id'])): ?>
        <div class="form-test">
            <h2>📝 Test Update Form</h2>
            <p><strong>Test form submission to debug the actual update process:</strong></p>
            
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="tourism_id" value="<?= $debugInfo['tourism_id'] ?>">
                
                <div>
                    <label>Tourism Name:</label>
                    <input type="text" name="nama_wisata" class="form-control" value="Test Tourism Updated" required>
                </div>
                
                <div>
                    <label>Category:</label>
                    <select name="kategori_wisata" class="form-control" required>
                        <option value="wisata_alam">Wisata Alam</option>
                        <option value="wisata_budaya">Wisata Budaya</option>
                        <option value="lainnya">Lainnya</option>
                    </select>
                </div>
                
                <div>
                    <label>Description:</label>
                    <textarea name="deskripsi" class="form-control" required>Test description for debugging</textarea>
                </div>
                
                <div>
                    <label>Address:</label>
                    <input type="text" name="alamat_lengkap" class="form-control" value="Test Address" required>
                </div>
                
                <div>
                    <label>Adult Ticket Price:</label>
                    <input type="number" name="harga_tiket_dewasa" class="form-control" value="10000" required>
                </div>
                
                <div>
                    <label>Main Photo:</label>
                    <input type="file" name="foto_utama" class="form-control" accept="image/*">
                    <small>Current: <?= $debugInfo['current_photo'] ?: 'None' ?></small>
                </div>
                
                <button type="submit" class="btn">🧪 Test Update Process</button>
            </form>
        </div>
        <?php endif; ?>
        
        <div class="debug-section">
            <h2>🔧 Troubleshooting Steps</h2>
            <ol>
                <li><strong>Check authentication:</strong> Make sure you're logged in as admin</li>
                <li><strong>Check form:</strong> Verify enctype="multipart/form-data"</li>
                <li><strong>Check validation:</strong> Ensure all required fields are filled</li>
                <li><strong>Check file size:</strong> Image should be under 2MB</li>
                <li><strong>Check file type:</strong> Only JPG, PNG allowed</li>
                <li><strong>Check permissions:</strong> Storage directories should be writable</li>
                <li><strong>Check routes:</strong> Verify route parameters match controller</li>
                <li><strong>Check logs:</strong> Look at storage/logs/laravel.log for errors</li>
            </ol>
            
            <h3>🔗 Test Links</h3>
            <ul>
                <li><a href="https://susukannet.id/login" target="_blank">Login Page</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata" target="_blank">Tourism Index</a></li>
                <?php if (isset($debugInfo['tourism_id'])): ?>
                <li><a href="https://susukannet.id/admin/objek-wisata/<?= $debugInfo['tourism_id'] ?>/edit" target="_blank">Edit Tourism</a></li>
                <?php endif; ?>
            </ul>
        </div>
    </div>
</body>
</html>
