<?php
/**
 * Script debugging untuk UMKM
 * Akses via: https://susukannet.id/debug-umkm.php
 */

// Include Laravel bootstrap
require_once __DIR__ . '/../vendor/autoload.php';
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\Umkm;
use App\Helpers\ImageHelper;
use App\Helpers\SharedHostingHelper;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

echo "<h1>🔍 UMKM Debugging Script</h1>";

try {
    // 1. Database Connection Test
    echo "<h2>📊 Database Connection Test</h2>";
    try {
        $connection = DB::connection()->getPdo();
        echo "<div class='success'>✅ Database connection: OK</div>";
        echo "<div class='info'>📋 Database: " . DB::connection()->getDatabaseName() . "</div>";
    } catch (Exception $e) {
        echo "<div class='error'>❌ Database connection failed: " . $e->getMessage() . "</div>";
    }
    
    echo "<hr>";
    
    // 2. UMKM Table Check
    echo "<h2>📋 UMKM Table Check</h2>";
    try {
        $umkmCount = Umkm::count();
        echo "<div class='info'>📋 Total UMKM records: {$umkmCount}</div>";
        
        if ($umkmCount > 0) {
            $umkm = Umkm::first();
            echo "<div class='info'>📋 First UMKM ID: {$umkm->id}</div>";
            echo "<div class='info'>📋 First UMKM Name: {$umkm->nama_umkm}</div>";
        }
    } catch (Exception $e) {
        echo "<div class='error'>❌ UMKM table error: " . $e->getMessage() . "</div>";
    }
    
    echo "<hr>";
    
    // 3. Storage Check
    echo "<h2>💾 Storage Check</h2>";
    
    $storagePaths = [
        'storage/app/public' => storage_path('app/public'),
        'public/storage' => public_path('storage'),
        'storage/umkm/logo' => storage_path('app/public/umkm/logo'),
        'storage/umkm/galeri' => storage_path('app/public/umkm/galeri'),
        'public/storage/umkm/logo' => public_path('storage/umkm/logo'),
        'public/storage/umkm/galeri' => public_path('storage/umkm/galeri'),
    ];
    
    foreach ($storagePaths as $name => $path) {
        $exists = file_exists($path);
        $isDir = is_dir($path);
        $writable = is_writable($path);
        
        echo "<div class='info'>📋 {$name}: {$path}</div>";
        echo "<div class='" . ($exists ? 'success' : 'error') . "'>" . ($exists ? '✅' : '❌') . " Exists: " . ($exists ? 'YES' : 'NO') . "</div>";
        
        if ($exists) {
            echo "<div class='" . ($isDir ? 'success' : 'error') . "'>" . ($isDir ? '✅' : '❌') . " Is Directory: " . ($isDir ? 'YES' : 'NO') . "</div>";
            echo "<div class='" . ($writable ? 'success' : 'error') . "'>" . ($writable ? '✅' : '❌') . " Writable: " . ($writable ? 'YES' : 'NO') . "</div>";
            
            if ($isDir) {
                $files = glob($path . '/*');
                echo "<div class='info'>📋 Files count: " . count($files) . "</div>";
                
                if (count($files) > 0) {
                    echo "<div class='info'>📋 Sample files:</div>";
                    foreach (array_slice($files, 0, 3) as $file) {
                        $filename = basename($file);
                        $size = filesize($file);
                        echo "<div class='info'>  - {$filename} ({$size} bytes)</div>";
                    }
                }
            }
        }
        echo "<br>";
    }
    
    echo "<hr>";
    
    // 4. Sample Files Check
    echo "<h2>🖼️ Sample Files Check</h2>";
    
    $sampleFiles = [
        'umkm/logo/sample-logo.png',
        'umkm/galeri/sample-gallery1.png'
    ];
    
    foreach ($sampleFiles as $file) {
        $storagePath = storage_path('app/public/' . $file);
        $publicPath = public_path('storage/' . $file);
        
        echo "<div class='info'>📋 Testing: {$file}</div>";
        
        $storageExists = file_exists($storagePath);
        $publicExists = file_exists($publicPath);
        
        echo "<div class='" . ($storageExists ? 'success' : 'error') . "'>" . ($storageExists ? '✅' : '❌') . " Storage: " . ($storageExists ? 'EXISTS' : 'NOT FOUND') . "</div>";
        echo "<div class='" . ($publicExists ? 'success' : 'error') . "'>" . ($publicExists ? '✅' : '❌') . " Public: " . ($publicExists ? 'EXISTS' : 'NOT FOUND') . "</div>";
        
        if ($storageExists) {
            echo "<div class='info'>📋 Storage size: " . filesize($storagePath) . " bytes</div>";
        }
        
        if ($publicExists) {
            echo "<div class='info'>📋 Public size: " . filesize($publicPath) . " bytes</div>";
        }
        
        // Test URL generation
        $url = 'https://susukannet.id/storage/' . $file;
        echo "<div class='info'>📋 Test URL: <a href='{$url}' target='_blank'>{$url}</a></div>";
        
        echo "<br>";
    }
    
    echo "<hr>";
    
    // 5. Helper Classes Test
    echo "<h2>🔧 Helper Classes Test</h2>";
    
    // Test ImageHelper
    echo "<h3>ImageHelper Test</h3>";
    $testInputs = [
        'string' => 'umkm/logo/sample-logo.png',
        'array_filename' => ['filename' => 'umkm/logo/sample-logo.png'],
        'array_index' => ['umkm/logo/sample-logo.png'],
        'null' => null,
        'empty' => ''
    ];
    
    foreach ($testInputs as $type => $input) {
        try {
            $result = ImageHelper::getImageUrl($input);
            echo "<div class='success'>✅ {$type}: {$result}</div>";
        } catch (Exception $e) {
            echo "<div class='error'>❌ {$type} Error: " . $e->getMessage() . "</div>";
        }
    }
    
    // Test SharedHostingHelper
    echo "<h3>SharedHostingHelper Test</h3>";
    try {
        $logoUrl = SharedHostingHelper::getUmkmLogoUrl('sample-logo.png');
        echo "<div class='success'>✅ Logo URL: {$logoUrl}</div>";
        
        $galleryUrl = SharedHostingHelper::getUmkmGalleryUrl('sample-gallery1.png');
        echo "<div class='success'>✅ Gallery URL: {$galleryUrl}</div>";
    } catch (Exception $e) {
        echo "<div class='error'>❌ SharedHostingHelper Error: " . $e->getMessage() . "</div>";
    }
    
    echo "<hr>";
    
    // 6. UMKM Model Test
    echo "<h2>🏢 UMKM Model Test</h2>";
    
    if ($umkmCount > 0) {
        $umkm = Umkm::first();
        
        echo "<div class='info'>📋 Testing UMKM: {$umkm->nama_umkm} (ID: {$umkm->id})</div>";
        
        // Test logo
        echo "<h3>Logo Test</h3>";
        try {
            $logoUrl = $umkm->logo_url;
            echo "<div class='success'>✅ Logo URL: <a href='{$logoUrl}' target='_blank'>{$logoUrl}</a></div>";
        } catch (Exception $e) {
            echo "<div class='error'>❌ Logo URL Error: " . $e->getMessage() . "</div>";
        }
        
        // Test gallery
        echo "<h3>Gallery Test</h3>";
        try {
            $galleryUrls = $umkm->galeri_foto_urls;
            echo "<div class='success'>✅ Gallery URLs: " . count($galleryUrls) . " items</div>";
            
            foreach ($galleryUrls as $index => $url) {
                echo "<div class='info'>📋 Gallery {$index}: <a href='{$url}' target='_blank'>{$url}</a></div>";
            }
        } catch (Exception $e) {
            echo "<div class='error'>❌ Gallery URLs Error: " . $e->getMessage() . "</div>";
        }
        
        // Raw data inspection
        echo "<h3>Raw Data Inspection</h3>";
        echo "<div class='info'>📋 Logo field type: " . gettype($umkm->logo) . "</div>";
        echo "<div class='info'>📋 Logo field value: " . json_encode($umkm->logo) . "</div>";
        echo "<div class='info'>📋 Galeri foto field type: " . gettype($umkm->galeri_foto) . "</div>";
        echo "<div class='info'>📋 Galeri foto field value: " . json_encode($umkm->galeri_foto) . "</div>";
        
    } else {
        echo "<div class='error'>❌ No UMKM records found for testing</div>";
    }
    
    echo "<hr>";
    
    // 7. Create Test UMKM
    echo "<h2>🧪 Create Test UMKM</h2>";
    
    try {
        $testUmkm = new Umkm();
        $testUmkm->nama_umkm = 'Test UMKM Debug';
        $testUmkm->kategori = 'makanan_minuman';
        $testUmkm->deskripsi = 'Test UMKM for debugging';
        $testUmkm->alamat = 'Test Address';
        $testUmkm->skala_usaha = 'mikro';
        $testUmkm->status = 'aktif';
        $testUmkm->logo = 'umkm/logo/sample-logo.png';
        $testUmkm->galeri_foto = ['umkm/galeri/sample-gallery1.png'];
        
        // Test accessors without saving
        $logoUrl = $testUmkm->logo_url;
        $galleryUrls = $testUmkm->galeri_foto_urls;
        
        echo "<div class='success'>✅ Test UMKM created successfully</div>";
        echo "<div class='success'>✅ Logo URL: {$logoUrl}</div>";
        echo "<div class='success'>✅ Gallery URLs: " . count($galleryUrls) . " items</div>";
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Test UMKM creation failed: " . $e->getMessage() . "</div>";
    }
    
    echo "<hr>";
    
    // 8. Environment Check
    echo "<h2>🌍 Environment Check</h2>";
    
    echo "<div class='info'>📋 App Environment: " . app()->environment() . "</div>";
    echo "<div class='info'>📋 App Debug: " . (config('app.debug') ? 'ON' : 'OFF') . "</div>";
    echo "<div class='info'>📋 App URL: " . config('app.url') . "</div>";
    echo "<div class='info'>📋 Storage Link: " . (file_exists(public_path('storage')) ? 'EXISTS' : 'NOT FOUND') . "</div>";
    
    // Check PHP extensions
    $requiredExtensions = ['gd', 'fileinfo', 'mbstring'];
    foreach ($requiredExtensions as $ext) {
        $loaded = extension_loaded($ext);
        echo "<div class='" . ($loaded ? 'success' : 'error') . "'>" . ($loaded ? '✅' : '❌') . " PHP {$ext}: " . ($loaded ? 'LOADED' : 'NOT LOADED') . "</div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>❌ General Error: " . $e->getMessage() . "</div>";
    echo "<div class='error'>❌ File: " . $e->getFile() . " Line: " . $e->getLine() . "</div>";
    echo "<div class='error'>❌ Stack trace:</div>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
}

echo "<h2>✅ Debugging Completed</h2>";
echo "<p><a href='/admin/umkm'>Kembali ke Admin UMKM</a></p>";

?>
<style>
body { font-family: Arial, sans-serif; margin: 20px; }
.success { color: green; font-weight: bold; background: #d4edda; padding: 5px; border-radius: 3px; margin: 2px 0; }
.error { color: red; font-weight: bold; background: #f8d7da; padding: 5px; border-radius: 3px; margin: 2px 0; }
.info { color: blue; background: #d1ecf1; padding: 5px; border-radius: 3px; margin: 2px 0; }
pre { background: #f8f9fa; padding: 10px; border-radius: 3px; overflow-x: auto; }
</style>
