<?php
/**
 * Fix Tourism Images - Clean up temporary file paths
 * This script fixes tourism images that have temporary file paths saved in the database
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel application
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Tourism Images - Clean Temporary Paths</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        .step { margin: 15px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
        pre { background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto; }
        .status-box { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .status-success { background: #d4edda; border-color: #c3e6cb; color: #155724; }
        .status-error { background: #f8d7da; border-color: #f5c6cb; color: #721c24; }
        .status-warning { background: #fff3cd; border-color: #ffeaa7; color: #856404; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🖼️ Fix Tourism Images - Clean Temporary Paths</h1>
        <p>This tool fixes tourism images that have temporary file paths in the database.</p>

        <?php
        echo "<h2>🔍 Checking Tourism Data</h2>";
        
        try {
            $totalTourism = ObjekWisata::count();
            $tourismWithImages = ObjekWisata::whereNotNull('foto_utama')->count();
            $tourismWithTempPaths = ObjekWisata::where('foto_utama', 'like', '%tmp%')->count();
            
            echo "<div class='status-box status-success'>✅ Total tourism objects: {$totalTourism}</div>";
            echo "<div class='status-box status-success'>✅ Tourism with images: {$tourismWithImages}</div>";
            echo "<div class='status-box status-warning'>⚠️ Tourism with temporary paths: {$tourismWithTempPaths}</div>";
            
            if ($tourismWithTempPaths > 0) {
                echo "<h2>🔧 Fixing Temporary Paths</h2>";
                
                $affectedTourism = ObjekWisata::where('foto_utama', 'like', '%tmp%')->get();
                
                $fixedCount = 0;
                
                foreach ($affectedTourism as $tourism) {
                    echo "<div class='step'>";
                    echo "<h4>{$tourism->nama_wisata}</h4>";
                    echo "<p><strong>Current foto_utama:</strong> {$tourism->foto_utama}</p>";
                    
                    // Clear the temporary path
                    $updated = $tourism->update(['foto_utama' => null]);
                    
                    if ($updated) {
                        echo "<p class='success'>✅ Cleared temporary path</p>";
                        $fixedCount++;
                    } else {
                        echo "<p class='error'>❌ Failed to clear temporary path</p>";
                    }
                    
                    echo "</div>";
                }
                
                echo "<div class='status-box status-success'>✅ Fixed {$fixedCount} tourism objects</div>";
            } else {
                echo "<div class='status-box status-success'>✅ No tourism objects with temporary paths found</div>";
            }
            
            // Check for galeri_foto issues
            echo "<h2>🖼️ Checking Gallery Images</h2>";
            
            $tourismWithGallery = ObjekWisata::whereNotNull('galeri_foto')->get();
            $galleryFixedCount = 0;
            
            foreach ($tourismWithGallery as $tourism) {
                $galeri = $tourism->galeri_foto;
                $needsFix = false;
                $cleanGaleri = [];
                
                if (is_array($galeri)) {
                    foreach ($galeri as $foto) {
                        if (strpos($foto, 'tmp') !== false) {
                            $needsFix = true;
                        } else {
                            $cleanGaleri[] = $foto;
                        }
                    }
                    
                    if ($needsFix) {
                        echo "<div class='step'>";
                        echo "<h4>{$tourism->nama_wisata}</h4>";
                        echo "<p><strong>Original gallery count:</strong> " . count($galeri) . "</p>";
                        echo "<p><strong>Clean gallery count:</strong> " . count($cleanGaleri) . "</p>";
                        
                        $updated = $tourism->update(['galeri_foto' => $cleanGaleri]);
                        
                        if ($updated) {
                            echo "<p class='success'>✅ Cleaned gallery images</p>";
                            $galleryFixedCount++;
                        } else {
                            echo "<p class='error'>❌ Failed to clean gallery images</p>";
                        }
                        
                        echo "</div>";
                    }
                }
            }
            
            echo "<div class='status-box status-success'>✅ Fixed gallery for {$galleryFixedCount} tourism objects</div>";
            
            // Create sample tourism data if none exists
            echo "<h2>📝 Sample Tourism Data</h2>";
            
            if ($totalTourism === 0) {
                echo "<div class='status-box status-warning'>⚠️ No tourism data found. Creating sample data...</div>";
                
                $sampleData = [
                    [
                        'nama_wisata' => 'Situ Rawa Gede',
                        'slug' => 'situ-rawa-gede',
                        'kategori_wisata' => 'wisata_alam',
                        'deskripsi' => 'Danau alami yang indah dengan pemandangan pegunungan. Tempat yang sempurna untuk bersantai dan menikmati keindahan alam.',
                        'alamat' => 'Desa Susukan, Kecamatan Susukan',
                        'harga_tiket_dewasa' => 10000,
                        'harga_tiket_anak' => 5000,
                        'latitude' => -6.123456,
                        'longitude' => 106.123456,
                        'status' => 'aktif',
                        'is_featured' => true,
                        'user_id' => 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ],
                    [
                        'nama_wisata' => 'Curug Cigamea',
                        'slug' => 'curug-cigamea',
                        'kategori_wisata' => 'wisata_alam',
                        'deskripsi' => 'Air terjun yang menawan dengan kolam alami yang jernih. Lokasi yang ideal untuk berenang dan berfoto.',
                        'alamat' => 'Desa Susukan, Kecamatan Susukan',
                        'harga_tiket_dewasa' => 15000,
                        'harga_tiket_anak' => 7500,
                        'latitude' => -6.124456,
                        'longitude' => 106.124456,
                        'status' => 'aktif',
                        'is_featured' => true,
                        'user_id' => 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ],
                    [
                        'nama_wisata' => 'Kebun Teh Hijau',
                        'slug' => 'kebun-teh-hijau',
                        'kategori_wisata' => 'agrowisata',
                        'deskripsi' => 'Perkebunan teh yang hijau dan sejuk. Wisatawan dapat belajar tentang proses pengolahan teh sambil menikmati udara segar pegunungan.',
                        'alamat' => 'Desa Susukan, Kecamatan Susukan',
                        'harga_tiket_dewasa' => 12000,
                        'harga_tiket_anak' => 6000,
                        'latitude' => -6.125456,
                        'longitude' => 106.125456,
                        'status' => 'aktif',
                        'is_featured' => false,
                        'user_id' => 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ],
                ];
                
                $createdCount = 0;
                foreach ($sampleData as $data) {
                    try {
                        ObjekWisata::create($data);
                        $createdCount++;
                        echo "<div class='status-box status-success'>✅ Created: {$data['nama_wisata']}</div>";
                    } catch (Exception $e) {
                        echo "<div class='status-box status-error'>❌ Failed to create {$data['nama_wisata']}: {$e->getMessage()}</div>";
                    }
                }
                
                echo "<div class='status-box status-success'>✅ Created {$createdCount} sample tourism objects</div>";
            } else {
                echo "<div class='status-box status-success'>✅ Tourism data already exists</div>";
            }
            
            // Final check
            echo "<h2>✅ Final Status</h2>";
            
            $finalTotalTourism = ObjekWisata::count();
            $finalTourismWithImages = ObjekWisata::whereNotNull('foto_utama')->count();
            $finalTourismWithTempPaths = ObjekWisata::where('foto_utama', 'like', '%tmp%')->count();
            
            echo "<div class='status-box status-success'>✅ Total tourism objects: {$finalTotalTourism}</div>";
            echo "<div class='status-box status-success'>✅ Tourism with valid images: {$finalTourismWithImages}</div>";
            echo "<div class='status-box status-success'>✅ Tourism with temporary paths: {$finalTourismWithTempPaths}</div>";
            
        } catch (Exception $e) {
            echo "<div class='status-box status-error'>❌ Error: " . $e->getMessage() . "</div>";
        }
        
        echo "<h2>🧪 Test URLs</h2>";
        
        $testUrls = [
            'Tourism Page' => '/wisata',
            'Admin Tourism' => '/admin/objek-wisata',
            'Tourism Images Fix' => '/fix-wisata-images-shared.php',
        ];
        
        foreach ($testUrls as $name => $url) {
            $fullUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . $url;
            echo "<div class='step'>";
            echo "<strong>$name:</strong> <a href='$fullUrl' target='_blank'>$fullUrl</a>";
            echo "</div>";
        }
        
        echo "<h2>📝 Next Steps</h2>";
        echo "<div class='step'>";
        echo "<h4>1. Upload Tourism Images</h4>";
        echo "<p>Upload proper tourism images through the admin panel:</p>";
        echo "<ul>";
        echo "<li>Go to Admin → Objek Wisata → Edit</li>";
        echo "<li>Upload foto_utama and galeri_foto</li>";
        echo "<li>Images will be stored in storage/app/public/wisata/</li>";
        echo "</ul>";
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h4>2. Test Image Display</h4>";
        echo "<p>Check if images display correctly on the frontend:</p>";
        echo "<ul>";
        echo "<li>Visit the <a href='/wisata' target='_blank'>Tourism Page</a></li>";
        echo "<li>Check if default images show when no image is uploaded</li>";
        echo "<li>Upload real images and verify they display</li>";
        echo "</ul>";
        echo "</div>";
        ?>
    </div>
</body>
</html>
