<?php
/**
 * Script untuk memperbaiki masalah gambar UMKM di shared hosting
 * Jalankan script ini setelah deploy ke shared hosting
 */

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>=== UMKM Images Fix Script for Shared Hosting ===</h2>";
echo "<p>Memperbaiki masalah gambar UMKM di shared hosting...</p>";

// Paths
$storagePath = __DIR__ . '/../storage/app/public';
$publicStoragePath = __DIR__ . '/storage';
$publicImagesPath = __DIR__ . '/images';

echo "<h3>1. Checking Directories</h3>";

// Create directories if they don't exist
$directories = [
    $storagePath . '/umkm',
    $storagePath . '/umkm/logo',
    $storagePath . '/umkm/galeri',
    $storagePath . '/produk',
    $storagePath . '/produk/galeri',
    $publicStoragePath . '/umkm',
    $publicStoragePath . '/umkm/logo', 
    $publicStoragePath . '/umkm/galeri',
    $publicStoragePath . '/produk',
    $publicStoragePath . '/produk/galeri'
];

foreach ($directories as $dir) {
    if (!is_dir($dir)) {
        if (mkdir($dir, 0755, true)) {
            echo "<p style='color: green;'>✓ Created directory: " . basename($dir) . "</p>";
        } else {
            echo "<p style='color: red;'>✗ Failed to create directory: " . basename($dir) . "</p>";
        }
    } else {
        echo "<p style='color: blue;'>✓ Directory exists: " . basename($dir) . "</p>";
    }
}

echo "<h3>2. Creating .htaccess Files</h3>";

// Create .htaccess files for image access
$htaccessContent = "Options -Indexes\n<Files \"*\">\n    Require all granted\n</Files>\n<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|svg)$\">\n    Require all granted\n</FilesMatch>\n";

$htaccessDirs = [
    $storagePath . '/umkm',
    $storagePath . '/umkm/logo',
    $storagePath . '/umkm/galeri',
    $storagePath . '/produk',
    $storagePath . '/produk/galeri',
    $publicStoragePath . '/umkm',
    $publicStoragePath . '/umkm/logo',
    $publicStoragePath . '/umkm/galeri',
    $publicStoragePath . '/produk',
    $publicStoragePath . '/produk/galeri'
];

foreach ($htaccessDirs as $dir) {
    $htaccessFile = $dir . '/.htaccess';
    if (!file_exists($htaccessFile)) {
        if (file_put_contents($htaccessFile, $htaccessContent)) {
            echo "<p style='color: green;'>✓ Created .htaccess: " . basename($dir) . "</p>";
        } else {
            echo "<p style='color: red;'>✗ Failed to create .htaccess: " . basename($dir) . "</p>";
        }
    } else {
        echo "<p style='color: blue;'>✓ .htaccess exists: " . basename($dir) . "</p>";
    }
}

echo "<h3>3. Checking Storage Symlink</h3>";

// Check storage symlink
if (is_link($publicStoragePath)) {
    echo "<p style='color: green;'>✓ Storage symlink exists</p>";
} else {
    echo "<p style='color: orange;'>! Storage symlink not found</p>";
    echo "<p>Creating manual copy of storage files...</p>";
    
    // Copy storage files to public/storage if symlink doesn't work
    if (is_dir($storagePath)) {
        $source = $storagePath;
        $dest = $publicStoragePath;
        
        if (!is_dir($dest)) {
            mkdir($dest, 0755, true);
        }
        
        // Copy UMKM directories
        $umkmDirs = ['umkm', 'produk'];
        foreach ($umkmDirs as $dir) {
            $sourceDir = $source . '/' . $dir;
            $destDir = $dest . '/' . $dir;
            
            if (is_dir($sourceDir)) {
                if (!is_dir($destDir)) {
                    mkdir($destDir, 0755, true);
                }
                
                // Copy files recursively
                $iterator = new RecursiveIteratorIterator(
                    new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
                    RecursiveIteratorIterator::SELF_FIRST
                );
                
                foreach ($iterator as $item) {
                    $destPath = $destDir . DIRECTORY_SEPARATOR . $iterator->getSubPathName();
                    
                    if ($item->isDir()) {
                        if (!is_dir($destPath)) {
                            mkdir($destPath, 0755, true);
                        }
                    } else {
                        if (!file_exists($destPath)) {
                            copy($item, $destPath);
                        }
                    }
                }
                
                echo "<p style='color: green;'>✓ Copied " . $dir . " directory</p>";
            }
        }
    }
}

echo "<h3>4. Checking Default Images</h3>";

// Check if default images exist
$defaultImages = [
    $publicImagesPath . '/default-umkm-logo.png',
    $publicImagesPath . '/default-product.png'
];

foreach ($defaultImages as $image) {
    if (file_exists($image)) {
        echo "<p style='color: green;'>✓ Default image exists: " . basename($image) . "</p>";
    } else {
        echo "<p style='color: red;'>✗ Default image missing: " . basename($image) . "</p>";
        
        // Create a simple default image if missing
        if (basename($image) === 'default-umkm-logo.png') {
            $imageData = base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==');
            if (file_put_contents($image, $imageData)) {
                echo "<p style='color: green;'>✓ Created default image: " . basename($image) . "</p>";
            }
        }
    }
}

echo "<h3>5. Testing Image URLs</h3>";

// Test image URLs
$testUrls = [
    '/storage/umkm/logo/',
    '/storage/umkm/galeri/',
    '/images/default-umkm-logo.png'
];

foreach ($testUrls as $url) {
    $fullUrl = 'https://' . $_SERVER['HTTP_HOST'] . $url;
    echo "<p>Testing: <a href='" . $fullUrl . "' target='_blank'>" . $url . "</a></p>";
}

echo "<h3>6. Database Check</h3>";

// Check database connection and UMKM data
try {
    // Simple database check without Laravel
    $config = include(__DIR__ . '/../config/database.php');
    $mysql = $config['connections']['mysql'];
    
    $pdo = new PDO(
        "mysql:host={$mysql['host']};dbname={$mysql['database']};charset=utf8mb4",
        $mysql['username'],
        $mysql['password'],
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM umkm");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    echo "<p style='color: green;'>✓ Database connected. Found " . $result['count'] . " UMKM records</p>";
    
    // Check UMKM with images
    $stmt = $pdo->query("SELECT id, nama_umkm, logo, galeri_foto FROM umkm WHERE logo IS NOT NULL OR galeri_foto IS NOT NULL LIMIT 5");
    $umkms = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($umkms) {
        echo "<p>UMKM with images:</p><ul>";
        foreach ($umkms as $umkm) {
            echo "<li>" . htmlspecialchars($umkm['nama_umkm']) . " (ID: " . $umkm['id'] . ")</li>";
        }
        echo "</ul>";
    } else {
        echo "<p style='color: orange;'>No UMKM with images found</p>";
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>✗ Database error: " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "<h3>7. Recommendations</h3>";
echo "<ul>";
echo "<li>Make sure storage symlink is working: <code>php artisan storage:link</code></li>";
echo "<li>Check file permissions: <code>chmod -R 755 storage/ public/storage/</code></li>";
echo "<li>Upload some test images to verify the setup</li>";
echo "<li>Check .htaccess files in storage directories</li>";
echo "</ul>";

echo "<h3>=== Setup Complete ===</h3>";
echo "<p>UMKM storage directories are ready! If images still don't show, check the browser console for specific 404 errors.</p>";
?>
