<?php
/**
 * Tourism Images Fix for Shared Hosting
 * This script fixes tourism image display issues on shared hosting
 */

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Tourism Images - Shared Hosting</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        .step { margin: 15px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
        pre { background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto; }
        .status-box { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .status-success { background: #d4edda; border-color: #c3e6cb; color: #155724; }
        .status-error { background: #f8d7da; border-color: #f5c6cb; color: #721c24; }
        .status-warning { background: #fff3cd; border-color: #ffeaa7; color: #856404; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🖼️ Tourism Images Fix - Shared Hosting</h1>
        <p>This tool fixes tourism image display issues on shared hosting environments.</p>

        <?php
        $baseDir = dirname(__DIR__);
        $storageDir = $baseDir . '/storage/app/public';
        $publicStorageDir = $baseDir . '/public/storage';
        
        echo "<h2>🔍 System Check</h2>";
        
        // Check directories
        $checks = [
            'Storage Directory' => is_dir($storageDir),
            'Public Storage Directory' => is_dir($publicStorageDir),
            'Tourism Storage Directory' => is_dir($storageDir . '/wisata'),
            'Tourism Public Directory' => is_dir($publicStorageDir . '/wisata'),
            'Tourism Galeri Storage' => is_dir($storageDir . '/wisata/galeri'),
            'Tourism Galeri Public' => is_dir($publicStorageDir . '/wisata/galeri'),
            'Default Tourism Image' => file_exists($baseDir . '/public/images/default-tourism.png'),
        ];
        
        foreach ($checks as $name => $status) {
            $class = $status ? 'status-success' : 'status-error';
            $icon = $status ? '✅' : '❌';
            echo "<div class='status-box $class'>$icon $name: " . ($status ? 'OK' : 'NOT FOUND') . "</div>";
        }

        echo "<h2>🔧 Fix Actions</h2>";

        // Create missing directories
        $dirsToCreate = [
            $storageDir . '/wisata',
            $storageDir . '/wisata/galeri', 
            $publicStorageDir . '/wisata',
            $publicStorageDir . '/wisata/galeri'
        ];

        foreach ($dirsToCreate as $dir) {
            if (!is_dir($dir)) {
                if (mkdir($dir, 0755, true)) {
                    echo "<div class='status-box status-success'>✅ Created directory: $dir</div>";
                } else {
                    echo "<div class='status-box status-error'>❌ Failed to create directory: $dir</div>";
                }
            } else {
                echo "<div class='status-box status-success'>✅ Directory already exists: $dir</div>";
            }
        }

        // Create default tourism image if not exists
        $defaultImageSrc = $baseDir . '/public/images/bg-pettren.png';
        $defaultImageDest = $baseDir . '/public/images/default-tourism.png';
        
        if (!file_exists($defaultImageDest) && file_exists($defaultImageSrc)) {
            if (copy($defaultImageSrc, $defaultImageDest)) {
                echo "<div class='status-box status-success'>✅ Created default tourism image</div>";
            } else {
                echo "<div class='status-box status-error'>❌ Failed to create default tourism image</div>";
            }
        }

        // Check for tourism data
        try {
            if (file_exists($baseDir . '/vendor/autoload.php')) {
                require_once $baseDir . '/vendor/autoload.php';
                
                // Try to load Laravel app
                $app = require_once $baseDir . '/bootstrap/app.php';
                $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
                
                echo "<h2>📊 Tourism Data Check</h2>";
                
                $tourismCount = \App\Models\ObjekWisata::count();
                $activeTourismCount = \App\Models\ObjekWisata::where('status', 'aktif')->count();
                $tourismWithImages = \App\Models\ObjekWisata::whereNotNull('foto_utama')->count();
                
                echo "<div class='status-box status-success'>✅ Total tourism objects: $tourismCount</div>";
                echo "<div class='status-box status-success'>✅ Active tourism objects: $activeTourismCount</div>";
                echo "<div class='status-box status-success'>✅ Tourism with images: $tourismWithImages</div>";
                
                // List some tourism objects for testing
                $sampleTourism = \App\Models\ObjekWisata::where('status', 'aktif')->limit(3)->get();
                
                echo "<h3>🎯 Sample Tourism Objects</h3>";
                
                foreach ($sampleTourism as $tourism) {
                    echo "<div class='step'>";
                    echo "<h4>{$tourism->nama_wisata}</h4>";
                    echo "<p><strong>Slug:</strong> {$tourism->slug}</p>";
                    echo "<p><strong>Image:</strong> " . ($tourism->foto_utama ?: 'No image') . "</p>";
                    echo "<p><strong>Image URL:</strong> <a href='{$tourism->foto_utama_url}' target='_blank'>{$tourism->foto_utama_url}</a></p>";
                    
                    // Test if image exists
                    if ($tourism->foto_utama) {
                        $imagePath = $storageDir . '/wisata/' . $tourism->foto_utama;
                        $publicImagePath = $publicStorageDir . '/wisata/' . $tourism->foto_utama;
                        
                        echo "<p><strong>Storage Image:</strong> " . (file_exists($imagePath) ? '✅ Exists' : '❌ Missing') . "</p>";
                        echo "<p><strong>Public Image:</strong> " . (file_exists($publicImagePath) ? '✅ Exists' : '❌ Missing') . "</p>";
                        
                        // Try to copy image if exists in storage but not in public
                        if (file_exists($imagePath) && !file_exists($publicImagePath)) {
                            $publicDir = dirname($publicImagePath);
                            if (!is_dir($publicDir)) {
                                mkdir($publicDir, 0755, true);
                            }
                            if (copy($imagePath, $publicImagePath)) {
                                echo "<p class='success'>✅ Copied image to public directory</p>";
                            } else {
                                echo "<p class='error'>❌ Failed to copy image</p>";
                            }
                        }
                    }
                    echo "</div>";
                }
                
            } else {
                echo "<div class='status-box status-warning'>⚠️ Laravel autoloader not found. Cannot check tourism data.</div>";
            }
        } catch (Exception $e) {
            echo "<div class='status-box status-error'>❌ Error checking tourism data: " . $e->getMessage() . "</div>";
        }

        echo "<h2>🧪 Test URLs</h2>";
        
        $testUrls = [
            'Tourism Page' => '/wisata',
            'Default Tourism Image' => '/images/default-tourism.png',
            'Storage Test' => '/storage/wisata/',
        ];
        
        foreach ($testUrls as $name => $url) {
            $fullUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . $url;
            echo "<div class='step'>";
            echo "<strong>$name:</strong> <a href='$fullUrl' target='_blank'>$fullUrl</a>";
            echo "</div>";
        }

        echo "<h2>📝 Manual Steps</h2>";
        echo "<div class='step'>";
        echo "<h4>1. Run Laravel Commands (via SSH/Terminal)</h4>";
        echo "<pre>php artisan storage:link
php artisan cache:clear
php artisan config:clear</pre>";
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h4>2. Set Proper Permissions</h4>";
        echo "<pre>chmod -R 755 public/storage/wisata
chmod -R 755 storage/app/public/wisata</pre>";
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h4>3. Upload Sample Images</h4>";
        echo "<p>Upload some sample tourism images to:</p>";
        echo "<ul>";
        echo "<li><code>storage/app/public/wisata/</code></li>";
        echo "<li><code>public/storage/wisata/</code></li>";
        echo "</ul>";
        echo "</div>";

        echo "<h2>✅ Completion</h2>";
        echo "<p>After running this script:</p>";
        echo "<ol>";
        echo "<li>Visit the <a href='/wisata' target='_blank'>Tourism Page</a> to test image display</li>";
        echo "<li>Check browser console for any 404 errors on images</li>";
        echo "<li>Upload actual tourism images to the correct directories</li>";
        echo "</ol>";
        ?>
    </div>
</body>
</html>
