/**
 * Admin Session Management
 * Handles CSRF token refresh and session management for admin pages
 */

class AdminSessionManager {
    constructor() {
        this.csrfToken = null;
        this.refreshInterval = null;
        this.lastActivity = Date.now();
        this.sessionTimeout = 30 * 60 * 1000; // 30 minutes in milliseconds
        this.refreshThreshold = 5 * 60 * 1000; // 5 minutes before expiry
        
        this.init();
    }

    init() {
        // Get initial CSRF token
        this.getCsrfToken();
        
        // Set up activity tracking
        this.setupActivityTracking();
        
        // Set up periodic refresh
        this.setupPeriodicRefresh();
        
        // Set up beforeunload handler
        this.setupBeforeUnload();
        
        // Set up global form submission handler
        this.setupGlobalFormHandler();
        
        console.log('Admin Session Manager initialized');
    }

    getCsrfToken() {
        const token = document.querySelector('meta[name="csrf-token"]');
        if (token) {
            this.csrfToken = token.getAttribute('content');
            this.updateCsrfTokenInForms();
        }
    }

    updateCsrfTokenInForms() {
        if (!this.csrfToken) return;
        
        // Update meta tag
        const metaToken = document.querySelector('meta[name="csrf-token"]');
        if (metaToken) {
            metaToken.setAttribute('content', this.csrfToken);
        }
        
        // Update all forms with new CSRF token
        const forms = document.querySelectorAll('form');
        forms.forEach(form => {
            let tokenInput = form.querySelector('input[name="_token"]');
            if (tokenInput) {
                tokenInput.value = this.csrfToken;
            } else {
                // Create token input if it doesn't exist
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = '_token';
                input.value = this.csrfToken;
                form.appendChild(input);
            }
        });
        
        // Update AJAX headers
        if (window.axios) {
            window.axios.defaults.headers.common['X-CSRF-TOKEN'] = this.csrfToken;
        }
        
        // Update jQuery AJAX if available
        if (window.$ && window.$.ajaxSetup) {
            window.$.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': this.csrfToken
                }
            });
        }
        
        // Update fetch requests globally
        this.setupFetchInterceptor();
        
        // Setup global AJAX error handler
        this.setupAjaxErrorHandler();
    }
    
    setupFetchInterceptor() {
        // Store original fetch
        if (!window.originalFetch) {
            window.originalFetch = window.fetch;
        }
        
        // Override fetch to include CSRF token
        window.fetch = (url, options = {}) => {
            // Only add CSRF token for same-origin requests
            if (typeof url === 'string' && url.startsWith('/')) {
                options.headers = options.headers || {};
                if (!options.headers['X-CSRF-TOKEN']) {
                    options.headers['X-CSRF-TOKEN'] = this.csrfToken;
                }
            }
            return window.originalFetch(url, options);
        };
    }

    setupActivityTracking() {
        const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'];
        
        events.forEach(event => {
            document.addEventListener(event, () => {
                this.lastActivity = Date.now();
            }, true);
        });
    }

    setupPeriodicRefresh() {
        // Check every 5 minutes
        this.refreshInterval = setInterval(() => {
            this.checkAndRefreshSession();
        }, 5 * 60 * 1000);
    }

    async checkAndRefreshSession() {
        const timeSinceActivity = Date.now() - this.lastActivity;
        
        // If user has been inactive for more than session timeout, don't refresh
        if (timeSinceActivity > this.sessionTimeout) {
            console.log('User inactive, skipping session refresh');
            return;
        }
        
        try {
            const response = await fetch('/admin/refresh-session', {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin'
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.csrfToken = data.csrf_token;
                    this.updateCsrfTokenInForms();
                    console.log('Session refreshed successfully');
                }
            } else if (response.status === 419) {
                // CSRF token mismatch - redirect to login
                this.handleSessionExpired();
            }
        } catch (error) {
            console.error('Error refreshing session:', error);
        }
    }
    
    // Handle form submission errors
    handleFormSubmissionError(error, form) {
        console.error('Form submission error:', error);
        
        if (error.status === 419) {
            // CSRF token mismatch
            this.showSessionExpiredNotification();
            return true; // Indicates error was handled
        }
        
        return false; // Error not handled
    }
    
    showSessionExpiredNotification() {
        if (window.Swal) {
            Swal.fire({
                title: 'Sesi Berakhir',
                text: 'Sesi Anda telah berakhir. Halaman akan di-refresh untuk memperbarui token.',
                icon: 'warning',
                confirmButtonText: 'OK',
                allowOutsideClick: false,
                allowEscapeKey: false
            }).then(() => {
                window.location.reload();
            });
        } else {
            alert('Sesi Anda telah berakhir. Halaman akan di-refresh.');
            window.location.reload();
        }
    }
    
    // Retry form submission after session refresh
    async retryFormSubmission(formData, originalUrl) {
        try {
            const response = await fetch('/admin/retry-form', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': this.csrfToken,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    form_data: formData,
                    original_url: originalUrl
                })
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.csrfToken = data.csrf_token;
                    this.updateCsrfTokenInForms();
                    
                    // Show success message
                    if (window.Swal) {
                        Swal.fire({
                            title: 'Token Diperbarui',
                            text: 'Session telah diperbarui. Silakan coba submit form lagi.',
                            icon: 'success',
                            confirmButtonText: 'OK'
                        });
                    }
                    
                    return true;
                }
            }
        } catch (error) {
            console.error('Error retrying form submission:', error);
        }
        
        return false;
    }

    async refreshSession() {
        try {
            const response = await fetch('/admin/refresh-session', {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin'
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.csrfToken = data.csrf_token;
                    this.updateCsrfTokenInForms();
                    return true;
                }
            }
        } catch (error) {
            console.error('Error refreshing session:', error);
        }
        return false;
    }

    handleSessionExpired() {
        // Show notification
        if (window.Swal) {
            Swal.fire({
                title: 'Sesi Berakhir',
                text: 'Sesi Anda telah berakhir. Silakan login kembali.',
                icon: 'warning',
                confirmButtonText: 'Login',
                allowOutsideClick: false,
                allowEscapeKey: false
            }).then(() => {
                window.location.href = '/login';
            });
        } else {
            alert('Sesi Anda telah berakhir. Silakan login kembali.');
            window.location.href = '/login';
        }
    }

    setupBeforeUnload() {
        window.addEventListener('beforeunload', () => {
            if (this.refreshInterval) {
                clearInterval(this.refreshInterval);
            }
        });
    }
    
    setupGlobalFormHandler() {
        // Intercept all form submissions
        document.addEventListener('submit', (e) => {
            const form = e.target;
            
            // Skip if form has data-no-csrf attribute
            if (form.hasAttribute('data-no-csrf')) {
                return;
            }
            
            // Ensure CSRF token is present
            this.ensureCsrfToken(form);
        });
    }
    
    ensureCsrfToken(form) {
        let tokenInput = form.querySelector('input[name="_token"]');
        if (!tokenInput) {
            // Create token input if it doesn't exist
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = '_token';
            input.value = this.csrfToken || document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
            form.appendChild(input);
        } else {
            // Update existing token
            tokenInput.value = this.csrfToken || document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
        }
    }
    
    setupAjaxErrorHandler() {
        // Handle jQuery AJAX errors
        if (window.$ && window.$.ajaxSetup) {
            window.$.ajaxSetup({
                error: (xhr, status, error) => {
                    if (xhr.status === 419) {
                        this.showSessionExpiredNotification();
                    }
                }
            });
        }
        
        // Handle fetch errors globally
        const originalFetch = window.fetch;
        window.fetch = async (url, options = {}) => {
            try {
                const response = await originalFetch(url, options);
                
                // Check for 419 error
                if (response.status === 419) {
                    this.showSessionExpiredNotification();
                    return response;
                }
                
                return response;
            } catch (error) {
                console.error('Fetch error:', error);
                throw error;
            }
        };
    }

    // Public method to manually refresh session
    async forceRefresh() {
        return await this.refreshSession();
    }

    // Public method to get current CSRF token
    getToken() {
        return this.csrfToken;
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // Only initialize on admin pages
    if (window.location.pathname.startsWith('/admin')) {
        window.adminSessionManager = new AdminSessionManager();
    }
});

// Global function for manual session refresh
window.refreshAdminSession = function() {
    if (window.adminSessionManager) {
        return window.adminSessionManager.forceRefresh();
    }
    return Promise.resolve(false);
};

// Global function for retry form submission
window.retryFormSubmission = function(formData, originalUrl) {
    if (window.adminSessionManager) {
        return window.adminSessionManager.retryFormSubmission(formData, originalUrl);
    }
    return Promise.resolve(false);
};

// Global function to get current CSRF token
window.getCsrfToken = function() {
    if (window.adminSessionManager) {
        return window.adminSessionManager.getToken();
    }
    return document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
};
